package com.franciaflex.faxtomail.ui.swing.content.pdfeditor;

/*
 * #%L
 * FaxToMail :: UI
 * $Id: PDFEditorUIHandler.java 162 2014-06-09 17:14:06Z kmorin $
 * $HeadURL: http://svn.codelutin.com/faxtomail/tags/faxtomail-0.1/faxtomail-ui-swing/src/main/java/com/franciaflex/faxtomail/ui/swing/content/pdfeditor/PDFEditorUIHandler.java $
 * %%
 * Copyright (C) 2014 Franciaflex, Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import java.awt.Component;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Image;
import java.awt.Insets;
import java.awt.Rectangle;
import java.awt.event.ContainerEvent;
import java.awt.event.ContainerListener;
import java.awt.geom.Rectangle2D;
import java.awt.image.RenderedImage;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.Date;
import java.util.List;

import javax.media.jai.PlanarImage;
import javax.swing.JComponent;
import javax.swing.JPanel;

import jaxx.runtime.swing.ComponentMover;
import jaxx.runtime.swing.ComponentResizer;
import jaxx.runtime.validator.swing.SwingValidator;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.franciaflex.faxtomail.persistence.entities.Attachment;
import com.franciaflex.faxtomail.persistence.entities.AttachmentFile;
import com.franciaflex.faxtomail.ui.swing.util.AbstractFaxToMailUIHandler;
import com.franciaflex.faxtomail.ui.swing.util.FaxToMailUIUtil;
import com.franciaflex.faxtomail.ui.swing.util.JImagePanel;
import com.itextpdf.text.Document;
import com.itextpdf.text.DocumentException;
import com.itextpdf.text.PageSize;
import com.itextpdf.text.Paragraph;
import com.itextpdf.text.pdf.PdfWriter;
import com.sun.media.jai.codec.ByteArraySeekableStream;
import com.sun.media.jai.codec.ImageCodec;
import com.sun.media.jai.codec.ImageDecoder;
import com.sun.media.jai.codec.SeekableStream;
import com.sun.pdfview.PDFFile;
import com.sun.pdfview.PDFPage;

import static org.nuiton.i18n.I18n.t;

/**
 * @author Kevin Morin (Code Lutin)
 * @since x.x
 */
public class PDFEditorUIHandler extends AbstractFaxToMailUIHandler<PDFEditorUIModel, PDFEditorUI> {

    private static final Log log = LogFactory.getLog(PDFEditorUIHandler.class);

    protected ComponentMover cm = new ComponentMover();
    protected ComponentResizer cr = new ComponentResizer();

    protected PDFFile pdf;

    @Override
    public void afterInit(PDFEditorUI pdfEditorUI) {
        initUI(pdfEditorUI);

        cm.setDragInsets(cr.getDragInsets());
        cm.setEdgeInsets(new Insets(0, 0, 0, 0));


        final PDFEditorUIModel model = getModel();
        ui.getContainer().addContainerListener(new ContainerListener() {
            @Override
            public void componentAdded(ContainerEvent e) {
                Component child = e.getChild();
                PDFEditorUIModel.Page currentPage = model.getCurrentPage();
                if (child.getClass().isAssignableFrom(PDFEditorNoteUI.class)) {
                    currentPage.addNote((PDFEditorNoteUI) child);

                } else if (child.getClass().isAssignableFrom(PDFEditorCrossUI.class)) {
                    currentPage.addCross((PDFEditorCrossUI) child);

                } else if (child.getClass().isAssignableFrom(PDFEditorLineUI.class)) {
                    currentPage.addLine((PDFEditorLineUI) child);

                } else if (child.getClass().isAssignableFrom(PDFEditorHighlighterUI.class)) {
                    currentPage.addHighlighter((PDFEditorHighlighterUI) child);
                }
            }

            @Override
            public void componentRemoved(ContainerEvent e) {
                Component child = e.getChild();
                PDFEditorUIModel.Page currentPage = model.getCurrentPage();
                if (child.getClass().isAssignableFrom(PDFEditorNoteUI.class)) {
                    currentPage.removeNote((PDFEditorNoteUI) child);

                } else if (child.getClass().isAssignableFrom(PDFEditorCrossUI.class)) {
                    currentPage.removeCross((PDFEditorCrossUI) child);

                } else if (child.getClass().isAssignableFrom(PDFEditorLineUI.class)) {
                    currentPage.removeLine((PDFEditorLineUI) child);

                } else if (child.getClass().isAssignableFrom(PDFEditorHighlighterUI.class)) {
                    currentPage.removeHighlighter((PDFEditorHighlighterUI) child);
                }
            }
        });

        model.addPropertyChangeListener(PDFEditorUIModel.PROPERTY_CURRENT_PAGE_INDEX,
                                        new PropertyChangeListener() {
                                            @Override
                                            public void propertyChange(PropertyChangeEvent evt) {
                                                Integer pageNb = (Integer) evt.getNewValue();
                                                Integer prevPageNb = (Integer) evt.getOldValue();
                                                updatePageNumber(pageNb, prevPageNb);
                                            }
                                        }
        );

        model.addPropertyChangeListener(new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                if (Attachment.PROPERTY_ORIGINAL_FILE.equals(evt.getPropertyName())
                        || Attachment.PROPERTY_EDITED_FILE.equals(evt.getPropertyName())) {

                    PDFEditorUIModel model = (PDFEditorUIModel) evt.getSource();
                    AttachmentFile attachmentFile = model.getNotNullFile();

                    if (attachmentFile != null) {
                        File file = attachmentFile.getFile();
                        if (!FaxToMailUIUtil.isFileAPDF(attachmentFile)) {
                            try {
                                file = convertFileToPdf(file);

                            } catch (IOException e) {
                                if (log.isErrorEnabled()) {
                                    log.error("", e);
                                }
                                getContext().getErrorHelper().showErrorDialog(t("faxtomail.pdfEditor.convertToPdf.error"));

                            } catch (DocumentException e) {
                                if (log.isErrorEnabled()) {
                                    log.error("", e);
                                }
                                getContext().getErrorHelper().showErrorDialog(t("faxtomail.pdfEditor.convertToPdf.error"));
                            }
                        }
                        if (FaxToMailUIUtil.isFileAPDF(attachmentFile)) {
                            try {
                                RandomAccessFile raf = new RandomAccessFile(file, "r");
                                FileChannel channel = raf.getChannel();
                                ByteBuffer buf = channel.map(FileChannel.MapMode.READ_ONLY, 0, channel.size());
                                pdf = new PDFFile(buf);

                                model.setPageNumber(pdf.getNumPages());
                                model.setCurrentPageIndex(1);

                            } catch (IOException e) {
                                if (log.isErrorEnabled()) {
                                    log.error("", e);
                                }
                                getContext().getErrorHelper().showErrorDialog(t("faxtomail.pdfEditor.readPdf.error"));
                            }
                        }
                        getUI().setCursor(Cursor.getDefaultCursor());
                        model.firePropertyChanged(PDFEditorUIModel.PROPERTY_CURRENT_PAGE_INDEX, null, 1);
                    }
                }
            }
        });

        model.addPropertyChangeListener(PDFEditorUIModel.PROPERTY_ZOOM, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                displayPage(model.getCurrentPageIndex(),
                            (Float) evt.getOldValue(),
                            model.getRotation());
            }
        });

        model.addPropertyChangeListener(PDFEditorUIModel.PROPERTY_ROTATION, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                displayPage(model.getCurrentPageIndex(),
                            model.getZoom(),
                            (Integer) evt.getOldValue());
            }
        });
    }

    @Override
    protected JComponent getComponentToFocus() {
        return null;
    }

    @Override
    public SwingValidator<PDFEditorUIModel> getValidator() {
        return null;
    }

    @Override
    public void onCloseUI() {

    }

    public void addNote() {
        PDFEditorNoteUI note = new PDFEditorNoteUI();
        String title = decorate(new Date()) + " - " + getContext().getCurrentUser().getTrigraph();
        note.setTitle(title);
        addPanel(note);
    }

    public void addCross() {
        PDFEditorCrossUI cross = new PDFEditorCrossUI();
        addPanel(cross);
        cr.registerComponent(cross);
    }

    public void addLine() {
        PDFEditorLineUI line = new PDFEditorLineUI();
        addPanel(line);
        cr.registerComponent(ComponentResizer.DIRECTION_HORIZONTAL, line);
    }

    public void addHighlighter() {
        PDFEditorHighlighterUI highlighter = new PDFEditorHighlighterUI();
        addPanel(highlighter);
        cr.registerComponent(highlighter);
    }

    protected void addPanel(JPanel panel) {
        JPanel container = ui.getContainer();
        container.add(panel, 0);

        Insets insets = container.getInsets();
        Dimension size = panel.getPreferredSize();
        Rectangle rect = container.getVisibleRect();
        panel.setBounds(rect.x + insets.left, rect.y + insets.top,
                        size.width, size.height);

        container.updateUI();

        cm.registerComponent(panel);
    }

    protected void updatePageNumber(Integer pageNb, Integer prevPageNb) {
        if (pdf != null) {
            if (pageNb != null) {
                if (prevPageNb != null) {
                    PDFEditorUIModel.Page p = getModel().getPage(prevPageNb);
                    for (JPanel panel : p.getNotes()) {
                        panel.setVisible(false);
                    }
                    for (JPanel panel : p.getCrosses()) {
                        panel.setVisible(false);
                    }
                }

                displayPage(pageNb, getModel().getZoom(), getModel().getRotation());
            }
        }
    }

    protected void displayPage(int pageNb, float previousZoom, int previousRotation) {
        PDFPage page = pdf.getPage(pageNb);
        // create the image
        Rectangle2D bBox = page.getBBox();
        Rectangle rect = new Rectangle(0, 0, (int) bBox.getWidth(),
                                       (int) bBox.getHeight());

        float zoom = getModel().getZoom();
        int rotation = getModel().getRotation();

        int width = (int) (zoom * rect.width);
        int height = (int) (zoom * rect.height);
        Image image = page.getImage(width, height,    // width & height
                                    rect,                       // clip rect
                                    null,                       // null for the ImageObserver
                                    true,                       // fill background with white
                                    true                        // block until drawing is done
        );

        JPanel container = getUI().getContainer();
        Dimension containerSize = new Dimension(rotation % 180 == 0 ? width : height,
                                                rotation % 180 == 0 ? height : width);
        container.setPreferredSize(containerSize);
        container.setMinimumSize(containerSize);
        container.setMaximumSize(containerSize);
        container.setSize(containerSize);

        JImagePanel documentPanel = ui.getDocumentPanel();
        documentPanel.setRotation(rotation);
        documentPanel.setImage(image);

        Insets insets = container.getInsets();
        rect = container.getBounds();

        float zoomRatio = zoom / previousZoom;
        int rotationDiff = rotation - previousRotation;

        PDFEditorUIModel.Page p = getModel().getPage(pageNb);

        for (PDFEditorNoteUI panel : p.getNotes()) {
            panel.setVisible(true);

            panel.setZoom(zoom);
            Rectangle bounds = panel.getBounds();
            int x, y;

            if (rotationDiff == 0) {
                x = bounds.x;
                y = bounds.y;

            } else if (rotationDiff == 90 || rotationDiff == -270) {
                x = rect.width - bounds.height / 2 - bounds.width / 2 - bounds.y;
                y = bounds.x + bounds.width / 2 - bounds.height / 2;

            } else {
                x = bounds.y + bounds.height / 2 - bounds.width / 2;
                y = rect.height - bounds.height / 2 - bounds.width / 2 - bounds.x;
            }

            panel.setBounds((int)(zoomRatio * x) + insets.left,
                            (int)(zoomRatio * y) + insets.top,
                            (int)(zoomRatio * bounds.width), (int)(zoomRatio * bounds.height));

        }

        boolean orientation180 = Math.abs(rotationDiff) % 180 == 0;

        for (PDFEditorCrossUI panel : p.getCrosses()) {
            panel.setVisible(true);

            Rectangle bounds = panel.getBounds();

            int newWidth = orientation180 ? bounds.width : bounds.height;
            int newHeight = orientation180 ? bounds.height : bounds.width;

            int x, y;

            if (rotationDiff == 0) {
                x = bounds.x;
                y = bounds.y;

            } else if (rotationDiff == 90 || rotationDiff == -270) {
                x = rect.width - newWidth - bounds.y;
                y = bounds.x;

            } else {
                x = bounds.y;
                y = rect.height - newHeight - bounds.x;
            }

            panel.setBounds((int) (zoomRatio * x) + insets.left,
                            (int) (zoomRatio * y) + insets.top,
                            (int) (zoomRatio * newWidth), (int) (zoomRatio * newHeight));

        }

        List<PDFEditorLineUI> lines = p.getLines();
        cr.deregisterComponent(lines.toArray(new PDFEditorLineUI[lines.size()]));

        for (PDFEditorLineUI panel : lines) {
            panel.setVisible(true);

            Rectangle bounds = panel.getBounds();

            boolean horizontal = panel.isHorizontal();
            panel.setHorizontal(orientation180 ? horizontal : ! horizontal);

            int newWidth = orientation180 ? bounds.width : bounds.height;
            int newHeight = orientation180 ? bounds.height : bounds.width;

            int x, y;

            if (rotationDiff == 0) {
                x = bounds.x;
                y = bounds.y;

            } else if (rotationDiff == 90 || rotationDiff == -270) {
                x = rect.width - newWidth - bounds.y;
                y = bounds.x;

            } else {
                x = bounds.y;
                y = rect.height - newHeight - bounds.x;
            }

            panel.setBounds((int) (zoomRatio * x) + insets.left,
                            (int) (zoomRatio * y) + insets.top,
                            (int) (zoomRatio * newWidth), (int) (zoomRatio * newHeight));

            cr.registerComponent(panel.isHorizontal() ? ComponentResizer.DIRECTION_HORIZONTAL : ComponentResizer.DIRECTION_VERTICAL,
                                 panel);
        }

        for (PDFEditorHighlighterUI panel : p.getHighlighters()) {
            panel.setVisible(true);

            Rectangle bounds = panel.getBounds();

            int newWidth = orientation180 ? bounds.width : bounds.height;
            int newHeight = orientation180 ? bounds.height : bounds.width;

            int x, y;

            if (rotationDiff == 0) {
                x = bounds.x;
                y = bounds.y;

            } else if (rotationDiff == 90 || rotationDiff == -270) {
                x = rect.width - newWidth - bounds.y;
                y = bounds.x;

            } else {
                x = bounds.y;
                y = rect.height - newHeight - bounds.x;
            }

            panel.setBounds((int) (zoomRatio * x) + insets.left,
                            (int) (zoomRatio * y) + insets.top,
                            (int) (zoomRatio * newWidth), (int) (zoomRatio * newHeight));

        }

        container.updateUI();
    }

    public File convertFileToPdf(File file) throws IOException, DocumentException {

        AttachmentFile attachmentFile = getModel().getNotNullFile();
        File target = File.createTempFile("faxtomail-", ".tmp");
        target.deleteOnExit();

        Document document = new Document();
        FileOutputStream fos = new FileOutputStream(target);
        PdfWriter writer = PdfWriter.getInstance(document, fos);
        writer.open();

        document.setPageSize(PageSize.A4);
        document.open();

        if (FaxToMailUIUtil.isFileATxt(attachmentFile)) {
            FileReader fr = new FileReader(file);
            BufferedReader br = new BufferedReader(fr);
            StringBuilder text = new StringBuilder();
            while (br.readLine() != null) {
                text.append(br.readLine()).append("\n");
            }
            document.add(new Paragraph(text.toString()));

        } else if (FaxToMailUIUtil.isFileATif(attachmentFile)) {
            FileInputStream fis = new FileInputStream(file);
            FileChannel channel = fis.getChannel();
            ByteBuffer buffer = ByteBuffer.allocate((int) channel.size());
            channel.read(buffer);
            SeekableStream stream = new ByteArraySeekableStream(buffer.array());
            String[] names = ImageCodec.getDecoderNames(stream);
            ImageDecoder tifImageDecoder = ImageCodec.createImageDecoder(names[0], stream, null);

            for (int i = 0 ; i < tifImageDecoder.getNumPages() ; i++) {
                RenderedImage renderedImage = tifImageDecoder.decodeAsRenderedImage(i);
                java.awt.Image awtImage = PlanarImage.wrapRenderedImage(renderedImage).getAsBufferedImage();
                document.newPage();
                com.itextpdf.text.Image image = com.itextpdf.text.Image.getInstance(writer, awtImage, 1.0f);
                image.scaleToFit(PageSize.A4.getWidth() - document.leftMargin() - document.rightMargin(),
                                 PageSize.A4.getHeight() - document.topMargin() - document.bottomMargin());
                document.add(image);
            }

        } else {
            URL resource = file.toURI().toURL();
            com.itextpdf.text.Image image = com.itextpdf.text.Image.getInstance(resource);
            float scaler = ((document.getPageSize().getWidth() - document.leftMargin() - document.rightMargin()) / image.getWidth()) * 100;
            image.scalePercent(scaler);
            document.add(image);
        }

        document.close();
        writer.close();
        
        // convert content to blob
        AttachmentFile attachmentFileNew = getContext().getEmailService().getAttachmentFileFromStream(new FileInputStream(target));
        attachmentFileNew.setFilename(attachmentFile.getFilename() + ".pdf");
        getModel().setEditedFile(attachmentFileNew);

        return target;
    }

    public void zoomOut() {
        PDFEditorUIModel model = getModel();
        float zoom = model.getZoom();
        model.setZoom(zoom - 0.25f);
    }

    public void zoomIn() {
        PDFEditorUIModel model = getModel();
        float zoom = model.getZoom();
        model.setZoom(zoom + 0.25f);
    }

    public void rotateClockwise() {
        int rotation = getModel().getRotation();
        getModel().setRotation((360 + rotation + 90) % 360);
    }

    public void rotateAntiClockwise() {
        int rotation = getModel().getRotation();
        getModel().setRotation((360 + rotation - 90) % 360);
    }

    public void print() {
        FaxToMailUIUtil.print(getModel().getEditedFile(), false);
    }
}
