package fr.ifremer.shared.application;

/*
 * #%L
 * Ifremer shared :: Application
 * $Id: ApplicationIOUtil.java 1379 2013-11-24 22:44:33Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/application/tags/ifremer-shared-1.0.1/application/src/main/java/fr/ifremer/shared/application/ApplicationIOUtil.java $
 * %%
 * Copyright (C) 2013 Ifremer, CodeLutin, Tony CHEMIT
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Charsets;
import com.google.common.io.Files;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.vfs2.FileObject;
import org.apache.commons.vfs2.FileSelector;
import org.apache.commons.vfs2.FileSystemManager;
import org.apache.commons.vfs2.FileType;
import org.apache.commons.vfs2.VFS;
import org.nuiton.util.ZipUtil;

import java.io.Closeable;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.Reader;
import java.io.Writer;
import java.util.List;
import java.util.UUID;

import static org.nuiton.i18n.I18n._;

/**
 * Useful methods around IO, all method errors are boxes a
 * {@link ApplicationTechnicalException}.
 * <p/>
 * Created on 11/24/13.
 *
 * @author Tony Chemit <chemit@codelutin.com>
 * @since 1.0
 */
public class ApplicationIOUtil {

    protected ApplicationIOUtil() {
        // no instance
    }

    public static void close(Closeable toClose, String errorMessage) {

        try {
            toClose.close();
        } catch (IOException e) {
            throw new ApplicationTechnicalException(errorMessage, e);
        }
    }

    public static void zip(File source, File archiveFile, String errorMessage) {
        try {
            ZipUtil.compress(archiveFile, source);
        } catch (IOException e) {
            throw new ApplicationTechnicalException(errorMessage, e);
        }
    }

    public static void zip(File source, File archiveFile, List<File> files, String errorMessage) {
        try {
            ZipUtil.compressFiles(archiveFile, source, files);
        } catch (IOException e) {
            throw new ApplicationTechnicalException(_(errorMessage, archiveFile), e);
        }
    }

    public static FileObject resolveFile(String file, String errorMessage) {
        try {
            FileSystemManager manager = VFS.getManager();
            FileObject targetObject = manager.resolveFile(file);
            return targetObject;
        } catch (IOException e) {
            throw new ApplicationTechnicalException(errorMessage, e);
        }
    }

    public static FileObject[] getChildren(FileObject objectFile, String errorMessage) {
        try {
            return objectFile.getChildren();
        } catch (IOException e) {
            throw new ApplicationTechnicalException(errorMessage, e);
        }
    }

    public static FileObject getChild(FileObject objectFile, String name, String errorMessage) {
        try {
            return objectFile.getChild(name);
        } catch (IOException e) {
            throw new ApplicationTechnicalException(errorMessage, e);
        }
    }


    public static FileType getType(FileObject objectFile, String errorMessage) {
        try {
            return objectFile.getType();
        } catch (IOException e) {
            throw new ApplicationTechnicalException(errorMessage, e);
        }
    }

    public static void explode(FileObject archiveFile,
                               File target,
                               FileSelector fileSelector, String errorMessage) {
        try {
            FileSystemManager manager = VFS.getManager();
            FileObject targetObject = manager.toFileObject(target);
            targetObject.copyFrom(archiveFile, fileSelector);
        } catch (IOException e) {
            throw new ApplicationTechnicalException(errorMessage, e);
        }
    }

    public static void forceMkdir(File dir, String errorMessage) {
        try {
            FileUtils.forceMkdir(dir);
        } catch (IOException e) {
            throw new ApplicationTechnicalException(errorMessage, e);
        }
    }

    public static void cleanDirectory(File dir, String errorMessage) {
        try {
            FileUtils.cleanDirectory(dir);
        } catch (IOException e) {
            throw new ApplicationTechnicalException(errorMessage, e);
        }
    }

    public static void deleteDirectory(File dir, String errorMessage) {
        try {
            FileUtils.deleteDirectory(dir);
        } catch (IOException e) {
            throw new ApplicationTechnicalException(errorMessage, e);
        }
    }

    public static void forceDeleteOnExit(File dir, String errorMessage) {
        try {
            FileUtils.forceDeleteOnExit(dir);
        } catch (IOException e) {
            throw new ApplicationTechnicalException(errorMessage, e);
        }
    }

    public static void copyDirectory(File source, File target, String errorMessage) {
        try {
            FileUtils.copyDirectory(source, target);
        } catch (IOException e) {
            throw new ApplicationTechnicalException(errorMessage, e);
        }
    }

    public static void copyFileToDirectory(File source, File target, String errorMessage) {
        try {
            FileUtils.copyFileToDirectory(source, target);
        } catch (IOException e) {
            throw new ApplicationTechnicalException(errorMessage, e);
        }
    }

    public static void copyFile(File source, File target, String errorMessage) {
        try {
            FileUtils.copyFile(source, target);
        } catch (IOException e) {
            throw new ApplicationTechnicalException(errorMessage, e);
        }
    }

    public static void deleteFile(File file, String errorMessage) {
        try {
            FileUtils.forceDelete(file);
        } catch (IOException e) {
            throw new ApplicationTechnicalException(errorMessage, e);
        }
    }

    public static String getBaseName(String file) {
        return FilenameUtils.getBaseName(file);
    }

    public static String getExtension(String file) {
        return FilenameUtils.getExtension(file);
    }

    public static Reader newReader(File file, String errorMessage) {
        try {
            return Files.newReader(file, Charsets.UTF_8);

        } catch (FileNotFoundException e) {
            throw new ApplicationTechnicalException(_(errorMessage, file), e);
        }
    }

    public static Writer newWriter(File file, String errorMessage
    ) {
        try {
            return Files.newWriter(file, Charsets.UTF_8);

        } catch (FileNotFoundException e) {
            throw new ApplicationTechnicalException(_(errorMessage, file), e);
        }
    }

    public static File explodeZip(File rootDirectory,
                                  File file,
                                  String errorMessage) {
        File tempDir = new File(rootDirectory, UUID.randomUUID().toString());
        try {
            ZipUtil.uncompress(file, tempDir);
        } catch (IOException e) {
            throw new ApplicationTechnicalException(_(errorMessage, file), e);
        }
        return tempDir;
    }

    public static String readContent(File file, String errorMessage) {

        try {
            return Files.readFirstLine(file, Charsets.UTF_8);

        } catch (IOException e) {
            throw new ApplicationTechnicalException(errorMessage, e);
        }
    }

    public static void writeContent(File file, String content, String errorMessage) {
        try {
            FileUtils.write(file, content, Charsets.UTF_8);
        } catch (IOException e) {
            throw new ApplicationTechnicalException(errorMessage, e);
        }
    }
}
