package fr.ifremer.shared.application;

/*
 * #%L
 * Ifremer shared :: Application
 * $Id: ApplicationDataUtil.java 1444 2013-12-09 21:07:13Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/application/tags/ifremer-shared-1.0.1/application/src/main/java/fr/ifremer/shared/application/ApplicationDataUtil.java $
 * %%
 * Copyright (C) 2013 Ifremer, CodeLutin, Tony CHEMIT
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang3.time.DurationFormatUtils;

import java.math.BigDecimal;
import java.math.MathContext;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.Date;

import static org.nuiton.i18n.I18n._;

/**
 * Useful methods on data.
 * <p/>
 * TODO Needs some javadoc.
 * <p/>
 * Created on 11/24/13.
 *
 * @author Tony Chemit <chemit@codelutin.com>
 * @since 1.0
 */
public class ApplicationDataUtil {

    public static final double EARTH_RADIUS = 6378288.0;

    private static final MathContext MATH_CONTEXT_4_DIGIT = new MathContext(4);

    private static DecimalFormatSymbols symbols;

    private static DecimalFormat decimalFormat;

    public static int computeDistanceInMeters(Float startLatitude,
                                              Float startLongitude,
                                              Float endLatitude,
                                              Float endLongitude) {

        double sLat = startLatitude * Math.PI / 180.0;
        double sLong = startLongitude * Math.PI / 180.0;
        double eLat = endLatitude * Math.PI / 180.0;
        double eLong = endLongitude * Math.PI / 180.0;

        Double d = EARTH_RADIUS *
                   (Math.PI / 2 - Math.asin(Math.sin(eLat) * Math.sin(sLat)
                                            + Math.cos(eLong - sLong) * Math.cos(eLat) * Math.cos(sLat)));
        return d.intValue();
    }

    public static String getDistanceInMilles(Float distance) {
        String distanceText;
        if (distance != null) {
            Float distanceInMilles = distance / 1852;
            distanceText = String.format("%.3f", distanceInMilles);

        } else {
            distanceText = "";
        }
        return distanceText;
    }

    public static float getRoundedLengthStep(float lengthStep, boolean aroundUp) {
        int intValue = (int) ((lengthStep + (aroundUp ? 0.001f : 0f)) * 10);
        float result = intValue / 10f;
        return result;
    }

    public static void setProperty(Object bean, String property, Object value) {
        Preconditions.checkNotNull(bean);
        Preconditions.checkNotNull(property);
        try {
            PropertyUtils.setSimpleProperty(bean, property, value);
        } catch (Exception e) {
            throw new ApplicationTechnicalException(_("application.property.set.error", property, bean.getClass().getName()), e);
        }
    }

    public static Object getProperty(Object bean, String property) {
        Preconditions.checkNotNull(bean);
        Preconditions.checkNotNull(property);
        try {
            return PropertyUtils.getSimpleProperty(bean, property);
        } catch (Exception e) {
            throw new ApplicationTechnicalException(_("application.property.get.error", property, bean.getClass().getName()), e);
        }
    }

    public static DecimalFormatSymbols getDecimalFormatSymbols() {
        if (symbols == null) {
            symbols = new DecimalFormatSymbols();
            symbols.setDecimalSeparator('.');
            symbols.setGroupingSeparator(' ');
        }
        return symbols;
    }

    public static DecimalFormat getDecimalFormat(int minDecimal, int maxDecimal) {
        if (decimalFormat == null) {
            decimalFormat = new DecimalFormat();
            decimalFormat.setDecimalFormatSymbols(getDecimalFormatSymbols());
            decimalFormat.setGroupingUsed(false);
        }
        decimalFormat.setMinimumFractionDigits(minDecimal);
        decimalFormat.setMaximumFractionDigits(maxDecimal);
        return decimalFormat;
    }

    public static String getWeightStringValue(Float weight) {
        String textValue;
        if (weight != null) {
            DecimalFormat weightDecimalFormat = getDecimalFormat(1, 3);
            textValue = weightDecimalFormat.format(weight);

        } else {
            textValue = "";
        }
        return textValue;
    }

    public static <N extends Number> N getValueOrComputedValue(N value, N computedValue) {
        return value == null ? computedValue : value;
    }

    public static <N extends Number> Boolean getValueOrComputedValueComputed(N value, N computedValue) {
        Boolean result;
        if (value == null) {

            result = computedValue == null ? null : true;
        } else {
            result = false;
        }
        return result;
    }

    /**
     * Round the given value to max 4 digits.
     *
     * @param value the float to round.
     * @return the rounded value
     * @since 1.0.1
     */
    public static float roundKiloGram(float value) {
        BigDecimal sumB = new BigDecimal(value);
        float result = sumB.abs(MATH_CONTEXT_4_DIGIT).floatValue();
        return result;
    }

    public static String getDuration(Date startDate,
                                     Date endDate,
                                     String format) {
        String duration = "";
        if (startDate != null && endDate != null && !startDate.after(endDate)) {
            duration = DurationFormatUtils.formatPeriod(
                    startDate.getTime(),
                    endDate.getTime(),
                    format);
        }
        return duration;
    }
}
