package org.sharengo.wikitty.hbase;

/* *##%
 * Copyright (c) 2009 poussin. All rights reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *##%*/

import static org.sharengo.wikitty.hbase.WikittyHBaseUtil.F_ADMIN;
import static org.sharengo.wikitty.hbase.WikittyHBaseUtil.F_DATA;
import static org.sharengo.wikitty.hbase.WikittyHBaseUtil.Q_NAME;
import static org.sharengo.wikitty.hbase.WikittyHBaseUtil.Q_REQUIRES;
import static org.sharengo.wikitty.hbase.WikittyHBaseUtil.Q_TAGVALUES;
import static org.sharengo.wikitty.hbase.WikittyHBaseUtil.Q_VERSION;
import static org.sharengo.wikitty.hbase.WikittyHBaseUtil.T_EXTENSION;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.hbase.KeyValue;
import org.apache.hadoop.hbase.client.Get;
import org.apache.hadoop.hbase.client.HTable;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.ResultScanner;
import org.apache.hadoop.hbase.util.Bytes;
import org.sharengo.wikitty.FieldType;
import org.sharengo.wikitty.UpdateResponse;
import org.sharengo.wikitty.WikittyException;
import org.sharengo.wikitty.WikittyExtension;
import org.sharengo.wikitty.WikittyExtensionStorage;
import org.sharengo.wikitty.WikittyTransaction;
import org.sharengo.wikitty.WikittyUtil;

/**
 *
 * @author poussin
 * @version $Revision: 1 $
 *
 * Last update: $Date: 2010-04-16 10:29:38 +0200 (ven., 16 avril 2010) $
 * by : $Author: echatellier $
 */
public class WikittyExtensionStorageHBase implements WikittyExtensionStorage {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittyExtensionStorageHBase.class);

    /** storage for extension */
    protected HTable hTable;

    /** cache for extension key: ext id (extname[extversion]) value: WikittyExtension
     * TODO poussin 20090813 use Map with soft reference (commons-collection)*/
    protected Map<String, WikittyExtension> extensionCache =
            new HashMap<String, WikittyExtension>();

    /** cache for last extension version; key: extName value: extVersion */
    protected Map<String, String> lastVersion = null;

    public WikittyExtensionStorageHBase() {
        try {
            hTable = new HTable(WikittyHBaseUtil.getHBaseConfiguration(), T_EXTENSION);

            // create cache for futur call
            lastVersion = new HashMap<String, String>();
            ResultScanner scanner = hTable.getScanner(F_ADMIN);
            for (Iterator<Result> it = scanner.iterator(); it.hasNext();) {
                Result row = it.next();
                String name = Bytes.toString(row.getValue(F_ADMIN, Q_NAME));
                String version = Bytes.toString(row.getValue(F_ADMIN, Q_VERSION));
                String prevVersion = lastVersion.get(name);
                if (prevVersion == null ||
                        WikittyUtil.versionGreaterThan(version, prevVersion)) {
                    lastVersion.put(name, version);
                }
            }
            scanner.close();

        } catch (IOException eee) {
            throw new WikittyException(eee);
        }
    }
    
    @Override
    public UpdateResponse store(WikittyTransaction transaction,
            Collection<WikittyExtension> extensions) {
        try {

            List<Put> puts = new ArrayList<Put>();
            for (WikittyExtension ext : extensions) {
               // extension id is extension name with version
               String extId = ext.getId();
               String extName = ext.getName();
               String extVersion = ext.getVersion();

               String extLastVersion = getLastVersion(transaction, extName);
               // else no permitted to update with the less version
               if(WikittyUtil.versionGreaterThan(extVersion, extLastVersion)) {
                   // create definition
                   Put put = new Put(Bytes.toBytes(extId));
                   put.add(F_ADMIN, Q_NAME, Bytes.toBytes(extName));
                   put.add(F_ADMIN, Q_VERSION, Bytes.toBytes(extVersion));
                   
                   String extTagValues = WikittyUtil.tagValuesToString(ext.getTagValues());
                   put.add(F_ADMIN, Q_TAGVALUES, Bytes.toBytes(extTagValues));

                   String requires = ext.getRequires();
                   if(requires != null) {
                       put.add(F_ADMIN, Q_REQUIRES, Bytes.toBytes(requires));
                   }

                   for (String fieldName : ext.getFieldNames()) {
                       FieldType type = ext.getFieldType(fieldName);
                       put.add(F_DATA, Bytes.toBytes(fieldName), Bytes.toBytes(type.toDefinition(fieldName)));
                   }

                   lastVersion.put(extName, extVersion);
                   puts.add(put);
               } else {
                   if(!WikittyUtil.versionEquals(extVersion, extLastVersion)) {
                        throw new WikittyException(String.format(
                                "Your extension '%s' is obsolete", extId));
                   }
               }
            }
            
            hTable.put(puts);

            // nothing to put in result
            UpdateResponse result = new UpdateResponse();
            return result;
        } catch (IOException eee) {
            throw new WikittyException(eee);
       }
    }

    @Override
    public boolean exists(WikittyTransaction transaction, String id) {
        if (extensionCache.containsKey(id)) {
            return true;
        }

        try {
            Get get = new Get(Bytes.toBytes(id));
            boolean result = hTable.exists(get);
            return result;
        } catch (IOException eee) {
            throw new WikittyException(eee);
       }
    }

    @Override
    public List<String> getAllExtensionIds(WikittyTransaction transaction) {
        try {
            ResultScanner scanner = hTable.getScanner(F_ADMIN);
            List<String> result = new ArrayList<String>();
            for (Iterator<Result> it = scanner.iterator(); it.hasNext();) {
                Result row = it.next();
                String name = Bytes.toString(row.getValue(F_ADMIN, Q_NAME));
                String version = Bytes.toString(row.getValue(F_ADMIN, Q_VERSION));

                result.add(WikittyExtension.computeId(name, version));
            }
            scanner.close();
            
            return result;
        } catch (IOException eee) {
            throw new WikittyException(eee);
       }
    }

    @Override
    public List<String> getAllExtensionsRequires(
            WikittyTransaction transaction, String extensionName) {
        try {
            ResultScanner scanner = hTable.getScanner(F_ADMIN);
            List<String> result = new ArrayList<String>();
            for (Iterator<Result> it = scanner.iterator(); it.hasNext();) {
                Result row = it.next();
                String requires = Bytes.toString(row.getValue(F_ADMIN, Q_REQUIRES));

                if (extensionName.equals(requires)) {
                    String name = Bytes.toString(row.getValue(F_ADMIN, Q_NAME));
                    String version = Bytes.toString(row.getValue(F_ADMIN, Q_VERSION));
                    result.add(WikittyExtension.computeId(name, version));
                }
            }
            scanner.close();
            
            return result;
        } catch (IOException eee) {
            throw new WikittyException(eee);
       }
    }
    
    /**
     * return last version available for specified extension name
     * @param name name of extension
     * @return last version availble for this version, or null if extension
     * doesn't exist
     */
    @Override
    public String getLastVersion(WikittyTransaction transaction, String extName) {
        String result = lastVersion.get(extName);
        log.debug("LastVersion=" + result + " for " + extName);
        return result;
    }

    @Override
    public WikittyExtension restore(WikittyTransaction transaction, String name, String version)
            throws WikittyException {
        String id = WikittyExtension.computeId(name, version);
        WikittyExtension result = extensionCache.get(id);
        log.debug("Extension found in cache=" + result);
        if (result == null) {
            try {
                Get get = new Get(Bytes.toBytes(id));
                Result row = hTable.get(get);
                if(row.isEmpty()) {
                    log.debug("Extension not found");
                    return null;
                }
                
                String extName = Bytes.toString(row.getValue(F_ADMIN, Q_NAME));
                String extVersion = Bytes.toString(row.getValue(F_ADMIN, Q_VERSION));
                String requires = Bytes.toString(row.getValue(F_ADMIN, Q_REQUIRES));
                LinkedHashMap<String, FieldType> fieldTypes = new LinkedHashMap<String, FieldType>();
                for (KeyValue keyValue : row.list()) {
                    if (Bytes.equals(F_DATA, keyValue.getFamily())) {
                        String fieldDef = Bytes.toString(keyValue.getValue());
                        FieldType fieldType = new FieldType();
                        String fieldName = WikittyUtil.parseField(fieldDef, fieldType);
                        fieldTypes.put(fieldName, fieldType);
                    }
                }

                result = new WikittyExtension(extName, extVersion, requires, fieldTypes);

                String tagValues = Bytes.toString(row.getValue(F_ADMIN, Q_TAGVALUES));
                result.setTagValues(WikittyUtil.tagValuesToMap(tagValues));

                extensionCache.put(id, result);
            } catch (IOException eee) {
                throw new WikittyException(String.format("Can't load extension %s", id), eee);
            }
        }
        log.debug("Extension restored=" + result);
        return result;
    }

    @Override
    public void clear(WikittyTransaction transaction) {
        // Nothing do
    }

}
