package jaxx.runtime.swing.editor;

/*-
 * #%L
 * JAXX :: Widgets
 * %%
 * Copyright (C) 2008 - 2017 Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JTextField;
import javax.swing.JToolBar;
import jaxx.runtime.JAXXBinding;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.JAXXObject;
import jaxx.runtime.JAXXObjectDescriptor;
import jaxx.runtime.JAXXUtil;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.binding.SimpleJAXXObjectBinding;
import jaxx.runtime.context.DefaultJAXXContext;
import jaxx.runtime.swing.BaseActionPanel;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import static org.nuiton.i18n.I18n.t;

public class FileEditor extends BaseActionPanel implements JAXXObject {

    /*-----------------------------------------------------------------------*/
    /*---------------- Constants for all javaBean properties ----------------*/
    /*-----------------------------------------------------------------------*/

    public static final String PROPERTY_ACCEPT_ALL_FILE_FILTER_USED = "acceptAllFileFilterUsed";
    public static final String PROPERTY_DIRECTORY_ENABLED = "directoryEnabled";
    public static final String PROPERTY_EXTS = "exts";
    public static final String PROPERTY_EXTS_DESCRIPTION = "extsDescription";
    public static final String PROPERTY_FILE_ENABLED = "fileEnabled";
    public static final String PROPERTY_SHOW_RESET = "showReset";
    public static final String PROPERTY_START_PATH = "startPath";
    public static final String PROPERTY_TITLE = "title";

    /*-----------------------------------------------------------------------*/
    /*------------------ Constants for all public bindings ------------------*/
    /*-----------------------------------------------------------------------*/

    public static final String BINDING_BOUTON_XSL_LOCATION_ENABLED = "boutonXslLocation.enabled";
    public static final String BINDING_PATH_FIELD_ENABLED = "pathField.enabled";
    public static final String BINDING_PATH_FIELD_TEXT = "pathField.text";
    public static final String BINDING_RESET_BUTTON_ENABLED = "resetButton.enabled";
    public static final String BINDING_TOOLBAR_VISIBLE = "toolbar.visible";

    /*-----------------------------------------------------------------------*/
    /*------------------------- Other static fields -------------------------*/
    /*-----------------------------------------------------------------------*/

    private static final String $jaxxObjectDescriptor = "H4sIAAAAAAAAAKVUzU8TQRSfVkot3x+KGDEBRGNC0urBRFOifEiDpAoBTIi9ON0d6JBhZ5x5C8vF+Cf4J+jdi4k3T8aDZw9ejP+CMR68Gt/sblmKBRpo0t3N+/i93/vNvPfuJ8kYTca2aBDkte8B32b5xZn19aXqFnPgITOO5gqkJtEvlSbpCul09+0GyLVK2aYX4vTCnNxW0mPegeximXQY2BPM1BgDIFcbMxxjCqv77mKgfF1H3SfVDPXN71/p1+6rt2lCAoXserCV0ZOykk7ayiTNXSADWGmHFgT1NpGG5t4m8u22tjlBjXlCt9kL8pJky6RdUY1gQMZbbznECPMDBaR/YpYaNuMAl94y9Zi4BWSigbLZRQJ55nIEyZe4YPPhp1IhSDuQbI16rmAayGQriQtRdJJ/iToOUzAjhA3CPzD91DAUot/2nLc65GelFIx6SVafyzXeCKn35j1aFcy19v7E37lhKzZ15QxQDcsUagiTlIikTsIywEGESg8nxjYWgDlk67W2utKo4yF3ztTk7gozDA7xyAI2VUUq5IJlEcSKLa6heZZqG5VL+tEWYdYHkB6QwYaEyGoDx5KqCvsrcSZQx6FGeKQbOhoL9FeljyjrRpSlQ+t93IiDNLnScLg4IvlkRJI7nKqQjPbRDGS48v9UraArmqfhQ/NkAUPv36HBbx9/fCjVh6gba19sGnpgB+DlVloqvITclu6NJsgHLgqPqSpW8BCYCK8LekeaEFuN3UgO60UXz6bnF6ipIUQm+/3T56HnX8+RdIl0CEndErXxj0gOang0NSncQD2YDhl17Z7HZ5/lBqRd0D1UFkjPlEuBjla55+I53A9QhJEmIuwzqea+/BlcfT9dFyKFxC4fGZ6IkXlG2rknuMfCfRKviqb7o1MZ5rsyWQnNlkTKvjtUfG9vh887zTrtrkrtMr1MuQd2fAemIhqjFHC0qj4wbNpm3wu7sV/FEHy8BfDcBooOdpxtyPTpgbI73PAY5u7pYdq4E03IERjXW6HCkv10BpiuDen4Jpb9GHVawcqFWCfI3ApQp11ua1zZbXMmKGteO0ahm60cFsQsTo1hzUvHIEyeuQ+L8A/TP3sp/wgAAA==";
    private static final Log log = LogFactory.getLog(FileEditor.class);
    private static final long serialVersionUID = 1L;

    /*-----------------------------------------------------------------------*/
    /*--------------------------- Internal states ---------------------------*/
    /*-----------------------------------------------------------------------*/

    protected List<Object> $activeBindings = new ArrayList<Object>();
    protected Map<String, Object> $bindingSources = new HashMap<String, Object>();
    protected final Map<String, JAXXBinding> $bindings = new TreeMap<String, JAXXBinding>();
    protected Map<String, Object> $objectMap = new HashMap<String, Object>();
    protected Map<?, ?> $previousValues = new HashMap<Object, Object>();
    private boolean allComponentsCreated;
    protected final JAXXContext delegateContext = new DefaultJAXXContext();

    /*-----------------------------------------------------------------------*/
    /*------------------------ Protected components  ------------------------*/
    /*-----------------------------------------------------------------------*/

    protected Boolean acceptAllFileFilterUsed;
    protected JButton boutonXslLocation;
    protected Boolean directoryEnabled;
    protected String exts;
    protected String extsDescription;
    protected Boolean fileEnabled;
    protected FileEditorHandler handler;
    protected JTextField pathField;
    protected JButton resetButton;
    protected Boolean showReset;
    protected String startPath;
    protected String title;
    protected JToolBar toolbar;

    /*-----------------------------------------------------------------------*/
    /*------------------------- Private components  -------------------------*/
    /*-----------------------------------------------------------------------*/

    private FileEditor $BaseActionPanel0 = this;

    /*-----------------------------------------------------------------------*/
    /*---------------------- Raw body code from script ----------------------*/
    /*-----------------------------------------------------------------------*/

    protected File selectedFile;
    
    
    public void setSelectedFile(File selectedFile) {
        this.selectedFile = selectedFile;
        setStartPath(selectedFile == null ? null : selectedFile.getAbsolutePath());
        fireActionEvent();
    }
    
    public void setSelectedFilePath(String startPath) {
        setSelectedFile(startPath == null ? null : new File(startPath));
    }
    
    /**public void setSelectedFile(String startPath) {
        this.selectedFile = null;
        setStartPath(startPath);
        fireActionEvent();
    }*/
    
    public File getSelectedFile() {
        if (selectedFile == null) {
            if (StringUtils.isNotEmpty(startPath)) {
                selectedFile = new File(startPath);
            }
        }
        return selectedFile;
    }
    
    public void setDialogOwner(JDialog dialogOwner) {
      handler.setDialogOwner(dialogOwner);
    }

    /*-----------------------------------------------------------------------*/
    /*---------------------------- Constructors  ----------------------------*/
    /*-----------------------------------------------------------------------*/

    public FileEditor() {
        $initialize();
    }

    public FileEditor(JAXXContext parentContext) {
        JAXXUtil.initContext(this, parentContext);
        $initialize();
    }

    /*-----------------------------------------------------------------------*/
    /*--------------------------- Statics methods ---------------------------*/
    /*-----------------------------------------------------------------------*/

    public static JAXXObjectDescriptor $getJAXXObjectDescriptor() {
        return JAXXUtil.decodeCompressedJAXXObjectDescriptor($jaxxObjectDescriptor);
    }

    /*-----------------------------------------------------------------------*/
    /*---------------------- JAXXObject implementation ----------------------*/
    /*-----------------------------------------------------------------------*/

    @Override
    public void applyDataBinding(String $binding) {
        if (allComponentsCreated && $bindings.containsKey($binding)) {
            getDataBinding($binding).applyDataBinding();
        }
        processDataBinding($binding);
    }

    @Override
    public void firePropertyChange(String propertyName, Object oldValue, Object newValue) {
        super.firePropertyChange(propertyName, oldValue, newValue);
    }

    @Override
    public Map<String, Object> get$objectMap() {
        return $objectMap;
    }

    @Override
    public JAXXBinding getDataBinding(String bindingId) {
        return $bindings.get(bindingId);
    }

    @Override
    public JAXXBinding[] getDataBindings() {
        return $bindings.values().toArray(new JAXXBinding[$bindings.size()]);
    }

    @Override
    public Object getObjectById(String id) {
        return $objectMap.get(id);
    }

    @Override
    public void processDataBinding(String $binding, boolean $force) {
        if (!$force && $activeBindings.contains($binding)) { 
            return;
        }
        $activeBindings.add($binding);
        try {
            if (allComponentsCreated && $bindings.containsKey($binding)) {
                getDataBinding($binding).processDataBinding();
            }
        } finally {
            $activeBindings.remove($binding);
        }
    }

    @Override
    public void processDataBinding(String $binding) {
        processDataBinding($binding, false);
    }

    @Override
    public void registerDataBinding(JAXXBinding binding) {
        $bindings.put(binding.getId(), binding);
    }

    @Override
    public void removeDataBinding(String $binding) {
        if (allComponentsCreated && $bindings.containsKey($binding)) {
            getDataBinding($binding).removeDataBinding();
        }
    }

    /*-----------------------------------------------------------------------*/
    /*--------------------- JAXXContext implementation  ---------------------*/
    /*-----------------------------------------------------------------------*/

    @Override
    public <T> T getContextValue(Class<T> clazz) {
        return delegateContext.getContextValue(clazz, null);
    }

    @Override
    public <T> T getContextValue(Class<T> clazz, String name) {
        return delegateContext.getContextValue(clazz, name);
    }

    @Override
    public JAXXContext getDelegateContext() {
        return delegateContext;
    }

    @Override
    public <O extends Container> O getParentContainer(Class<O> clazz) {
        return SwingUtil.getParentContainer(this, clazz);
    }

    @Override
    public <O extends Container> O getParentContainer(Object source, Class<O> clazz) {
        return SwingUtil.getParentContainer(source, clazz);
    }

    @Override
    public <T> void removeContextValue(Class<T> clazz) {
        delegateContext.removeContextValue(clazz, null);
    }

    @Override
    public <T> void removeContextValue(Class<T> clazz, String name) {
        delegateContext.removeContextValue(clazz, name);
    }

    @Override
    public <T> void setContextValue(T o) {
        delegateContext.setContextValue(o, null);
    }

    @Override
    public <T> void setContextValue(T o, String name) {
        delegateContext.setContextValue(o, name);
    }

    /*-----------------------------------------------------------------------*/
    /*---------------------------- Event methods ----------------------------*/
    /*-----------------------------------------------------------------------*/

    public void doActionPerformed__on__boutonXslLocation(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        handler.openLocation();
    }

    public void doActionPerformed__on__resetButton(ActionEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        setSelectedFile(null);
    }

    public void doKeyReleased__on__pathField(KeyEvent event) {
        if (log.isDebugEnabled()) {
            log.debug(event);
        }
        setSelectedFilePath(pathField.getText());
    }

    /*-----------------------------------------------------------------------*/
    /*----------------------- Public acessor methods  -----------------------*/
    /*-----------------------------------------------------------------------*/

    public Boolean getAcceptAllFileFilterUsed() {
        return acceptAllFileFilterUsed;
    }

    public JButton getBoutonXslLocation() {
        return boutonXslLocation;
    }

    public Boolean getDirectoryEnabled() {
        return directoryEnabled;
    }

    public String getExts() {
        return exts;
    }

    public String getExtsDescription() {
        return extsDescription;
    }

    public Boolean getFileEnabled() {
        return fileEnabled;
    }

    public FileEditorHandler getHandler() {
        return handler;
    }

    public JTextField getPathField() {
        return pathField;
    }

    public JButton getResetButton() {
        return resetButton;
    }

    public Boolean getShowReset() {
        return showReset;
    }

    public String getStartPath() {
        return startPath;
    }

    public String getTitle() {
        return title;
    }

    public JToolBar getToolbar() {
        return toolbar;
    }

    public Boolean isAcceptAllFileFilterUsed() {
        return acceptAllFileFilterUsed !=null && acceptAllFileFilterUsed;
    }

    public Boolean isDirectoryEnabled() {
        return directoryEnabled !=null && directoryEnabled;
    }

    public Boolean isFileEnabled() {
        return fileEnabled !=null && fileEnabled;
    }

    public Boolean isShowReset() {
        return showReset !=null && showReset;
    }

    /*-----------------------------------------------------------------------*/
    /*----------------------- Public mutator methods  -----------------------*/
    /*-----------------------------------------------------------------------*/

    public void setAcceptAllFileFilterUsed(Boolean acceptAllFileFilterUsed) {
        Boolean oldValue = this.acceptAllFileFilterUsed;
        this.acceptAllFileFilterUsed = acceptAllFileFilterUsed;
        firePropertyChange(PROPERTY_ACCEPT_ALL_FILE_FILTER_USED, oldValue, acceptAllFileFilterUsed);
    }

    public void setDirectoryEnabled(Boolean directoryEnabled) {
        Boolean oldValue = this.directoryEnabled;
        this.directoryEnabled = directoryEnabled;
        firePropertyChange(PROPERTY_DIRECTORY_ENABLED, oldValue, directoryEnabled);
    }

    public void setExts(String exts) {
        String oldValue = this.exts;
        this.exts = exts;
        firePropertyChange(PROPERTY_EXTS, oldValue, exts);
    }

    public void setExtsDescription(String extsDescription) {
        String oldValue = this.extsDescription;
        this.extsDescription = extsDescription;
        firePropertyChange(PROPERTY_EXTS_DESCRIPTION, oldValue, extsDescription);
    }

    public void setFileEnabled(Boolean fileEnabled) {
        Boolean oldValue = this.fileEnabled;
        this.fileEnabled = fileEnabled;
        firePropertyChange(PROPERTY_FILE_ENABLED, oldValue, fileEnabled);
    }

    public void setShowReset(Boolean showReset) {
        Boolean oldValue = this.showReset;
        this.showReset = showReset;
        firePropertyChange(PROPERTY_SHOW_RESET, oldValue, showReset);
    }

    public void setStartPath(String startPath) {
        String oldValue = this.startPath;
        this.startPath = startPath;
        firePropertyChange(PROPERTY_START_PATH, oldValue, startPath);
    }

    public void setTitle(String title) {
        String oldValue = this.title;
        this.title = title;
        firePropertyChange(PROPERTY_TITLE, oldValue, title);
    }

    /*-----------------------------------------------------------------------*/
    /*--------------------- Components creation methods ---------------------*/
    /*-----------------------------------------------------------------------*/

    protected void addChildrenToToolbar() {
        if (!allComponentsCreated) {
            return;
        }
        toolbar.add(resetButton);
    }

    protected void createAcceptAllFileFilterUsed() {
        $objectMap.put("acceptAllFileFilterUsed", acceptAllFileFilterUsed = Boolean.TRUE);
    }

    protected void createBoutonXslLocation() {
        $objectMap.put("boutonXslLocation", boutonXslLocation = new JButton());
        
        boutonXslLocation.setName("boutonXslLocation");
        boutonXslLocation.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__boutonXslLocation"));
    }

    protected void createDirectoryEnabled() {
        $objectMap.put("directoryEnabled", directoryEnabled = Boolean.TRUE);
    }

    protected void createExts() {
        $objectMap.put("exts", exts = null);
    }

    protected void createExtsDescription() {
        $objectMap.put("extsDescription", extsDescription = null);
    }

    protected void createFileEnabled() {
        $objectMap.put("fileEnabled", fileEnabled = Boolean.TRUE);
    }

    protected void createHandler() {
        $objectMap.put("handler", handler = new FileEditorHandler(this));
    }

    protected void createPathField() {
        $objectMap.put("pathField", pathField = new JTextField());
        
        pathField.setName("pathField");
        pathField.setColumns(15);
        pathField.addKeyListener(JAXXUtil.getEventListener(KeyListener.class, "keyReleased", this, "doKeyReleased__on__pathField"));
    }

    protected void createResetButton() {
        $objectMap.put("resetButton", resetButton = new JButton());
        
        resetButton.setName("resetButton");
        resetButton.setFocusPainted(false);
        resetButton.setFocusable(false);
        resetButton.setToolTipText(t("fileeditor.action.reset.tip"));
        resetButton.addActionListener(JAXXUtil.getEventListener(ActionListener.class, "actionPerformed", this, "doActionPerformed__on__resetButton"));
    }

    protected void createShowReset() {
        $objectMap.put("showReset", showReset = false);
    }

    protected void createStartPath() {
        $objectMap.put("startPath", startPath = null);
    }

    protected void createTitle() {
        $objectMap.put("title", title = null);
    }

    protected void createToolbar() {
        $objectMap.put("toolbar", toolbar = new JToolBar());
        
        toolbar.setName("toolbar");
        toolbar.setBorderPainted(false);
        toolbar.setFloatable(false);
    }

    /*-----------------------------------------------------------------------*/
    /*------------------------ Internal jaxx methods ------------------------*/
    /*-----------------------------------------------------------------------*/

    private void $completeSetup() {
        allComponentsCreated = true;
        if (log.isDebugEnabled()) {
            log.debug(this);
        }
        // inline complete setup of $BaseActionPanel0
        add(toolbar, BorderLayout.WEST);
        add(pathField, BorderLayout.CENTER);
        add(boutonXslLocation, BorderLayout.EAST);
        addChildrenToToolbar();
        
        // apply 5 data bindings
        JAXXUtil.applyDataBinding(this, $bindings.keySet());
        
        // apply 3 property setters
        resetButton.setIcon(SwingUtil.createActionIcon("fileeditor-reset"));
        boutonXslLocation.setIcon(SwingUtil.createActionIcon("open"));
    }

    private void $initialize() {
        if (allComponentsCreated) {
            return;
        }
        if (log.isDebugEnabled()) {
            log.debug(this);
        }
        $objectMap.put("$BaseActionPanel0", $BaseActionPanel0);
        createHandler();
        createAcceptAllFileFilterUsed();
        createDirectoryEnabled();
        createFileEnabled();
        createStartPath();
        createTitle();
        createExts();
        createExtsDescription();
        createShowReset();
        createToolbar();
        createResetButton();
        createPathField();
        createBoutonXslLocation();
        // inline creation of $BaseActionPanel0
        setName("$BaseActionPanel0");
        setLayout(new BorderLayout());
        
        // registers 5 data bindings
        $registerDefaultBindings();
        $completeSetup();
    }

    private void $registerDefaultBindings() {
        // register 5 data bindings
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_TOOLBAR_VISIBLE, true ,"showReset") {
        
            @Override
            public void processDataBinding() {
                toolbar.setVisible(isShowReset());
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_RESET_BUTTON_ENABLED, true ,"enabled") {
        
            @Override
            public void processDataBinding() {
                resetButton.setEnabled(isEnabled());
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_PATH_FIELD_ENABLED, true ,"enabled") {
        
            @Override
            public void processDataBinding() {
                pathField.setEnabled(isEnabled());
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_PATH_FIELD_TEXT, true ,"startPath") {
        
            @Override
            public void processDataBinding() {
                SwingUtil.setText(pathField, getStartPath());
            }
        });
        registerDataBinding(new SimpleJAXXObjectBinding(this, BINDING_BOUTON_XSL_LOCATION_ENABLED, true ,"enabled") {
        
            @Override
            public void processDataBinding() {
                boutonXslLocation.setEnabled(isEnabled());
            }
        });
    }

}
