/*
 * #%L
 * JAXX :: Widgets
 * 
 * $Id: BeanUIUtil.java 2088 2010-09-12 15:55:29Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jaxx/tags/jaxx-2.2.2/jaxx-widgets/src/main/java/jaxx/runtime/swing/editor/bean/BeanUIUtil.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package jaxx.runtime.swing.editor.bean;

import jaxx.runtime.decorator.Decorator;
import jaxx.runtime.decorator.DecoratorUtils;
import jaxx.runtime.decorator.JXPathDecorator;
import jaxx.runtime.decorator.MultiJXPathDecorator;
import jaxx.runtime.swing.JAXXButtonGroup;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.autocomplete.AutoCompleteDecorator;
import org.jdesktop.swingx.autocomplete.AutoCompletePropertyChangeListener;
import org.jdesktop.swingx.autocomplete.ObjectToStringConverter;

import javax.swing.AbstractButton;
import javax.swing.ButtonGroup;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPopupMenu;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.SwingUtilities;
import javax.swing.text.Document;
import javax.swing.text.JTextComponent;
import java.awt.Component;
import java.awt.Dimension;
import java.beans.Introspector;
import java.beans.PropertyChangeListener;
import java.util.Date;

import static org.nuiton.i18n.I18n._;
import static org.nuiton.i18n.I18n.n_;

/**
 * Class with usefull methods used in bean uis.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 2.2
 */
public class BeanUIUtil {

    public static final String DEFAULT_POPUP_LABEL = n_("bean.popup.label");

    public static final String DEFAULT_SELECTED_TOOLTIP = n_("bean.sort.on");

    public static final String DEFAULT_NOT_SELECTED_TOOLTIP = n_("bean.sort.off");

    /**
     * Encapsule un {@link Decorator} dans un {@link ObjectToStringConverter}.
     *
     * @param decorator le decorateur a encapsuler.
     * @return le converter encapsule dans un {@link ObjectToStringConverter}
     */
    public static ObjectToStringConverter newDecoratedObjectToStringConverter(final Decorator<?> decorator) {

        return new ObjectToStringConverter() {

            @Override
            public String getPreferredStringForItem(Object item) {
                return item instanceof String ? (String) item : item == null ? "" : decorator.toString(item);
            }
        };
    }

    /**
     * Ajout l'auto-complétion sur une liste déroulante, en utilisant le
     * converteur donné pour afficher les données.
     *
     * @param combo     la combo à décorer
     * @param convertor le converter utilisé pour afficher les données.
     */
    public static void decorate(JComboBox combo, ObjectToStringConverter convertor) {

        AutoCompleteDecorator.decorate(combo, convertor);
    }

    /**
     * Désactive l'aut-complétion sur une liste déroulante, en y repositionnant
     * le modèle du document d'édition d'avant auto-complétion.
     *
     * @param combo            la liste déroulante à décorer
     * @param originalDocument le document original de l'édtieur de la
     *                         liste déroulante.
     */
    public static void undecorate(JComboBox combo, Document originalDocument) {

        // has not to be editable
        combo.setEditable(false);

        // configure the text component=editor component
        Component c = combo.getEditor().getEditorComponent();
        JTextComponent editorComponent = (JTextComponent) c;
        editorComponent.setDocument(originalDocument);
        editorComponent.setText(null);
        //remove old property change listener
        for (PropertyChangeListener l : c.getPropertyChangeListeners("editor")) {
            if (l instanceof AutoCompletePropertyChangeListener) {
                c.removePropertyChangeListener("editor", l);
            }
        }
    }

    public static <O> MultiJXPathDecorator<O> createDecorator(JXPathDecorator<O> decorator) {
        if (decorator == null) {
            throw new NullPointerException(
                    "can not have a null decorator as parameter");
        }
        String separator;
        String separatorReplacement;

        if (decorator instanceof MultiJXPathDecorator<?>) {

            separator = ((MultiJXPathDecorator<?>) decorator).getSeparator();
            separatorReplacement = ((MultiJXPathDecorator<?>) decorator).getSeparatorReplacement();

        } else {

            separator = "??" + new Date().getTime();
            separatorReplacement = " - ";
        }

        return DecoratorUtils.newMultiJXPathDecorator(
                decorator.getInternalClass(),
                decorator.getInitialExpression(),
                separator,
                separatorReplacement
        );
    }

    public static abstract class PopupHandler implements Runnable {

        public static final Log log = LogFactory.getLog(PopupHandler.class);

        public abstract JPopupMenu getPopup();

        public abstract JComponent getInvoker();

        @Override
        public void run() {

            updatePopup();

            Dimension dim = getPopup().getPreferredSize();

            JComponent invoker = getInvoker();
            getPopup().show(
                    invoker,
                    (int) (invoker.getPreferredSize().getWidth() - dim.getWidth()),
                    invoker.getHeight()
            );
        }

        /** Toggle the popup visible state. */
        public void togglePopup() {
            boolean newValue = !getPopup().isVisible();

            if (log.isTraceEnabled()) {
                log.trace(newValue);
            }

            if (!newValue) {
                if (getPopup() != null) {
                    getPopup().setVisible(false);
                }
                return;
            }
            SwingUtilities.invokeLater(this);
        }

        protected void updatePopup() {
            getPopup().pack();
        }

        /**
         * Creation de l'ui pour modifier le décorateur.
         *
         * @param selectedTip
         * @param notSelectedTip
         * @param i18nPrefix
         * @param title
         * @param indexes
         * @param popupLabel
         * @param sortUp
         * @param sortDown
         * @param decorator le decorateur a utiliser
         */
        protected void preparePopup(String selectedTip,
                                    String notSelectedTip,
                                    String i18nPrefix,
                                    String title,
                                    ButtonGroup indexes,
                                    JLabel popupLabel,
                                    AbstractButton sortUp,
                                    AbstractButton sortDown,
                                    MultiJXPathDecorator<?> decorator) {
            if (selectedTip == null) {
                // use default selected tip text
                selectedTip = DEFAULT_SELECTED_TOOLTIP;
            }
            if (notSelectedTip == null) {
                // use default selected tip text
                notSelectedTip = DEFAULT_NOT_SELECTED_TOOLTIP;
            }
            JPopupMenu popup = getPopup();

            //Container container = ui.getIndexesContainer();
            for (int i = 0, max = decorator.getNbContext(); i < max; i++) {
                String property = i18nPrefix + decorator.getProperty(i);
                String propertyI18n = _(property);
                JRadioButtonMenuItem button = new JRadioButtonMenuItem(propertyI18n);
                button.putClientProperty(JAXXButtonGroup.BUTTON8GROUP_CLIENT_PROPERTY, indexes);
                button.putClientProperty(JAXXButtonGroup.VALUE_CLIENT_PROPERTY, i);
                popup.add(button);
                if (selectedTip != null) {
                    button.putClientProperty(JAXXButtonGroup.SELECTED_TIP_CLIENT_PROPERTY, _(selectedTip, propertyI18n));
                }
                if (notSelectedTip != null) {
                    button.putClientProperty(JAXXButtonGroup.NOT_SELECTED_TIP_CLIENT_PROPERTY, _(notSelectedTip, propertyI18n));
                }
                button.setSelected(false);
                indexes.add(button);
            }
            if (title == null) {
                // use default popup title
                title = DEFAULT_POPUP_LABEL;

                Class<?> internalClass = decorator.getInternalClass();
                String beanI18nKey;
                if (internalClass == null) {
                    beanI18nKey = n_("bean.unknown.type");
                } else {
                    beanI18nKey = i18nPrefix + Introspector.decapitalize(internalClass.getSimpleName());
                }
                String beanI18n = _(beanI18nKey);
                title = _(title, beanI18n);
            } else {
                title = _(title);
            }

            sortDown.putClientProperty(JAXXButtonGroup.SELECTED_TIP_CLIENT_PROPERTY, _("bean.sort.down.tip"));
            sortDown.putClientProperty(JAXXButtonGroup.NOT_SELECTED_TIP_CLIENT_PROPERTY, _("bean.sort.down.toSelect.tip"));

            sortUp.putClientProperty(JAXXButtonGroup.SELECTED_TIP_CLIENT_PROPERTY, _("bean.sort.up.tip"));
            sortUp.putClientProperty(JAXXButtonGroup.NOT_SELECTED_TIP_CLIENT_PROPERTY, _("bean.sort.up.toSelect.tip"));

            popupLabel.setText(title);
            getPopup().setLabel(title);
            getPopup().invalidate();
        }
    }
}
