/*
 * #%L
 * JAXX :: Runtime
 * 
 * $Id: SwingValidatorMessageListModel.java 1847 2010-04-16 12:27:48Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jaxx/tags/jaxx-2.1.1/jaxx-runtime/src/main/java/jaxx/runtime/validator/swing/SwingValidatorMessageListModel.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package jaxx.runtime.validator.swing;

import jaxx.runtime.validator.BeanValidatorEvent;
import jaxx.runtime.validator.BeanValidatorField;
import jaxx.runtime.validator.BeanValidatorListener;
import jaxx.runtime.validator.BeanValidatorScope;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.*;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

/**
 * The model of the list of validation's messages
 *
 * @author tchemit <chemit@codelutin.com>
 */
public class SwingValidatorMessageListModel
        extends AbstractListModel
        implements BeanValidatorListener {

    private static final long serialVersionUID = 1L;

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static Log log = LogFactory.getLog(SwingValidatorMessageListModel.class);

    /** list of registred validators */
    protected transient List<SwingValidator<?>> validators;

    /** list of messages actual displayed */
    protected List<SwingValidatorMessage> data;

    public SwingValidatorMessageListModel() {
        validators = new ArrayList<SwingValidator<?>>();
        data = new ArrayList<SwingValidatorMessage>();
    }

    public boolean isEmpty() {
        return getSize() == 0;
    }

    public void registerValidator(SwingValidator<?> validator) {
        if (validators.contains(validator)) {
            throw new IllegalArgumentException(
                    "the validator " + validator + " is already registred in "
                    + this);
        }
        validators.add(validator);
        validator.addBeanValidatorListener(this);
    }

    public void clear() {
        int i = data.size();
        if (i > 0) {
            data.clear();
            fireIntervalRemoved(this, 0, i - 1);
        }
    }

    @Override
    public int getSize() {
        return data.size();
    }

    @Override
    public Object getElementAt(int index) {
        ensureRowIndex(index);
        return data.get(index);
    }

    @Override
    public void onFieldChanged(BeanValidatorEvent event) {
        String[] toDelete = event.getMessagesToDelete();
        String[] toAdd = event.getMessagesToAdd();
        BeanValidatorField<?> field = event.getField();
        BeanValidatorScope scope = event.getScope();
        boolean mustAdd = toAdd != null && toAdd.length > 0;
        boolean mustDel = toDelete != null && toDelete.length > 0;

        if (log.isTraceEnabled()) {
            log.trace("----------------------------------------------------------");
            log.trace(field + " - (" + getSize() + ") toAdd     " + mustAdd);
            log.trace(field + " - (" + getSize() + ") toDelete  " + mustDel);
        }

        SwingValidator<?> validator = (SwingValidator<?>) event.getSource();

        if (mustDel) {

            // removes datas and notify if no messages to add
            removeMessages(validator, field, scope, !mustAdd, toDelete);
        }

        if (mustAdd) {

            // add new messages, sort datas and notify
            addMessages(validator, field, scope, true, toAdd);
        }
    }

    protected void ensureRowIndex(int index) throws ArrayIndexOutOfBoundsException {
        if (index < -1 || index >= getSize()) {
            throw new ArrayIndexOutOfBoundsException(
                    "the rowIndex was " + index + ", but should be int [0," +
                    (getSize() - 1) + "]");
        }
    }

    protected void addMessages(SwingValidator<?> validator,
                               BeanValidatorField<?> field,
                               BeanValidatorScope scope,
                               boolean sort,
                               String... messages) {

        JComponent editor = validator.getFieldRepresentation(field.getName());
        // add new errors
        for (String error : messages) {
            SwingValidatorMessage row = new SwingValidatorMessage(
                    validator,
                    field,
                    error,
                    scope,
                    editor
            );
            data.add(row);
            if (!sort) {
                fireIntervalAdded(this, data.size() - 1, data.size() - 1);
            }
        }

        if (sort) {

            // resort datas
            Collections.sort(data);

            // notify
            fireContentsChanged(this, 0, getSize() - 1);
        }
    }

    protected void removeMessages(SwingValidator<?> validator,
                                  BeanValidatorField<?> field,
                                  BeanValidatorScope scope,
                                  boolean notify,
                                  String... messages) {

        List<String> messagesToDel =
                new ArrayList<String>(Arrays.asList(messages));

        // do it in reverse mode (only one pass in that way since index
        // will stay coherent while removing them)

        for (int i = getSize() - 1; i > -1; i--) {
            SwingValidatorMessage error = data.get(i);
            if (error.getValidator().equals(validator) &&
                error.getScope() == scope &&
                error.getField().equals(field) &&
                messagesToDel.contains(error.getMessage())) {
                // remove the message
                data.remove(i);
                if (notify) {
                    fireIntervalRemoved(this, i, i);
                }
            }
        }
    }
}
