/*
 * #%L
 * JAXX :: Runtime
 * 
 * $Id: AbstractNavigationHelper.java 1986 2010-06-23 08:58:14Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jaxx/tags/jaxx-2.1.1/jaxx-runtime/src/main/java/jaxx/runtime/swing/navigation/AbstractNavigationHelper.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package jaxx.runtime.swing.navigation;

import jaxx.runtime.JAXXContext;
import jaxx.runtime.JAXXObject;
import jaxx.runtime.swing.navigation.handler.NavigationHandler;
import jaxx.runtime.swing.navigation.tree.NavigationTreeHelper;
import jaxx.runtime.swing.navigation.tree.NavigationTreeModel;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTreeTable;

import javax.swing.*;
import javax.swing.tree.TreePath;
import java.lang.reflect.InvocationTargetException;
import java.util.Enumeration;
import java.util.regex.Pattern;

/**
 * Helper object associated to a given navigation tree system.
 * <p/>
 * To helper is context safe (base on a {@link NavigationContextHelper}.
 *
 * @author letellier <letellier@codelutin.com>
 * @param <E> type of nodes in model
 * @see NavigationTreeModel
 * @since 2.0.1
 * @deprecated since 2.1, prefer use the simplify api {@code jaxx.runtime.swing.tree}.
 */
@Deprecated
public abstract class AbstractNavigationHelper<E extends NavigationNode<E>> extends NavigationContextHelper<E> {

    /** Logger */
    static private final Log log =
            LogFactory.getLog(NavigationTreeHelper.class);

    /**
     * Create the model.
     *
     * @param context the context to associate with fresh model
     * @return the new model build with data from the given context
     */
    public abstract NavigationModel<E> createTreeModel(JAXXContext context);

    /**
     * Create the tree handler.
     *
     * @param context the context to associate with fresh handler
     * @return the new handler
     */
    public abstract NavigationHandler<E> createTreeHandler(JAXXObject context);

    public AbstractNavigationHelper(String contextPrefix) {
        super(contextPrefix);
    }

    public Object getContextValue(JAXXContext context, String path)
            throws InvocationTargetException, NoSuchMethodException,
                   IllegalAccessException {
        NavigationModel<E> treeModel = getSafeModel(context);
        return treeModel.getBean(path);
    }

    public E findNode(JAXXContext context, String path) {
        NavigationModel<E> treeModel = getSafeModel(context);
        return treeModel.findNode(path);
    }

    public E findNode(JAXXContext context, String path,
                      String regex) {
        NavigationModel<E> treeModel = getSafeModel(context);
        return treeModel.findNode(path, regex);
    }

    public E findNode(JAXXContext context, String path,
                      Pattern regex) {

        NavigationModel<E> treeModel = getSafeModel(context);
        return treeModel.findNode(path, regex);
    }

    public E findNode(JAXXContext context, String path,
                      String regex, String suffix) {

        NavigationModel<E> treeModel = getSafeModel(context);

        E node = treeModel.findNode(path, regex);
        if (node != null && suffix != null) {
            node = treeModel.findNode(node, suffix);
        }
        return node;
    }

    public E findNode(JAXXContext context, String path,
                      Pattern regex, String suffix) {

        NavigationModel<E> treeModel = getSafeModel(context);

        E node = treeModel.findNode(path, regex);
        if (node != null && suffix != null) {
            node = treeModel.findNode(node, suffix);
        }
        return node;
    }

    /**
     * Sélection d'un noeud dans l'arbre de navigation à partir de son path.
     *
     * @param context le contexte applicatif
     * @param path    le path absolue du noeud dans l'arbre
     */
    public void selectNode(JAXXContext context, String path) {
        E node = findNode(context, path);
        if (log.isDebugEnabled()) {
            log.debug(path + " :: " + node);
        }
        if (node != null) {
            selectNode(context, node);
        }
    }

    /**
     * Sélection d'un noeud dans l'arbre de navigation.
     *
     * @param context le contexte applicatif
     * @param node    le noeud à sélectionner dans l'arbre
     */
    public void selectNode(JAXXContext context, E node) {


        NavigationModel<E> navigationModel = getSafeModel(context);
        if (log.isDebugEnabled()) {
            log.debug(node);
        }
        TreePath path = new TreePath(navigationModel.getPathToRoot(node));
        JTree tree = getTree(context);
        if (tree == null) {
            // FIXME : Refactor in other helper ?
            // If its JXTreeTable
            JXTreeTable treeTable = getSafeTreeTable(context);
            treeTable.getTreeSelectionModel().setSelectionPath(path);
            treeTable.scrollPathToVisible(path);
            return;
        }
        tree.setSelectionPath(path);
        tree.scrollPathToVisible(path);
    }

    /**
     * Sélection du parent du noeud selectionne dans l'arbre de navigation.
     *
     * @param context le contexte applicatif
     */
    public void gotoParentNode(JAXXContext context) {

        E node = getSelectedNode(context);

        if (node == null) {
            // pas de noeud selectionne
            throw new NullPointerException("no selected node in context");
        }
        node = node.getParent();

        selectNode(context, node);
    }

    /**
     * Obtain the first ancestor with the matching internalClass
     *
     * @param current   the node to test
     * @param beanClass the type of the internal class to seek of
     * @return the first ancestor node with the matching class or {@code null}
     *         if not found
     */
    public E getParentNode(E current,
                           Class<?> beanClass) {
        if (current == null) {
            // ancestor not found
            return null;
        }
        if (beanClass.isAssignableFrom(current.getInternalClass())) {
            // matching node
            return current;
        }
        // try in the parent of node
        return getParentNode(current.getParent(), beanClass);
    }

    /**
     * Sélection d'un fils du noeud selectionne dans l'arbre de navigation.
     *
     * @param context    le contexte applicatif
     * @param childIndex index du fils a selectionner
     */
    public void gotoChildNode(JAXXContext context, int childIndex) {

        E node = getSelectedNode(context);

        if (node == null) {
            // pas de noeud selectionne
            throw new NullPointerException("no selected node in context");
        }
        node = node.getChildAt(childIndex);

        selectNode(context, node);
    }

    /**
     * Demande une opération de repaint sur un noeud de l'arbre de navigation.
     * <p/>
     * <b>Note:</b> La descendance du noeud n'est pas repainte.
     *
     * @param context le contexte applicatif
     * @param node    le noeud à repaindre
     */
    public void repaintNode(JAXXContext context, E node) {
        repaintNode(context, node, false);
    }

    /**
     * Demande une opération de repaint sur un noeud de l'arbre de navigation.
     * <p/>
     * <b>Note:</b> La descendance du noeud est repainte si le paramètre
     * <code>deep</code> est à <code>true</code>.
     *
     * @param context le contexte applicatif
     * @param node    le noeud à repaindre
     * @param deep    un flag pour activer la repainte de la descendance du
     *                noeud
     */
    public void repaintNode(JAXXContext context, E node,
                            boolean deep) {
        NavigationModel<E> navigationModel = getSafeModel(context);
        if (log.isDebugEnabled()) {
            log.debug(node);
        }
        navigationModel.nodeChanged(node);
        if (deep) {
            // repaint childs nodes
            //todo we should only repaint necessary nodes ?
            Enumeration<? extends E> e = node.children();
            while (e.hasMoreElements()) {
                E child = e.nextElement();
                repaintNode(context, child, true);
            }
        }
    }

    public NavigationModel<E> getSafeModel(JAXXContext context)
            throws NullPointerException {
        NavigationModel<E> model = getModel(context);
        if (model == null) {
            throw new NullPointerException(
                    "could not find tree model with key " +
                    getModelContextEntry() + " in context " + context);
        }
        return model;
    }

    public JTree getSafeTree(JAXXContext context) throws NullPointerException {
        JTree tree = getTree(context);
        if (tree == null) {
            throw new NullPointerException(
                    "could not find tree with key " + getTreeContextEntry() +
                    " in context " + context);
        }
        return tree;
    }

    public JXTreeTable getSafeTreeTable(JAXXContext context)
            throws NullPointerException {
        JXTreeTable treeTable = getTreeTable(context);
        if (treeTable == null) {
            throw new NullPointerException(
                    "could not find tree with key " +
                    getTreeTableContextEntry() + " in context " + context);
        }
        return treeTable;
    }
}
