package org.nuiton.jaxx.application.swing.tab;

/*
 * #%L
 * JAXX :: Application Swing
 * %%
 * Copyright (C) 2008 - 2014 Code Lutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import jaxx.runtime.SwingUtil;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.beans.BeanUtil;

import javax.swing.AbstractButton;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTabbedPane;
import javax.swing.UIManager;
import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import static org.nuiton.i18n.I18n.t;

/**
 * Custom tab component which adds a * in the end of the title
 * when the content is modified.
 *
 * @author Kevin Morin - morin@codelutin.com
 * @since 2.8
 */
public class CustomTab extends JPanel {

    private static final long serialVersionUID = 1L;

    private static final Log log = LogFactory.getLog(CustomTab.class);

    protected TabContentModel model;

    protected TabContainerHandler handler;

    protected JLabel title = new JLabel();

    public TabContentModel getModel() {
        return model;
    }

    public CustomTab(TabContentModel model, TabContainerHandler handler) {
        super(new FlowLayout(FlowLayout.LEFT, 0, 0));

        this.model = model;
        this.handler = handler;

        try {
            BeanUtil.addPropertyChangeListener(
                    new PropertyChangeListener() {

                        @Override
                        public void propertyChange(PropertyChangeEvent evt) {
                            updateTitle();
                        }
                    }, this.model);

        } catch (Exception ex) {
            log.error("Error while adding the listener to the model modifications", ex);
        }

        setBackground(null);

        updateTitle();
        String actionIcon = model.getIcon();
        if (actionIcon != null) {
            title.setIcon(SwingUtil.createActionIcon(actionIcon));
        }
        title.setBorder(BorderFactory.createEmptyBorder(0, 0, 0, 5));
        add(title);

        if (model.isCloseable()) {
            JButton button = new TabButton();
            add(button);
        }

        setBorder(BorderFactory.createEmptyBorder(2, 0, 0, 0));
    }

    @Override
    public void setBackground(Color bg) {
        if (bg == null) {
            bg = new Color(0, 0, 0, 0);
        }
        super.setBackground(bg);
        revalidate();
    }

    @Override
    public void setForeground(Color fg) {
        super.setForeground(fg);
        if (title != null) {
            title.setForeground(fg);
        }
    }

    protected void updateTitle() {
        Font f = UIManager.getDefaults().getFont("Label.font");
        String titleValue = t(model.getTitle());

        int style;
        if (model.isModify()) {
            style = Font.BOLD;
            titleValue += "*";

        } else if (model.isEmpty()) {
            style = Font.ITALIC;

        } else {
            style = Font.PLAIN;
        }

        title.setText(titleValue);
        title.setFont(f.deriveFont(style));
    }

    protected class TabButton extends JButton implements ActionListener {

        public TabButton() {
            int size = 17;
            setPreferredSize(new Dimension(size, size));
            setToolTipText(t("jaxx.application.tab.customtab.close.label"));
            //Make it transparent
            setContentAreaFilled(false);
            //No need to be focusable
            setFocusable(false);
            setBorder(BorderFactory.createEtchedBorder());
            setBorderPainted(false);
            //Making nice rollover effect
            //we use the same listener for all buttons
            addMouseListener(buttonMouseListener);
            setRolloverEnabled(true);
            //Close the proper tab by clicking the button
            addActionListener(this);
        }

        @Override
        public void actionPerformed(ActionEvent e) {
//            if (tabbedPane != null) {
//                if (i != -1) {
//                    tabbedPane.remove(i);
//                }
//            }
            if (handler != null) {
                JTabbedPane tabPanel = handler.getTabPanel();
                int i = tabPanel.indexOfTabComponent(CustomTab.this);
                handler.removeTab(i);
            }
        }

        //paint the cross
        @Override
        protected void paintComponent(Graphics g) {
            super.paintComponent(g);

            Graphics2D g2 = (Graphics2D) g.create();
            //shift the image for pressed buttons
            if (getModel().isPressed()) {
                g2.translate(1, 1);
            }
            g2.setStroke(new BasicStroke(2));
            if (getModel().isRollover()) {
//                g2.setBackground(Color.MAGENTA);
                g2.setColor(Color.MAGENTA);
            } else {
                g2.setColor(Color.BLACK);
            }
            int delta = 5;
            g2.drawLine(delta, delta, getWidth() - delta - 1, getHeight() - delta - 1);
            g2.drawLine(getWidth() - delta - 1, delta, delta, getHeight() - delta - 1);
            g2.dispose();
        }
    }

    private MouseListener buttonMouseListener = new MouseAdapter() {
        public void mouseEntered(MouseEvent e) {
            Component component = e.getComponent();
            if (component instanceof AbstractButton) {
                AbstractButton button = (AbstractButton) component;
                button.setBorderPainted(true);
            }
        }

        public void mouseExited(MouseEvent e) {
            Component component = e.getComponent();
            if (component instanceof AbstractButton) {
                AbstractButton button = (AbstractButton) component;
                button.setBorderPainted(false);
            }
        }
    };
}
