package javax.constraints.impl.search;

import javax.constraints.ProblemState;
import javax.constraints.SearchStrategy;
import javax.constraints.Solution;
import javax.constraints.Solver;
import javax.constraints.Var;
import javax.constraints.impl.AbstractProblem;

/**
 * This class is used by the Solver's method "findOptimalSolutionDichotomize". Its method
 * "execute" is trying to find a solution that minimizes the parameter
 * "objective" using the current search strategy.
 * The "tolerance" specifies a difference between different optimal
 * solutions that can be ignored. It may be considered as a precision of the
 * search algorithm.
 *
 */

public class Dichotomize {

	Solver solver;
	Var objective;
	SearchStrategy searchStrategy;
	int objectiveMin;
	int objectiveMax;
	int tolerance;
	int numberOfSolutions;
	int numberOfChoicePoints;
	int numberOfFailures;
	int numberOfTries;
	boolean checkLowerHalf;
	Solution solution;
	AbstractProblem p;
	int prevMax;
	int midObjective;
	int totalTimeLimit; // in seconds. 0 - means no total time limit
	long startTime;

	public Dichotomize(Solver solver, Var objective) {
		this.solver = solver;
		p = (AbstractProblem)solver.getProblem();
		this.searchStrategy = solver.getSearchStrategy();
		this.objective = objective;
		if (objective.getName().isEmpty())
			objective.setName("Objective"); 
		if (p.getVar(objective.getName()) == null) {
			p.add(objective);
		}
		
		objectiveMin = objective.getMin();
		objectiveMax = objective.getMax();
		this.tolerance = solver.getOptimizationTolerance();
		this.totalTimeLimit = solver.getTimeLimit();
		startTime = System.currentTimeMillis();
		checkLowerHalf = false;
		numberOfTries = 0;
		solution = null;
		prevMax = 0;
		midObjective = 0;

	}

	/**
	 * The actual minimization algorithm executes a dichotomized search. During
	 * the search it modifies an interval [objectiveMin; objectiveMax]. First it
	 * is trying to find a solution in the [objectiveMin; objectiveMid]. If it
	 * fails, it is looking at [objectiveMid+1; objectiveMax]. During this
	 * process it switches the search target: one time in looks at in the upper
	 * half of the selected interval, another time - to the lower half.
	 * Successful search stops when objectiveMax - objectiveMin <= tolerance.
	 */
	public Solution execute() {

		p.debug("Dichotomize with objective[" + objectiveMin + ";"
				+ objectiveMax + "]");

		numberOfTries++;

		// dichotomized search
		solver.setTimeLimitStart(); // reset TimeLimit for one solution search
//		long currTime = System.currentTimeMillis();
//		int timeRemaining = (int)((totalTimeLimit*1000)-(currTime-startTime));
//		if( timeRemaining > 0 && timeRemaining < p.getTimeLimit() ) {
//			solver.setTimeLimit( (timeRemaining) );
//		}
		
		Solution newSolution = null;
		// Check TotalTimeLimit
//		if (totalTimeLimit > 0) {
//			currTime = System.currentTimeMillis();
//			if ( totalTimeLimit * 1000 < currTime - startTime) {
//				p.setTimeLimitExceeded(true);
//				p.log("Elapsed total time: " + (currTime - startTime) + " mills");
//				p.log("The search is interrupted by TotalTimeLimit "+totalTimeLimit+ " seconds");
//				return (newSolution!=null?newSolution:solution); // THE END !!!
//			}
//		}
		
		try {
			p.post(objective,">=",objectiveMin);
			p.post(objective,"<=",objectiveMax);
			newSolution = solver.findSolution(ProblemState.RESTORE);
		} catch (Exception e) {
			p.debug("No solutions within [" + objectiveMin + ";" + objectiveMax +"]");
		}
		

		if (newSolution != null) {
			// success
			// if there is a solution in this interval:
			// 1) objectiveMax = objectiveValue - 1
			// 2) check tolerance condition, if satisfied return solution, else
			// 3) split current interval in lower and upper half
			// 4) consider lower part
			numberOfSolutions++;
			solution = newSolution;
			solution.setSolutionNumber(numberOfSolutions);
			//TODO fix this for VarReal objectives..
			int objectiveValue = solution.getValue(objective.getName());
			p.debug("Found solution #" + numberOfSolutions + " objective="
					+ objectiveValue);
			//solution.log();

			objectiveMax = objectiveValue - tolerance;

			if (java.lang.Math.abs(objectiveValue - objectiveMin) <= 0) {
				p.debug("This solution is optimal!");
				return solution; // THE END !!!
			}
			// Check MaxNumnerOfSolutions
			if (solver.getMaxNumberOfSolutions() > 0
				&& numberOfSolutions == solver.getMaxNumberOfSolutions()) {
				p.log("The search is interrupted: MaxNumberOfSolutions has been reached");
				return solution; // THE END !!!
			}

			midObjective = (int) Math.floor((objectiveMin + objectiveMax) / 2);
			p.debug(objective.toString());
			p.debug("Try objective [" + objectiveMin + ";" + midObjective + "]");
			prevMax = objectiveMax;
			objectiveMax = midObjective;
			checkLowerHalf = true;
		} else {
			// failure
			// if there is no solution in this interval:
			// 1) if we are checking upper half and solution != null, return
			// solution, else
			// 2) check upper half
			p.debug("Failure!");

			// if (objectiveMax - objectiveMin <= tolerance+1) {
			// String text = "No solutions";
			// if (solution != null)
			// text = "Last solution was optimal!";
			// p.debug(text);
			// return solution; // previously found solution or null
			// }

			if (checkLowerHalf) {
				midObjective++;
				objectiveMax = prevMax;
				if(midObjective > objectiveMax)
					return solution;
				p.debug("Try objective [" + midObjective + ";" + objectiveMax
						+ "]");
				objectiveMin = midObjective;
				checkLowerHalf = false;
			} else {
				String text = "No solutions";
				if (solution != null)
					text = "Last solution was optimal!";
				p.debug(text);
				return solution; // previously found solution or null
			}

		}
		return execute();
	}
}
