/*
 * Decompiled with CFR 0.152.
 */
package org.chorem.pollen.services.service.security;

import com.auth0.jwt.JWT;
import com.auth0.jwt.JWTVerifier;
import com.auth0.jwt.algorithms.Algorithm;
import com.auth0.jwt.exceptions.JWTVerificationException;
import com.auth0.jwt.interfaces.DecodedJWT;
import com.google.common.base.Preconditions;
import com.google.common.primitives.Longs;
import java.nio.ByteBuffer;
import java.time.Clock;
import java.time.ZoneId;
import java.util.Arrays;
import java.util.Base64;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.chorem.pollen.persistence.entity.PollenPrincipal;
import org.chorem.pollen.persistence.entity.PollenPrincipalDao;
import org.chorem.pollen.persistence.entity.PollenToken;
import org.chorem.pollen.persistence.entity.PollenTokenDao;
import org.chorem.pollen.persistence.entity.PollenUser;
import org.chorem.pollen.persistence.entity.PollenUserEmailAddress;
import org.chorem.pollen.persistence.entity.UserCredential;
import org.chorem.pollen.persistence.topia.TopiaIdFactory;
import org.chorem.pollen.persistence.topia.TopiaNoResultException;
import org.chorem.pollen.services.bean.PollenEntityId;
import org.chorem.pollen.services.bean.PollenEntityRef;
import org.chorem.pollen.services.service.PollenServiceSupport;
import org.chorem.pollen.services.service.security.PollenAuthenticationException;
import org.chorem.pollen.services.service.security.PollenEmailNotValidatedException;
import org.chorem.pollen.services.service.security.PollenInvalidPasswordException;
import org.chorem.pollen.services.service.security.PollenSecurityContext;
import org.chorem.pollen.services.service.security.PollenUnauthorizedException;
import org.chorem.pollen.services.service.security.PollenUserBannedException;
import org.chorem.pollen.services.service.security.PollenUserUnknownException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SecurityService
extends PollenServiceSupport {
    private static final Logger log = LoggerFactory.getLogger(SecurityService.class);

    @Override
    public void checkIsConnected() {
        PollenSecurityContext securityContext = this.getSecurityContext();
        if (!securityContext.isConnected()) {
            throw new PollenUnauthorizedException("connected");
        }
    }

    @Override
    public void checkIsConnectedRequired() {
        if (this.getPollenServiceConfig().isUserConnectedRequired()) {
            this.checkIsConnected();
        }
    }

    @Override
    public void checkIsAdmin() {
        PollenSecurityContext securityContext = this.getSecurityContext();
        if (!securityContext.isAdmin()) {
            throw new PollenUnauthorizedException("administrator");
        }
    }

    public PollenEntityRef<PollenUser> login(String login, String password, Boolean rememberMe) throws PollenAuthenticationException, PollenEmailNotValidatedException, PollenUserBannedException {
        PollenUser user;
        block7: {
            if (StringUtils.isBlank((CharSequence)password)) {
                throw new PollenAuthenticationException();
            }
            user = this.getPollenUserDao().findUserWithEmailAddressOrNull(login);
            if (user == null) {
                throw new PollenAuthenticationException();
            }
            try {
                this.checkUserPassword(user, password);
            }
            catch (PollenInvalidPasswordException e) {
                boolean valid = false;
                for (UserCredential credential : CollectionUtils.emptyIfNull((Collection)user.getUserCredential())) {
                    valid = valid || this.getUserCredentialDao().isCredentialValid(credential.getProvider(), credential.getUserId(), user.getTopiaId(), login);
                }
                if (valid) break block7;
                throw new PollenAuthenticationException(e);
            }
        }
        if (!user.isEmailValidated()) {
            throw new PollenEmailNotValidatedException();
        }
        if (user.isBanned()) {
            throw new PollenUserBannedException();
        }
        this.getSecurityContext().setPollenUser(user);
        return PollenEntityRef.of(user);
    }

    public void logout() {
        this.getSecurityContext().setPollenUser(null);
    }

    public void lostPassword(String login) throws PollenUserUnknownException, PollenEmailNotValidatedException {
        Preconditions.checkNotNull((Object)login);
        PollenUser user = this.getPollenUserDao().findUserWithEmailAddressOrNull(login);
        if (user == null) {
            throw new PollenUserUnknownException();
        }
        if (!user.isEmailValidated()) {
            throw new PollenEmailNotValidatedException();
        }
        String newPassword = this.serviceContext.generatePassword();
        this.getSecurityService().setUserPassword(user, newPassword);
        this.commit();
        this.getNotificationService().onUserLostPasswordAsked(user, newPassword);
    }

    public void deleteObsoleteSessionTokens() {
        Set sessionTokens = this.getSessionTokenDao().findAllBeforeEndDate(this.getNow());
        if (log.isDebugEnabled()) {
            sessionTokens.forEach(sessionToken -> log.debug(String.format("SessionToken %s delete expired : %s", sessionToken.getPollenToken().getToken(), sessionToken.getPollenToken().getEndDate())));
        }
        this.getSessionTokenDao().deleteAll((Iterable)sessionTokens);
        this.commit();
    }

    public Optional<String> getToken() {
        if (this.getSecurityContext().isConnected()) {
            String token = this.getToken(this.getConnectedUser());
            return Optional.of(token);
        }
        return Optional.empty();
    }

    private String getToken(PollenUser user) {
        Date now = this.getNow();
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(now);
        calendar.add(13, this.getPollenServiceConfig().getTokenTimeout());
        Date expireDate = calendar.getTime();
        return JWT.create().withIssuer(this.getPollenServiceConfig().getTokenIssue()).withIssuedAt(now).withExpiresAt(expireDate).withSubject(PollenEntityId.getReduceId(user)).sign(this.getAlgorithm());
    }

    public Optional<PollenUser> getUserFromToken(String token) {
        PollenUser user = null;
        if (StringUtils.isNotBlank((CharSequence)token)) {
            try {
                JWTVerifier.BaseVerification verification = (JWTVerifier.BaseVerification)JWT.require((Algorithm)this.getAlgorithm()).withIssuer(this.getPollenServiceConfig().getTokenIssue());
                Clock now = Clock.fixed(this.getNow().toInstant(), ZoneId.systemDefault());
                JWTVerifier verifier = verification.build(now);
                DecodedJWT jwt = verifier.verify(token);
                String userId = jwt.getSubject();
                String pollenUserTopiaId = TopiaIdFactory.newTopiaId(PollenUser.class, (String)userId);
                user = (PollenUser)this.getPollenUserDao().forTopiaIdEquals(pollenUserTopiaId).findUnique();
            }
            catch (JWTVerificationException e) {
                log.info("Invalid Token : {} - {}", (Object)e.getMessage(), (Object)token);
            }
            catch (TopiaNoResultException e) {
                log.error("User not found user from token : {}", (Object)token);
            }
        }
        return Optional.ofNullable(user);
    }

    protected Algorithm getAlgorithm() {
        byte[] secret = this.getPollenServiceConfig().getTokenSecretBytes();
        return Algorithm.HMAC256((byte[])secret);
    }

    public String generateEmailToken(PollenUserEmailAddress email) {
        long expireDate = this.getNow().getTime() + (long)this.getPollenServiceConfig().getTokenTimeout() * 1000L;
        byte[] bytesEmail = email.getEmailAddress().getBytes();
        ByteBuffer buffer = ByteBuffer.allocate(8 + bytesEmail.length);
        buffer.putLong(expireDate);
        buffer.put(bytesEmail);
        byte[] message = buffer.array();
        byte[] encryptMessage = this.getCryptoService().encryptMessageSymmetric(message);
        return Base64.getUrlEncoder().encodeToString(encryptMessage);
    }

    public PollenUserEmailAddress getEmailAdresseFromToken(String token) {
        PollenUserEmailAddress pollenUserEmailAddress = null;
        if (StringUtils.isNotBlank((CharSequence)token)) {
            byte[] encryptMessage = Base64.getUrlDecoder().decode(token);
            byte[] message = this.getCryptoService().decryptMessageSymmetric(encryptMessage);
            byte[] bytesExpireDate = Arrays.copyOfRange(message, 0, 8);
            byte[] bytesEmail = Arrays.copyOfRange(message, 8, message.length);
            long expireDate = Longs.fromByteArray((byte[])bytesExpireDate);
            String emailAdresse = new String(bytesEmail);
            if (this.getNow().getTime() <= expireDate) {
                pollenUserEmailAddress = (PollenUserEmailAddress)this.getPollenUserEmailAddressDao().forEmailAddressEquals(emailAdresse).findUniqueOrNull();
            }
        }
        return pollenUserEmailAddress;
    }

    public PollenToken generateNewToken() {
        String token = this.serviceContext.generateToken();
        PollenTokenDao dao = this.getPersistenceContext().getPollenTokenDao();
        PollenToken pollenToken = dao.createByNaturalId(token);
        pollenToken.setCreationDate(this.serviceContext.getNow());
        return pollenToken;
    }

    public PollenPrincipal generatePollenPrincipal() {
        PollenPrincipalDao dao = this.getPersistenceContext().getPollenPrincipalDao();
        PollenPrincipal principal = (PollenPrincipal)dao.create();
        PollenToken token = this.generateNewToken();
        principal.setPermission(token);
        return principal;
    }

    public Optional<PollenPrincipal> getPollenPrincipalByPermissionToken(String permissionToken) {
        PollenPrincipal principal = null;
        if (permissionToken != null) {
            principal = this.getPollenPrincipalDao().findByPermissionToken(permissionToken);
        }
        return Optional.ofNullable(principal);
    }

    public void setUserPassword(PollenUser user, String newPassword) {
        String salt = this.serviceContext.generateSalt();
        String encodedPassword = this.serviceContext.encodePassword(salt, newPassword);
        user.setSalt(salt);
        user.setPassword(encodedPassword);
    }

    public void checkUserPassword(PollenUser user, String password) throws PollenInvalidPasswordException {
        boolean valid;
        String encodedPassword = null;
        if (password != null && user.getSalt() != null) {
            encodedPassword = this.serviceContext.encodePassword(user.getSalt(), password);
        }
        if (!(valid = Objects.equals(encodedPassword, user.getPassword()))) {
            for (UserCredential credential : CollectionUtils.emptyIfNull((Collection)user.getUserCredential())) {
                valid = valid || this.getUserCredentialDao().isCredentialValid(credential.getProvider(), credential.getUserId(), user.getTopiaId(), credential.getEmail());
            }
        }
        if (user.isBanned() || !valid) {
            throw new PollenInvalidPasswordException();
        }
    }
}

