/*
 * #%L
 * Lima Swing
 * 
 * $Id: FinancialTransactionsPane.java 3777 2014-04-14 07:29:08Z sbavencoff $
 * $HeadURL: https://svn.chorem.org/lima/tags/lima-0.7.3/lima-swing/src/main/java/org/chorem/lima/ui/home/FinancialTransactionsPane.java $
 * %%
 * Copyright (C) 2008 - 2012 CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package org.chorem.lima.ui.home;

import static org.nuiton.i18n.I18n.t;

import java.util.List;

import javax.swing.event.HyperlinkEvent;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.LimaContext;
import org.chorem.lima.business.LimaException;
import org.chorem.lima.business.ServiceListener;
import org.chorem.lima.business.api.EntryBookService;
import org.chorem.lima.business.api.FinancialTransactionService;
import org.chorem.lima.business.api.FiscalPeriodService;
import org.chorem.lima.business.api.ImportService;
import org.chorem.lima.entity.EntryBook;
import org.chorem.lima.entity.FinancialTransaction;
import org.chorem.lima.entity.FiscalPeriod;
import org.chorem.lima.service.LimaServiceFactory;
import org.chorem.lima.ui.MainView;

import com.google.common.collect.Multiset.Entry;

/**
 * Home view pane that display statistics about unbalanced transactions.
 * 
 * @author chatellier
 * @version $Revision: 3777 $
 * 
 * Last update : $Date: 2014-04-14 09:29:08 +0200 (Mon, 14 Apr 2014) $
 * By : $Author: sbavencoff $
 */
public class FinancialTransactionsPane extends AbstractHomePane implements ServiceListener {

    private static final long serialVersionUID = 1L;

    private static final Log log = LogFactory.getLog(FinancialTransactionsPane.class);

    protected FinancialTransactionService financialTransactionService;

    protected FiscalPeriodService fiscalPeriodService;
    
    protected EntryBookService entryBookService;

    public FinancialTransactionsPane(HomeView view) {
        super(view);

        financialTransactionService = LimaServiceFactory.getService(FinancialTransactionService.class);
        fiscalPeriodService = LimaServiceFactory.getService(FiscalPeriodService.class);
        entryBookService = LimaServiceFactory.getService(EntryBookService.class);
        LimaServiceFactory.addServiceListener(FinancialTransactionService.class, this);
        LimaServiceFactory.addServiceListener(FiscalPeriodService.class, this);
        LimaServiceFactory.addServiceListener(ImportService.class, this);

        refresh();
    }

    @Override
    public void hyperlinkUpdate(HyperlinkEvent e) {
        if (HyperlinkEvent.EventType.ACTIVATED.equals(e.getEventType())) {

            if (e.getDescription().equals("#financialtransactionunbalanced")) {
                MainView ui = LimaContext.MAIN_UI_ENTRY_DEF.getContextValue(view);
                ui.getHandler().showTransactionUnbalancedView(ui);
            } else if (e.getDescription().equals("#financialtransactionbalanced")) {
                MainView ui = LimaContext.MAIN_UI_ENTRY_DEF.getContextValue(view);
                ui.getHandler().showTransactionView(ui);
            }
        }
    }

    public void refresh() {
        
        log.debug("Rafraîchissement financial transaction pane");
        
        String htmlBegin = "<font face='sans-serif' size=3>"
            + "<p style=vertical-align:'bottom', horizontal-align:'center'>";
        String htmlEnd = "</p></font>";

        try {

            List<FiscalPeriod> unblockedFiscalPeriods =
                fiscalPeriodService.getAllUnblockedFiscalPeriods();
            List<EntryBook> entryBooks = entryBookService.getAllEntryBooks();
            
            //Au moins un exercice ouvert et un journal ouvert
            if (unblockedFiscalPeriods.size() != 0 && entryBooks.size() != 0) {
                List<FinancialTransaction> financialTransactionsInexact =
                    financialTransactionService.
                        getAllInexactFinancialTransactions(unblockedFiscalPeriods.get(0));
                if (financialTransactionsInexact.size() > 0) {
                    setBackground(RED_BACKGROUND);
                    String transactionsString = financialTransactionsInexact.size()
                                                + " " + t("lima.ui.home.transaction.inexact")
                                                + "<br/><br/><a href='#financialtransactionunbalanced'>"
                                                + t("lima.ui.home.transaction.modifiy.inexact") + "</a>";
                    //set Text
                    setText(htmlBegin + transactionsString + htmlEnd);
                } else {
                    List<FinancialTransaction> financialTransactionsBal =
                            financialTransactionService.getAllFinancialTransactionsBalanced(unblockedFiscalPeriods.get(0));

                    if (financialTransactionsBal.size() > 0) {
                        setBackground(GREEN_BACKGROUND);
                        String transactionsString = financialTransactionsBal.size()
                                                    + " " + t("lima.ui.home.transaction.balanced")
                                                    + "<br/><br/><a href='#financialtransactionbalanced'>"
                                                    + t("lima.ui.home.transaction.modifiy.balanced") + "</a>";
                        setText(htmlBegin + transactionsString + htmlEnd);
                    } else {
                        setBackground(RED_BACKGROUND);
                        String transactionsString = t("lima.ui.home.transaction.nothing")
                                                    + "<br/><br/><a href='#financialtransactionbalanced'>"
                                                    + t("lima.ui.home.transaction.create") + "</a>";
                        setText(htmlBegin + transactionsString + htmlEnd);
                    }

                }
            }else {
                setBackground(RED_BACKGROUND);
                setText("");
            }

        } catch (LimaException eee) {
            log.debug("Can't get datas account editor pane home", eee);
        }
    }

    @Override
    public void notifyMethod(String serviceName, String methodeName) {        
        log.debug("Nom de la méthode : " + methodeName);
        if (methodeName.contains("FiscalPeriod") || methodeName.contains("FinancialTransaction")
            || methodeName.contains("Entry") || methodeName.contains("importEntries")
            || methodeName.contains("importAll") || methodeName.contains("importAs")) {            
            refresh();
        }
    }

}
