/*
 * #%L
 * Lima Swing
 * 
 * $Id: AccountViewHandler.java 3780 2014-05-05 16:28:39Z dcosse $
 * $HeadURL: https://svn.chorem.org/lima/tags/lima-0.7.2/lima-swing/src/main/java/org/chorem/lima/ui/account/AccountViewHandler.java $
 * %%
 * Copyright (C) 2008 - 2012 CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package org.chorem.lima.ui.account;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.business.ServiceListener;
import org.chorem.lima.business.api.AccountService;
import org.chorem.lima.business.api.ImportService;
import org.chorem.lima.entity.Account;
import org.chorem.lima.enums.AccountsChartEnum;
import org.chorem.lima.enums.ImportExportEnum;
import org.chorem.lima.service.LimaServiceFactory;
import org.chorem.lima.ui.importexport.ImportExport;
import org.jdesktop.swingx.JXTreeTable;
import org.jdesktop.swingx.treetable.DefaultMutableTreeTableNode;
import org.jdesktop.swingx.treetable.DefaultTreeTableModel;
import org.jdesktop.swingx.treetable.MutableTreeTableNode;
import org.jdesktop.swingx.treetable.TreeTableNode;

import javax.swing.*;
import javax.swing.tree.TreePath;
import java.awt.event.ActionEvent;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;

import static org.nuiton.i18n.I18n.t;

/**
 * Handler associated with account view.
 *
 * @author chatellier
 * @version $Revision: 3780 $
 *          <p/>
 *          Last update : $Date: 2014-05-05 18:28:39 +0200 (Mon, 05 May 2014) $
 *          By : $Author: dcosse $
 */
public class AccountViewHandler implements ServiceListener {

    /** log. */
    private static final Log log = LogFactory.getLog(AccountViewHandler.class);

    protected AccountService accountService;

    protected AccountView view;

    /**
     * Sort account with label length.
     */
    protected static Comparator<Account> accountLengthComparator = new Comparator<Account>() {
        @Override
        public int compare(Account o1, Account o2) {
            int result = o1.getAccountNumber().length() - o2.getAccountNumber().length();
            if (result == 0) {
                // same length, compare accountNumber
                result = o1.getAccountNumber().compareTo(o2.getAccountNumber());
            }
            return result;
        }
    };
    /**
     * Sort Account number by lenght in reverse order.
     */
    protected static Comparator<String> reverseAccountLengthComparator = new Comparator<String>() {
        @Override
        public int compare(String o1, String o2) {
            int result = o2.length() - o1.length();
            if (result == 0) {
                // same length, compare accountNumber
                result = o2.compareTo(o1);
            }
            return result;
        }
    };

    public AccountViewHandler(AccountView view) {
        this.view = view;
        // Gets factory service        
        LimaServiceFactory.addServiceListener(ImportService.class, this);
        accountService = LimaServiceFactory.getService(AccountService.class);
    }

    /**
     * Init initialized view by loading account data from service.
     */
    public void init() {
        JXTreeTable table = view.getAccountsTreeTable();

        //To block reaction of the dual key 'ctrl+a' (Selection of all lines)
        InputMap inputMap = view.getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
        ActionMap actionMap = view.getActionMap();
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_A, InputEvent.CTRL_MASK), "none");

        // add action on Ctrl + N
        String binding = "new-account";
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_N, KeyEvent.CTRL_DOWN_MASK), binding);
        actionMap.put(binding, new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                addAccount();
            }
        });

        // add action on Delete
        binding = "remove-account";
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_DELETE, 0), binding);
        actionMap.put(binding, new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                removeAccount();
            }
        });

        // add action on Ctrl + M
        binding = "modify-account";
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_M, KeyEvent.CTRL_DOWN_MASK), binding);
        actionMap.put(binding, new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                updateAccount();
            }
        });

        table.addMouseListener(new MouseAdapter() {
            @Override
            public void mousePressed(MouseEvent e) {
                JXTreeTable source = (JXTreeTable) e.getSource();
                if (source.rowAtPoint(e.getPoint()) == -1) {
                    source.clearSelection();
                }
            }

            @Override
            public void mouseClicked(MouseEvent e) {
                JXTreeTable source = (JXTreeTable) e.getSource();
                if (source.rowAtPoint(e.getPoint()) >= 0 && e.getClickCount() == 2 ) {
                    updateAccount();
                }
            }
        });
        
        loadAllAccounts();
    }

    /**
     * Load all accounts from service and display it into tree table.
     */
    protected void loadAllAccounts() {
        // default data load
        List<Account> accounts = accountService.getAllAccounts();
        Collections.sort(accounts, accountLengthComparator);
        if (log.isDebugEnabled()) {
            log.debug(String.format("Loaded %d accounts from service", accounts.size()));
        }

        // render in tree node hierarchy for DefaultTreeTableModel
        SortedMap<String, DefaultMutableTreeTableNode> nodeCache = new TreeMap<String, DefaultMutableTreeTableNode>(reverseAccountLengthComparator);
        DefaultMutableTreeTableNode root = new DefaultMutableTreeTableNode(null);
        for (Account account : accounts) {
            // find parent
            DefaultMutableTreeTableNode parentNode = root;
            Iterator<Map.Entry<String, DefaultMutableTreeTableNode>> itNodes = nodeCache.entrySet().iterator();
            while (itNodes.hasNext()) {
                Map.Entry<String, DefaultMutableTreeTableNode> entry = itNodes.next();
                String accountNumber = entry.getKey();
                if (account.getAccountNumber().startsWith(accountNumber)) {
                    parentNode = entry.getValue();
                    break;
                }
            }
            
            // make current node
            DefaultMutableTreeTableNode node = new DefaultMutableTreeTableNode(account);
            parentNode.add(node);

            nodeCache.put(account.getAccountNumber(), node);
        }
        
        // refreshing tree's model
        DefaultTreeTableModel model = new AccountTreeTableModel(root);
        model.setColumnIdentifiers(Arrays.asList(t("lima.table.number"), t("lima.table.label")));
        JXTreeTable table = view.getAccountsTreeTable();
        table.setTreeTableModel(model);
    }

    /**
     * Display add account view
     */
    public void addAccount() {
        final AccountForm accountForm = new AccountForm(view);

        InputMap inputMap = accountForm.getRootPane().getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
        ActionMap actionMap = accountForm.getRootPane().getActionMap();
        String binding = "dispose";
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0), binding);
        actionMap.put(binding, new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                accountForm.dispose();
            }
        });

        accountForm.setLocationRelativeTo(view);
        accountForm.setVisible(true);
    }

    /**
     * Add new account with account form.
     *
     * @param dialog the account form
     */
    public void addAccount(AccountForm dialog) {

        try {
            Account newAccount = dialog.getAccount();
            newAccount = accountService.createAccount(newAccount);

            // update tree
            JXTreeTable treeTable = view.getAccountsTreeTable();
            DefaultTreeTableModel model = (DefaultTreeTableModel)treeTable.getTreeTableModel();
            DefaultMutableTreeTableNode node = (DefaultMutableTreeTableNode)findParentNode(model.getRoot(), newAccount.getAccountNumber());
            DefaultMutableTreeTableNode newNode = new DefaultMutableTreeTableNode(newAccount);

            List<MutableTreeTableNode> nodesToMove = findSubNodes(node, newAccount.getAccountNumber());
            for (MutableTreeTableNode nodeToMove : nodesToMove) {
                model.removeNodeFromParent((MutableTreeTableNode)nodeToMove);
                newNode.add(nodeToMove);
            }

            model.insertNodeInto(newNode, node, node.getChildCount());
            treeTable.expandPath(new TreePath(model.getPathToRoot(node)));
        } finally {
            dialog.dispose();
        }
    }

    /**
     * Find potential parent node for account number.
     * 
     * @param currentNode node 
     * @param accountNumber node label to search parent
     * @return found parent (can't be null)
     */
    protected TreeTableNode findParentNode(TreeTableNode currentNode,
            String accountNumber) {

        TreeTableNode result = null;
        Account account = (Account)currentNode.getUserObject();

        if (account == null || accountNumber.startsWith(account.getAccountNumber())) {
            for (int childIndex = 0; childIndex < currentNode.getChildCount() && result == null; childIndex++) {
                TreeTableNode child = currentNode.getChildAt(childIndex);
                result = findParentNode(child, accountNumber);
            }

            if (result == null) {
                result = currentNode;
            }
        }

        return result;
    }

    /**
     * Find all subnodes in currentNode with account label starting with accountNumber.
     * 
     * @param currentNode currentNode to search into
     * @param accountNumber accountNumber number to search
     * @return node list
     */
    protected List<MutableTreeTableNode> findSubNodes(TreeTableNode currentNode, String accountNumber) {
        List<MutableTreeTableNode> nodes = new ArrayList<MutableTreeTableNode>();
        for (int childIndex = 0; childIndex < currentNode.getChildCount(); childIndex++) {
            MutableTreeTableNode child = (MutableTreeTableNode)currentNode.getChildAt(childIndex);
            Account account = (Account)child.getUserObject();
            if (account.getAccountNumber().startsWith(accountNumber)) {
                nodes.add(child);
            }
        }
        return nodes;
    }

    /**
     * Open update account (or subledger) form with selected account
     * from the tree.
     */
    public void updateAccount() {
        JXTreeTable treeTable = view.getAccountsTreeTable();

        // get selected account
        int selectedRow = treeTable.getSelectedRow();
        TreePath treePath = treeTable.getPathForRow(selectedRow);
        TreeTableNode lastPathComponent = (TreeTableNode) treePath.getLastPathComponent();
        Account selectedAccount = (Account)lastPathComponent.getUserObject();

        // display edit form
        final UpdateAccountForm accountForm = new UpdateAccountForm(view);

        InputMap inputMap = accountForm.getRootPane().getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
        ActionMap actionMap = accountForm.getRootPane().getActionMap();
        String binding = "dispose";
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0), binding);
        actionMap.put(binding, new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                accountForm.dispose();
            }
        });


        accountForm.setAccount(selectedAccount);
        accountForm.setLocationRelativeTo(view);
        accountForm.setVisible(true);
    }

    /**
     * Perform update account to service.
     * 
     * @param dialog dialog containing account
     */
    public void updateAccount(UpdateAccountForm dialog) {

        try {
            Account account = dialog.getAccount();
            account = accountService.updateAccount(account);
    
            // update tree
            JXTreeTable treeTable = view.getAccountsTreeTable();
            DefaultTreeTableModel model = (DefaultTreeTableModel)treeTable.getTreeTableModel();
            int selectedRow = treeTable.getSelectedRow();
            TreePath treePath = treeTable.getPathForRow(selectedRow);
            //TreeTableNode lastPathComponent = (TreeTableNode) treePath.getLastPathComponent();
            //lastPathComponent.setUserObject(account);
            model.valueForPathChanged(treePath, account);

        } finally {
            // close dialog
            dialog.dispose();
        }
    }

    /**
     * Ask for user to remove for selected account, and remove it if confirmed.
     */
    public void removeAccount() {

        // maybe this code can be factorised
        JXTreeTable treeTable = view.getAccountsTreeTable();
        int selectedRow = treeTable.getSelectedRow();
        TreePath treePath = treeTable.getPathForRow(selectedRow);
        MutableTreeTableNode lastNode = (MutableTreeTableNode)treePath.getLastPathComponent();
        Account account = (Account)lastNode.getUserObject();

        int response = JOptionPane.showConfirmDialog(view,
             t("lima.ui.account.removeaccountconfirm", account.getAccountNumber()),
             t("lima.ui.account.removeaccounttitle"),
             JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE);

        if (response == JOptionPane.YES_OPTION) {
            accountService.removeAccount(account);

            // add all sub accounts to parent
            DefaultTreeTableModel model = (DefaultTreeTableModel)treeTable.getTreeTableModel();
            MutableTreeTableNode parent = (MutableTreeTableNode)lastNode.getParent();
            for (int childIndex = lastNode.getChildCount() -1 ; childIndex >= 0 ; childIndex--) {
                MutableTreeTableNode child = (MutableTreeTableNode)lastNode.getChildAt(childIndex);
                model.insertNodeInto(child, parent, parent.getChildCount());
            }

            // remove node
            model.removeNodeFromParent(lastNode);
        }
    }

    public void importAccountsChart() {

        final AccountImportForm form = new AccountImportForm(view);

        InputMap inputMap = form.getRootPane().getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
        ActionMap actionMap = form.getRootPane().getActionMap();
        String binding = "dispose";
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0), binding);
        actionMap.put(binding, new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                form.performCancel();
            }
        });

        form.setLocationRelativeTo(view);
        form.setVisible(true);

        Object value = form.getButtonGroup().getSelectedValue();
        // if action confirmed
        if (value != null) {
            ImportExport importExport = new ImportExport(view);
            AccountsChartEnum defaultAccountsChartEnum =
                    (AccountsChartEnum) value;
            //Import accounts chart
            switch (defaultAccountsChartEnum) {
                case IMPORTEBP:
                    importExport.importExport(ImportExportEnum.EBP_ACCOUNTCHARTS_IMPORT,
                                              defaultAccountsChartEnum.getFilePath(), false);
                    break;

                default:
                    importExport.importExport(ImportExportEnum.CSV_ACCOUNTCHARTS_IMPORT,
                                              defaultAccountsChartEnum.getFilePath(), false);
                    break;
            }
        }
    }

    /**
     * Called when import methods are called on services.
     */
    @Override
    public void notifyMethod(String serviceName, String methodName) {

        if (methodName.contains("importAccounts") ||
                methodName.contains("importAll") ||
                methodName.contains("importAsCSV")) {
            
            if (log.isInfoEnabled()) {
                log.info(String.format("Service notification %s, reloading accounts", methodName));
            }

            loadAllAccounts();
        } else {
            if (log.isDebugEnabled()) {
                log.debug("Ignoring service notification " + serviceName + ":" + methodName);
            }
        }
    }
}
