/*
 * #%L
 * Lima Swing
 * 
 * $Id: ErrorHelper.java 3373 2012-04-17 16:00:24Z echatellier $
 * $HeadURL: https://svn.chorem.org/lima/tags/lima-0.7.1/lima-swing/src/main/java/org/chorem/lima/util/ErrorHelper.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package org.chorem.lima.util;

import static org.nuiton.i18n.I18n._;

import java.awt.Component;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import javax.swing.JOptionPane;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.mail.EmailException;
import org.apache.commons.mail.MultiPartEmail;
import org.chorem.lima.LimaConfig;
import org.jdesktop.swingx.JXErrorPane;
import org.jdesktop.swingx.error.ErrorInfo;
import org.jdesktop.swingx.error.ErrorReporter;

/**
 * Error helper.
 * <p/>
 * Used to easily switch real exception interface. Currently used : swingx.
 *
 * @author chatellier
 * @version $Revision: 3373 $
 *          <p/>
 *          Last update : $Date: 2012-04-17 18:00:24 +0200 (Tue, 17 Apr 2012) $
 *          By : $Author: echatellier $
 */
public class ErrorHelper implements ErrorReporter {

    /** Log. */
    private static final Log log = LogFactory.getLog(ErrorHelper.class);

    protected LimaConfig config;

    public ErrorHelper(LimaConfig config) {
        this.config = config;
    }

    /**
     * Display a user friendly error frame.
     *
     * @param parent  parent component
     * @param message message for user
     * @param cause   exception cause
     */
    public void showErrorDialog(Component parent, String message,
                                       Throwable cause) {
        JXErrorPane pane = new JXErrorPane();
        ErrorInfo info = new ErrorInfo(_("lima.common.error"),
                                       _("lima.error.errorpane.htmlmessage", message), null, null,
                                       cause, null, null);
        pane.setErrorInfo(info);
        pane.setErrorReporter(this);
        JXErrorPane.showDialog(parent, pane);
    }

    /**
     * Display a user friendly error frame.
     *
     * @param message message for user
     */
    public void showErrorDialog(String message) {
        showErrorDialog(message, null);
    }

    /**
     * Display a user friendly error frame.
     *
     * @param message message for user
     * @param cause   exception cause
     */
    public void showErrorDialog(String message, Throwable cause) {
        showErrorDialog(null, message, cause);
    }

    /*
     * @see org.jdesktop.swingx.error.ErrorReporter#reportError(org.jdesktop.swingx.error.ErrorInfo)
     */
    @Override
    public void reportError(ErrorInfo errorInfo) throws NullPointerException {

        try {
            String emailTo = config.getSupportEmail();

            MultiPartEmail email = new MultiPartEmail();
            // smtp
            email.setHostName("smtp");
            // to
            email.addTo(emailTo, "Support");
            // from
            email.setFrom("no-reply@nuiton.org", "Lima");
            // subject
            email.setSubject("Project error notification : Lima");

            // message description
            StringBuffer message = new StringBuffer();
            message.append(formatMessage("Project", "Lima " + config.getVersion()));
            message.append(formatMessage("Date", new Date().toString()));
            message.append(formatMessage("Title", errorInfo.getTitle()));
            message.append(formatMessage("Description", errorInfo.getBasicErrorMessage().replaceAll("<[^>]+>", "")));

            // message configuration
            message.append(formatMessage("Configuration", null));
            List<String> propertiesNames = new ArrayList<String>(config.getOptions().stringPropertyNames());
            Collections.sort(propertiesNames);
            for (String propertyName : propertiesNames) {
                // security, don't send string containing password :
                if (!propertyName.contains("pass")) {
                    message.append("\t" + propertyName + " : " + config.getOptions().getProperty(propertyName) + "\n");
                }
            }

            // message exception
            StringWriter out = new StringWriter();
            PrintWriter writer = new PrintWriter(out);
            errorInfo.getErrorException().printStackTrace(writer);
            message.append(formatMessage("Exception", out.toString()));

            // send mail
            email.send();

            JOptionPane.showMessageDialog(null, "A report message has been sent to " + emailTo);
        } catch (EmailException ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't send report email", ex);
            }
        }
    }

    protected String formatMessage(String category, String content) {
        String formatted = category + " :\n";
        if (StringUtils.isNotEmpty(content)) {
            formatted += "\t" + content + "\n";
        }
        return formatted;
    }
}
