package fr.onema.sispea.struts.referential.action;

/*
 * #%L
 * SISPEA web application
 * %%
 * Copyright (C) 2014 - 2015 ONEMA
 * %%
 * ONEMA - Tous droits réservés
 * #L%
 */


import fr.onema.sispea.SispeaException;
import fr.onema.sispea.service.data.DataRatioDto;
import fr.onema.sispea.service.data.ExerciseDto;
import fr.onema.sispea.service.data.SPEALotDto;
import fr.onema.sispea.service.data.SPEALotStatus;
import fr.onema.sispea.service.exchange.ExportParameters;
import fr.onema.sispea.service.exchange.FileExchangeService;
import fr.onema.sispea.service.exchange.FileType;
import fr.onema.sispea.service.exchange.rpqs.RpqsConstants;
import fr.onema.sispea.service.referential.DDTDto;
import fr.onema.sispea.service.referential.DepartmentDto;
import fr.onema.sispea.service.referential.EquipmentDto;
import fr.onema.sispea.service.referential.MunicipalityDto;
import fr.onema.sispea.service.referential.OrganismDto;
import fr.onema.sispea.service.referential.OrganismType;
import fr.onema.sispea.service.referential.SpeaMemberDto;
import fr.onema.sispea.service.referential.TerritoryDto;
import fr.onema.sispea.service.user.MandateDto;
import fr.onema.sispea.service.user.Right;
import fr.onema.sispea.service.user.UserDto;
import fr.onema.sispea.struts.SispeaSession;
import fr.onema.sispea.struts.common.bean.MenuBean;
import fr.onema.sispea.struts.common.menu.MenuConstants;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * This action handles the SPEA visualisation and modification.
 * It allows the end user to visualize a service and to edit its mandated users list.
 *
 * @author CS
 */
public class ShowSPEAPresentationAction extends AbstractSPEAPresentationAction {

    private static final long serialVersionUID = 1L;

    private static final Logger logger = Logger.getLogger(ShowSPEAPresentationAction.class);

    @Autowired
    protected transient FileExchangeService fileExchangeService;

    /**
     * Input parameter.
     * Set to true to generate the RPQS for the SPEA.
     */
    protected boolean rpqs;

    /**
     * Input parameter.
     * Set to true to modify the SPEA comment and link.
     */
    protected boolean edit;

    /**
     * Used to store the last RPQS generation date for the SPEA
     */
    protected Calendar lastRPQSDate;

    /**
     * Store for each spea the lot status associated.
     */
    protected Map<Integer, SPEALotStatus> speaLotStatusMap = new HashMap<>();

    /**
     * Store for each spea the ratio associated.
     */
    protected Map<Integer, DataRatioDto> speaDataRatioMap = new HashMap<>();

    /**
     * Store for spea member his ddt or departement name.
     */
    protected Map<SpeaMemberDto, String> memberDepartementOrDdtName = new HashMap<>();

    /**
     * Store for spea municipality his name with his inseeCode.
     */
    protected Map<String, String> municipalityLabelMaps = new HashMap<>();

    /**
     * true if current user can view this territory's users
     */
    protected boolean canViewUsers;

    /**
     * can view ?
     */
    protected boolean canView;

    /**
     * can check ?
     */
    protected boolean canCheck;

    /**
     * can edit ?
     */
    protected boolean canEdit;

    /**
     * can view mandate ?
     */
    protected boolean canViewMandate;

    /**
     * can mandate ?
     */
    protected boolean canMandate;

    /**
     * Territory associated to current organism displayed.
     */
    protected TerritoryDto territory;

    protected int associatedTerritoryId;

    /**
     * Set to true if the user has view access to the SPEA lot for the requested exercise. false otherwise.<br />
     * This boolean is used by the {@link #buildMenu()} method.
     */
    private boolean hasViewAccess;

    @Override
    protected String doExecute() throws Exception {

        // init result
        String result = SUCCESS;
        clearMessages();

        // log
        if (logger.isDebugEnabled()) {
            logger.debug("show spea presentation");
        }

//        // set canedit to null
//        setCanEdit(null);

        // menu
        openMenu(MenuConstants.MENU_KEY_PRESENTATION);

        // check params
        initPagingSizes();
        checkExercise();
        checkId();

        // read spea
        spea = referentialService.readSPEA(orgId, exercise, false);

        // check spea
        if (spea == null) {
            throw new SispeaException("fr.onema.sispea.jsp.fixRef.showSPEAPresentation.error.emptySpea");
        }

        territory = territoryService.readTerritory(spea.getId(), OrganismType.SPEA);
        associatedTerritoryId = getAssociatedTerritoryId(territory);

        // Compute the user's SPEA lot view access right
        hasViewAccess = hasViewAccess(spea, exercise);
        spea.setHasViewAccess(hasViewAccess);

        // read DDT
        ddt = referentialService.readAssociatedDDT(spea.getId(), exercise);

        // read mandates
        spea.setMandates(userService.readSpeaMandates(spea.getId()));

        // Load associated users.
        loadUsers(territory);

        canView = checkUserRight(Right.View, territory, exercise);
        canCheck = checkUserRight(Right.Check, territory, exercise);
        canEdit = checkUserRight(Right.Edit, territory, exercise);
        canViewUsers = computeCanViewUsers(getCurrentUser(), territory, exercise);
        canMandate = checkUserRight(Right.Mandate, territory, exercise);
        canViewMandate = checkUserRight(Right.ViewMandates, territory, exercise);

        // paging
        manageMemberPaging();
        manageEquipmentPaging();
        manageMandatePaging();

        // change menu ?
        openMenu(getProperMenuKey(spea, territory));

        // Do not change the menu here !
        // The execute method of the SispeaActionSupport will override it.
        // Override the buildMenu method to set the visibility of the menus

        // Do we need to generate the RPQS
        if (rpqs) {

            // Generate the RPQS
            generateRpqs(getCurrentUser(), spea, exercise);

            // Set the action message
            addActionMessage(getText("fr.onema.sispea.jsp.fixRef.showSPEAPresentation.rpqs.generating"));
        }

        // Load last RPQS date
        loadSpeaRPQSDate();

        Integer speaId = spea.getId();

        if (logger.isDebugEnabled()) {
            logger.debug("Loading data for spea " + speaId);
        }

        Integer exerciseId = getExercise().getId();

        SPEALotStatus speaLotStatus = speaLotService.readSpeaLotStatus(speaId, exerciseId);
        speaLotStatusMap.put(speaId, speaLotStatus);

        DataRatioDto displayRatio = speaLotService.getDisplayRatio(speaId, exerciseId, spea.getHasViewAccess());
        speaDataRatioMap.put(speaId, displayRatio);

        for (SpeaMemberDto member : getSpea().getMembers()) {

            OrganismDto organism = member.getOrganism();

            String name;
            Integer organismTypeId = (int) organism.getType().getDbId();
            if (organismTypeId == 4) {
                DDTDto ddt = readDdt(organism.getId(), organismTypeId);
                name = ddt.getName();

            } else {

                DepartmentDto department = readDepartment(organism.getId(), organismTypeId);
                name = department.getName();

            }

            memberDepartementOrDdtName.put(member, name);

        }

        for (EquipmentDto equipment : getSpea().getEquipments()) {

            String inseeCode = equipment.getMunicipalityINSEE();

            MunicipalityDto municipality = referentialService.findMunicipalityByInseeCode(inseeCode);

            String municipalityName;
            if (municipality == null || municipality.getName() == null) {
                municipalityName = inseeCode;
            } else {
                municipalityName = municipality.getName() + " (" + inseeCode + ")";
            }

            municipalityLabelMaps.put(inseeCode, municipalityName);

        }

        return result;
    }

    /**
     * Initializes the menu.
     * Hides some menus if the user has not view access to the SPEA data.
     *
     * @throws SispeaException
     */
    @Override
    public void buildMenu() throws SispeaException {
        super.buildMenu();

        // Hide the 'Indicator view access' and 'All data access' menu items if the user has not view access to the SPEA lot
        if (!hasViewAccess) {
            SispeaSession session = getSispeaSession();
            MenuBean rootMenu = session.getMenu();
            if (rootMenu != null) {
                MenuBean speaMenu = rootMenu.getMenu(MenuConstants.MENU_KEY_SEE_SPEA);

                if (speaMenu != null) {
                    speaMenu.setVisible(MenuConstants.MENU_KEY_SPEA_DATA, false);
                    speaMenu.setVisible(MenuConstants.MENU_KEY_SPEA_INDICATORS, false);
                }
            }
        }
    }

    /**
     * This method generates the RPQS for the SPEA.
     *
     * @param pCurrentUser
     * @param pOrganism
     * @param pExercise
     * @throws SispeaException
     */
    protected void generateRpqs(UserDto pCurrentUser, OrganismDto pOrganism, ExerciseDto pExercise) throws SispeaException {

        String territoryName = null;
        try {
            // Prepare the export parameters.
            //
            // The export is run with the SPEA territory
            TerritoryDto territory = territoryService.readTerritory(pOrganism);
            territoryName = territory.getName();

            // For huge territories, cut it in smaller ones
            List<TerritoryDto> territories = territoryService.buildExtractTerritoryList(territory, pExercise.getId(), pCurrentUser.getId());

            // View right
            //Boolean hasViewRight = territoryService.hasViewRight(pCurrentUser, territory, pExercise);
            Boolean hasViewRight = userService.checkUserRight(pCurrentUser, Right.View, territory, pExercise);

            ExportParameters parameters = new ExportParameters();
            parameters.setUser(pCurrentUser);
            parameters.setTerritories(territories);
            parameters.setLang(getLocale().getLanguage());
            parameters.setHasViewRight(hasViewRight);
            parameters.setFileType(FileType.RPQS);
            parameters.setWithData(false);
            parameters.setWithReferential(false);
            parameters.setExtractTerritoryName(territoryName);
            parameters.setFileNamePrefix(RpqsConstants.RPQS_FILENAME_PREFIX);
            parameters.setFileNameDateFormat("yyyyMMdd_HHmmss");
            parameters.setExercise(pExercise);
            // Do not set any competences. They are not needed to generate the RPQS.
            parameters.setCompetences(null);

            // Generate the export
            fileExchangeService.download(parameters);

        } catch (Exception e) {
            String msg = "Error while generating RPQS for the territory '" + territoryName + "'";
            logger.error(msg, e);
            throw new SispeaException(getText("fr.onema.sispea.jsp.fixRef.showSPEAPresentation.rpqs.error", pOrganism.getName()));
        }
    }

    /**
     * loads spea rpqs date if exists
     *
     * @throws SispeaException
     */
    protected void loadSpeaRPQSDate() throws SispeaException {
        Boolean hasViewRight = userService.checkUserRight(getCurrentUser(), Right.View, spea, exercise);
        SPEALotDto lot = speaLotService.readSpeaLot(chosenExerciseId, spea.getId(), hasViewRight);
        // check lot
        if (lot == null) {
            lastRPQSDate = null;
        } else {
            lastRPQSDate = lot.getLastRPQSDate();
        }
    }

    public boolean isRpqs() {
        return rpqs;
    }

    public void setRpqs(boolean pRpqs) {
        rpqs = pRpqs;
    }

    public boolean isEdit() {
        return edit;
    }

    public void setEdit(boolean pEdit) {
        edit = pEdit;
    }

    public Calendar getLastRPQSDate() {
        return lastRPQSDate;
    }

    public void setLastRPQSDate(Calendar pLastRPQSDate) {
        lastRPQSDate = pLastRPQSDate;
    }

    public SPEALotStatus getSpeaLotStatus(Integer speaId) {
        return speaLotStatusMap.get(speaId);
    }

    public DataRatioDto getSpeaDataRatio(Integer speaId) {
        return speaDataRatioMap.get(speaId);
    }

    public String getMemberDepartementOrDdtName(SpeaMemberDto member) {
        return memberDepartementOrDdtName.get(member);
    }

    public boolean isCanViewUsers() {
        return canViewUsers;
    }

    public void setCanViewUsers(boolean canViewUsers) {
        this.canViewUsers = canViewUsers;
    }

    public boolean isCanView() {
        return canView;
    }

    public void setCanView(boolean canView) {
        this.canView = canView;
    }

    public boolean isCanCheck() {
        return canCheck;
    }

    public void setCanCheck(boolean canCheck) {
        this.canCheck = canCheck;
    }

    public boolean isCanEdit() {
        return canEdit;
    }

    public void setCanEdit(boolean canEdit) {
        this.canEdit = canEdit;
    }

    public boolean isCanViewMandate() {
        return canViewMandate;
    }

    public void setCanViewMandate(boolean canViewMandate) {
        this.canViewMandate = canViewMandate;
    }

    public TerritoryDto getTerritory() {
        return territory;
    }

    public int getAssociatedTerritoryId() {
        return associatedTerritoryId;
    }

    public void setAssociatedTerritoryId(int associatedTerritoryId) {
        this.associatedTerritoryId = associatedTerritoryId;
    }

    /**
     * @param pUserLogin
     * @return true if user's login is in mandated list
     */
    public boolean isUserMandated(String pUserLogin) {
        // init result
        Boolean lRes = false;

        // check param
        // get spea mandates
        if (pUserLogin != null
            && getSpea() != null
            && getSpea().getMandates() != null) {
            // loop on mandates
            for (MandateDto lMandate : getSpea().getMandates()) {
                if (pUserLogin.equals(lMandate.getMandated().getLogin())) {
                    lRes = true;
                    break;
                }
            }
        }

        // result
        return lRes;
    }

    public String getEquipmentMunicipality(String pInseeCode) {
        return municipalityLabelMaps.get(pInseeCode);
    }

    public boolean isCanMandate() {
        return canMandate;
    }

    public void setCanMandate(boolean canMandate) {
        this.canMandate = canMandate;
    }

}
