package fr.onema.sispea.struts.referential.action;

/*
 * #%L
 * SISPEA web application
 * %%
 * Copyright (C) 2014 - 2015 ONEMA
 * %%
 * ONEMA - Tous droits réservés
 * #L%
 */


import fr.onema.sispea.SispeaException;
import fr.onema.sispea.service.data.IndicatorDto;
import fr.onema.sispea.service.data.IndicatorService;
import fr.onema.sispea.service.data.MissionsCombining;
import fr.onema.sispea.service.data.SPEALotDto;
import fr.onema.sispea.service.data.SPEALotService;
import fr.onema.sispea.service.data.SpeaComparatorItemDto;
import fr.onema.sispea.service.data.VPDto;
import fr.onema.sispea.service.referential.CompetenceDto;
import fr.onema.sispea.service.referential.MissionDto;
import fr.onema.sispea.service.referential.OrganismDto;
import fr.onema.sispea.service.referential.OrganismType;
import fr.onema.sispea.service.referential.ReferentialService;
import fr.onema.sispea.service.referential.SPEADto;
import fr.onema.sispea.service.referential.TerritoryDto;
import fr.onema.sispea.service.referential.TerritoryType;
import fr.onema.sispea.service.user.Right;
import fr.onema.sispea.struts.search.bean.BackType;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

/**
 * This abstract action is the basis for all actions used to manage comparator.
 *
 * @author CS
 */
public abstract class AbstractComparatorAction extends AbstractExerciseListAction {

    protected static final String COMPARATOR_TYPE_ALL = "ALL";

    protected static final String COMPARATOR_TYPE_IND = "IND";

    private static final long serialVersionUID = 1L;

    /**
     * Default hidden value
     */
    private static final String defaultHiddenValue = "####";

    @Autowired
    protected transient IndicatorService indicatorService;

    @Autowired
    protected transient ReferentialService referentialService;

    @Autowired
    protected transient SPEALotService speaLotService;

    /**
     * the pertient indicator list
     */
    protected List<IndicatorDto> ipList;

    /**
     * the descriptive indicator list
     */
    protected List<IndicatorDto> idList;

    /**
     * the variables list
     */
    protected List<VPDto> pvsList = new ArrayList<>();

    /**
     * the ratio list
     */
    protected List<VPDto> ratiosList = new ArrayList<>();

    /**
     * the competence id to display
     */
    protected Integer competenceId;

    /**
     * the competence to display
     */
    protected CompetenceDto competence;

    /**
     * the type of comparison (IND or ALL)
     */
    protected String comparatorType = COMPARATOR_TYPE_ALL;

    /**
     * the competence list
     */
    protected List<CompetenceDto> competenceList;

    /**
     * the speas lots of the comparator table
     */
    protected List<SpeaComparatorItemDto> comparatorSpeasLots = new ArrayList<>();

    /**
     * the available speas
     */
    protected List<SPEADto> availableSpeas;

    /**
     * the chosen spea Id
     */
    protected Integer speaId;

    /**
     * Selected territory from "Enrichir"
     */
    protected Integer selectionId;

    /**
     * @throws SispeaException
     */
    protected void loadComparator() throws SispeaException {
        if (getCurrentUser() == null) {
            // message
            throw new SispeaException(getText("fr.onema.sispea.autentication.notAllowed"));
        }

        // set chosen exercice id
        if (chosenExerciseId == null && getSispeaSession().getChosenExerciseId() != null) {
            chosenExerciseId = getSispeaSession().getChosenExerciseId();
        }
        // check exercise
        checkExercise();

        // read competences
        competenceList = referentialService.readCompetences();
        Collections.sort(competenceList);

        // set chosen competence id
        if (competenceId == null && getSispeaSession().getCompetenceId() != null) {
            competenceId = getSispeaSession().getCompetenceId();
        } else if (competenceId != null && getSispeaSession().getCompetenceId() != null && !competenceId.equals(getSispeaSession().getCompetenceId())) {
            // user changed competence: reset
            addActionMessage(getText("fr.onema.sispea.referential.comparator.message.reset"));
        }

        // if still null -> drinking water
        if (competenceId == null) {
            competenceId = CompetenceDto.COMPETENCE_DRINKING_WATER_ID;
        }

        // set competence
        competence = null;
        for (Iterator<CompetenceDto> iterator = competenceList.iterator(); iterator.hasNext() && competence == null; ) {
            CompetenceDto lCompetence = iterator.next();
            if (lCompetence.getId().equals(competenceId)) {
                competence = lCompetence;
            }
        }
        if (competence == null) {
            throw new SispeaException(getText("fr.onema.sispea.referential.comparator.error.wrongCompetence", "", competenceId.toString()));
        }

        // read indicators
        idList = indicatorService.readDIs(competenceId);
        ipList = indicatorService.readPIs(competenceId);

        // Get the curent year
        Short year = null;
        try {
            year = exerciseService.readExercise(getChosenExerciseId()).getYear();
        } catch (SispeaException e) {
            //Do nothing
        }

        // VALIDITY : remove obsolete indicator
        Iterator it = ipList.iterator();
        while (it.hasNext()) {
            Boolean isValid = null;
            IndicatorDto indicator = (IndicatorDto) it.next();
            if (indicator != null && year != null) {
                isValid = indicatorService.isInValidityPeriod(indicator.getCode(), year);
            }
            if (isValid != null && !isValid) {
                it.remove();
            }
        }

        // read performance variables and ratios
        for (VPDto lPV : competence.getPvs()) {
            if (lPV.isRatio()) {
                ratiosList.add(lPV);
            } else {
                pvsList.add(lPV);
            }
        }

        // Sorting
        Collections.sort(ipList);
        Collections.sort(idList);
        Collections.sort(pvsList);
        Collections.sort(ratiosList);

        // build comparator spea data list
        buildComparatorSpeaDataList(getSispeaSession().getComparatorSpeasIds());

        // set chosen exercice id in session
        if (chosenExerciseId != null) {
            getSispeaSession().setChosenExerciseId(chosenExerciseId);
        }

        // session competence
        Integer lSessionCompetenceId = getSispeaSession().getCompetenceId();
        if (!competenceId.equals(lSessionCompetenceId)) {
            // change session competence id
            getSispeaSession().setCompetenceId(competenceId);
            // reset spea comparator list
            getSispeaSession().setComparatorSpeasIds(new ArrayList<Integer>());
        }

        // build available spea list
        // get user's selection list
        loadAvailableSpeas();
    }

    /**
     * loads available speas from user's selections
     *
     * @throws SispeaException
     */
    protected void loadAvailableSpeas() throws SispeaException {
        //init
        availableSpeas = new ArrayList<>();
        List<TerritoryDto> lSelectedTerritories;

        // read selections
        if (getCurrentUser() == null) {
            // the user is not authenticated
            // just get session selected territories
            lSelectedTerritories = getSispeaSession().getSelectedTerritories();

        } else {
            // the user is authenticated
            // read territories
            lSelectedTerritories = referentialService.readUserSelections(getCurrentUser());
        }

        // filter speas
        for (TerritoryDto lTerritory : lSelectedTerritories) {
            if (lTerritory.getType() == TerritoryType.SPEA) {
                // load spea
                OrganismDto lOrg = territoryService.getOrganismFromTerritoryId(lTerritory.getId());
                SPEADto lSpea = referentialService.readSPEA(lOrg.getId(), false);
                // filter on competence
                if (lSpea.getCompetence().getId().equals(competenceId)) {
                    // add to available list
                    if (!getSispeaSession().getComparatorSpeasIds().contains(lSpea.getId())) {
                        availableSpeas.add(lSpea);
                    }
                }
            }

        }

        // Sort list
        Collections.sort(availableSpeas);
    }


    /**
     * build spea data list to print
     *
     * @param pComparatorSpeasIds
     * @throws SispeaException
     */
    private void buildComparatorSpeaDataList(List<Integer> pComparatorSpeasIds) throws SispeaException {

        // init : get SPEALot List
        if (exercise.getId() != null && pComparatorSpeasIds != null) {

            // check user right
            for (Integer lSpeaId : pComparatorSpeasIds) {
                // load territory
                TerritoryDto lSpeaTerritory = territoryService.readTerritory(lSpeaId, OrganismType.SPEA);

                // load spea
                SPEADto lSpea = referentialService.readSPEA(lSpeaId, exercise, false);

                // check competence
                if (lSpea != null && lSpea.getCompetence().getId().equals(competenceId)) {
                    // check
                    Boolean lHasViewRight = checkUserRight(Right.View, lSpeaTerritory, exercise);

                    // load lot
                    SPEALotDto lLot = speaLotService.readSpeaLotForComparator(exercise.getId(), lSpeaId, lHasViewRight);

                    // new item
                    SpeaComparatorItemDto lItem = new SpeaComparatorItemDto(lSpea, lLot, lHasViewRight);

                    //a dd to list
                    comparatorSpeasLots.add(lItem);
                }
            }
        }
    }

    /**
     * Get the result of a comparison of a data on 1 or more services
     *
     * @param pDataCode
     * @param pSpeaId
     * @param pIpValue  If true indicator value returned, vp value else
     * @return
     */
    protected String getComparatorDataValue(String pDataCode, Integer pSpeaId, boolean pIpValue) {

        // init
        String lRes = null;
        boolean lFound = false;

        // search for lot
        for (Iterator<SpeaComparatorItemDto> iterator = comparatorSpeasLots.iterator(); iterator.hasNext() && !lFound; ) {
            SpeaComparatorItemDto lItem = iterator.next();
            SPEALotDto lSPEALot = lItem.getSpeaLot();
            SPEADto lSpea = lItem.getSpea();
            boolean lHasViwRight = lItem.isHasViewRight();

            // test lot id
            if (lSpea.getId().equals(pSpeaId)) {
                lFound = true;
                lRes = speaLotService.getComparatorDataValue(pDataCode, lSPEALot, lHasViwRight, getCurrentUser(), pIpValue);

            }
        }

        // result
        return lRes;
    }

    /**
     * Get the result of a comparison of an indicator on 1 or more services
     *
     * @param pIndicatorCode
     * @param pSpeaId
     * @return
     */
    public String getComparatorIndicatorValue(String pIndicatorCode, Integer pSpeaId) {
        return getComparatorDataValue(pIndicatorCode, pSpeaId, true);
    }

    /**
     * Get the result of a comparison of a VP on 1 or more services
     *
     * @param pVpCode
     * @param pSpeaId
     * @return
     */
    public String getComparatorVpValue(String pVpCode, Integer pSpeaId) {
        return getComparatorDataValue(pVpCode, pSpeaId, false);
    }


    // GETTERS AND SETTERS

    /**
     * get the list of competences
     */
    public Map<String, String> getCompetenceList() throws SispeaException {

        //init list
        TreeMap<String, String> lRes = new TreeMap<>();

        for (CompetenceDto lCompetence : competenceList) {
            lRes.put(lCompetence.getId().toString(), lCompetence.getName());
        }

        return lRes;
    }

    public void setCompetenceList(List<CompetenceDto> pCompetenceList) {
        competenceList = pCompetenceList;
    }

    /**
     * get the list of available Spea display
     *
     * @return
     * @throws SispeaException
     */
    public Map<String, String> getAvailableSpeasDisplayList() throws SispeaException {

        //init list
        Map<String, String> lRes = new LinkedHashMap<>();

        if (availableSpeas != null) {
            for (SPEADto lSPEA : availableSpeas) {
                String lSpeaName = lSPEA.getName();
                if (lSPEA.getCollectivity() != null) {
                    lSpeaName = lSPEA.getCollectivity().getName() + " - " + lSpeaName;
                }
                lRes.put(lSPEA.getId().toString(), lSpeaName);
            }
        }

        return lRes;
    }

    /**
     * get the list of available comparison setup types.
     *
     * @return
     * @throws SispeaException
     */
    public Map<String, String> getComparatorTypes() throws SispeaException {
        //init list
        HashMap<String, String> lRes = new HashMap<>();

        lRes.put(COMPARATOR_TYPE_IND, getText("fr.onema.sispea.referential.comparator.field.type.indicateurs"));
        lRes.put(COMPARATOR_TYPE_ALL, getText("fr.onema.sispea.referential.comparator.field.type.all"));

        return lRes;
    }

    /**
     * Return the list of missions of current competence
     *
     * @param pCompetenceId
     * @return
     * @throws SispeaException
     */
    public List<MissionsCombining> getMissionsList(int pCompetenceId) throws SispeaException {

        //init list
        List<MissionsCombining> lRes = new ArrayList<>();

        // build associations
        for (MissionsCombining lMissionsCombining : MissionsCombining.values()) {

            if ((pCompetenceId == CompetenceDto.ALL_COMPETENCES)
                || (lMissionsCombining.getCompetenceId() == pCompetenceId)) {
                lRes.add(lMissionsCombining);
            }
        }

        return lRes;
    }

    /**
     * Return the formated string of missions
     *
     * @param pSpeaId
     * @return
     * @throws SispeaException
     */
    public String getSpeaMissions(int pSpeaId) throws SispeaException {
        String lRes = "";
        boolean lFound = false;

        // search for lot
        for (Iterator<SpeaComparatorItemDto> iterator = comparatorSpeasLots.iterator(); iterator.hasNext() && !lFound; ) {
            SpeaComparatorItemDto lItem = iterator.next();
            SPEADto lSpea = lItem.getSpea();

            // test lot id
            if (lSpea.getId().equals(pSpeaId)) {
                if (lSpea.getMissions().values().size() == 3) {
                    lRes = getText("fr.onema.sispea.referential.comparator.field.missions.all");
                } else {
                    for (Iterator<MissionDto> iterator2 = lSpea.getMissions().values().iterator(); iterator2.hasNext(); ) {
                        MissionDto lMission = iterator2.next();
                        if (iterator2.hasNext()) {
                            lRes += lMission.getDescription() + ", ";
                        } else {
                            lRes += lMission.getDescription();
                        }
                    }
                }
            }
        }

        return lRes;
    }

    /**
     * Compute and return the number of columns in the table
     *
     * @return
     */
    public int getColumnsCount() {
        try {
            if (comparatorSpeasLots != null) {
                return comparatorSpeasLots.size() + 2;
            } else {
                return 2;
            }
        } catch (Exception e) {
            return 2;
        }
    }

    public int getSpeaTerritoryTypeId() {
        return TerritoryType.SPEA.getId();
    }

    public Integer getCompetenceId() {
        return competenceId;
    }

    public void setCompetenceId(Integer pCompetenceId) {
        competenceId = pCompetenceId;
    }

    public Integer getSpeaId() {
        return speaId;
    }

    public void setSpeaId(Integer pSpeaId) {
        speaId = pSpeaId;
    }


    public List<SPEADto> getAvailableSpeas() {
        return availableSpeas;
    }

    public void setAvailableSpeas(List<SPEADto> pAvailableSpeas) {
        availableSpeas = pAvailableSpeas;
    }

    public List<SpeaComparatorItemDto> getComparatorSpeasLots() {
        return comparatorSpeasLots;
    }

    public void setComparatorSpeasLots(List<SpeaComparatorItemDto> pComparatorSpeasLots) {
        comparatorSpeasLots = pComparatorSpeasLots;
    }

    public String getDefaultHiddenValue() {
        return defaultHiddenValue;
    }

    public Integer getBackPageId() {
        return BackType.comparator.getId();
    }

    public Integer getSelectionId() {
        return selectionId;
    }

    public void setSelectionId(Integer pSelectionId) {
        selectionId = pSelectionId;
    }

    public String getComparatorType() {
        return comparatorType;
    }

    public void setComparatorType(String pComparatorType) {
        comparatorType = pComparatorType;
    }

    public List<VPDto> getPvsList() {
        return pvsList;
    }

    public void setPvsList(List<VPDto> pPvsList) {
        pvsList = pPvsList;
    }

    public List<IndicatorDto> getIpList() {
        return ipList;
    }

    public void setIpList(List<IndicatorDto> pIpList) {
        ipList = pIpList;
    }

    public List<IndicatorDto> getIdList() {
        return idList;
    }

    public void setIdList(List<IndicatorDto> pIdList) {
        idList = pIdList;
    }

    public List<VPDto> getRatiosList() {
        return ratiosList;
    }

    public void setRatiosList(List<VPDto> pRatiosList) {
        ratiosList = pRatiosList;
    }

    public CompetenceDto getCompetence() {
        return competence;
    }

    public void setCompetence(CompetenceDto pCompetence) {
        competence = pCompetence;
    }

}
