package fr.onema.sispea.struts.common.action;

/*
 * #%L
 * SISPEA web application
 * %%
 * Copyright (C) 2014 - 2015 ONEMA
 * %%
 * ONEMA - Tous droits réservés
 * #L%
 */


import fr.onema.sispea.SispeaException;
import fr.onema.sispea.service.common.LoginService;
import fr.onema.sispea.service.user.UserDto;
import fr.onema.sispea.struts.AbstractSispeaAction;
import fr.onema.sispea.struts.common.menu.MenuConstants;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.Calendar;

/**
 * This action handles the user login validity test.
 * It tests if the user name/password is valid.
 *
 * @author CS
 */
public class LoginAction extends AbstractSispeaAction {

    private static final long serialVersionUID = 1L;

    private static final Logger logger = Logger.getLogger(LoginAction.class);

    @Autowired
    protected transient LoginService loginService;

    /**
     * The user login.
     */
    protected String login;

    /**
     * The user bean.
     */
    protected UserDto user;

    /**
     * The user's password.
     */
    protected String password;

    /**
     * The initial url asked for by the user before redirected to the
     * login page. This attribute should only be used in the login process.
     */
    protected String urlGoingTo;

    @Override
    protected String doExecute() throws Exception {

        // If we get here, the login and the password has been validated

        // Do we have the user in the db / ldap
        user = loginService.checkUser(login, password);

        // Following line can be activated if the LDAP server is unavailable.
        // It returns a corked user having all rights and with a super virtual reference territory.
//        user = getUserCork();

        // if user is null -> not authenticated
        String nextPage = SUCCESS;
        if (user != null) {
            if (logger.isInfoEnabled()) {
                logger.info("User " + user.getLogin() + " successfully authenticated (type " + user.getType().getId() + ")");
            }

            // check if the user is suspended
            if (user.isSuspended()) {
                throw new SispeaException("fr.onema.sispea.service.login.errors.suspended");
            }

            // check the validity end date if not permanent
            Calendar lNow = Calendar.getInstance();
            if (!user.isPermanent()) {
                // check validity
                if (user.getValidityEndDate() == null) {
                    throw new SispeaException("fr.onema.sispea.service.login.errors.notValidAnymore");
                } else if (user.getValidityEndDate().compareTo(lNow) <= 0) {
                    // user not allowed
                    throw new SispeaException("fr.onema.sispea.service.login.errors.notValidAnymore");
                }
            }

            // Set the user in the session
            getSispeaSession().setUser(user);

            // save last connection date
            user.setLastConnectionDate(lNow);
            loginService.updateLastConnectionDate(user);

            // Do we have a requested page before starting the login process?
            if (getSispeaSession().getUrlGoingTo() == null) {
                // If a user of type collectivity logins in and the user has not specified
                // any specific page, redirect to
                // - The collectivity page if the collectivity has more than one SPEA
                // - The SPEA data page for the current exercise if the SPEA has only one SPEA
                if (user.isCollectivity()) {
                    if (logger.isDebugEnabled()) {
                        logger.debug("Redirecting collectivity user");
                    }
                    nextPage = "collectivityUserRedirect";
                }
            } else {

                urlGoingTo = getSispeaSession().getUrlGoingTo();
                getSispeaSession().setUrlGoingTo(null);
                nextPage = "redirect";

                if (logger.isDebugEnabled()) {
                    logger.debug("Redirecting to " + urlGoingTo);
                }
            }
        } else {
            if (logger.isDebugEnabled()) {
                logger.debug("Failed to authenticate user " + login);
            }
            nextPage = INPUT;
            password = null;
            addFieldError("login", getText("fr.onema.sispea.common.login.errors.failed"));
        }

        openMenu(MenuConstants.MENU_KEY_PERSONAL_SPACE);

        return nextPage;
    }

//    /**
//     * Method used as cork when LDAP server is not available.
//     * @return a statically created administrator user (having all rights and a supervirtual territory).
//     */
//    @SuppressWarnings("unused")
//    private User getUserCork() {
//        User lUser = new User();
//        user.setFirstName("Xavier");
//        user.setEmail("xavier.perrot@c-s.fr");
//        user.setId(1);
//        user.setLocalAdmin(true);
//        user.setLogin("xperrot");
//        user.setName("Perrot");
//        user.setTerritory(Territory.getSuperVirtualTerritory());
//        user.setType(UserType.GeneralAdmin);
//
//        return lUser;
//    }

    public UserDto getUser() {
        return user;
    }

    public void setUser(UserDto pUser) {
        user = pUser;
    }

    public String getLogin() {
        return login;
    }

    public void setLogin(String pLogin) {
        login = pLogin;
    }

    public String getPassword() {
        return password;
    }

    public void setPassword(String pPassword) {
        password = pPassword;
    }

    public String getUrlGoingTo() {
        return urlGoingTo;
    }

}
