/*
 * #%L
 * Wao :: Web Interface
 * 
 * $Id: Synthesis.java 1533 2012-02-23 15:40:52Z bleny $
 * $HeadURL: http://svn.forge.codelutin.com/svn/wao/tags/wao-3.4.1/wao-ui/src/main/java/fr/ifremer/wao/ui/pages/Synthesis.java $
 * %%
 * Copyright (C) 2009 - 2010 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

package fr.ifremer.wao.ui.pages;

import fr.ifremer.wao.WaoException;
import fr.ifremer.wao.bean.BoardingResult;
import fr.ifremer.wao.bean.ConnectedUser;
import fr.ifremer.wao.bean.ContactAverageReactivity;
import fr.ifremer.wao.bean.ContactFilter;
import fr.ifremer.wao.bean.ContactState;
import fr.ifremer.wao.bean.ContactStateStatistics;
import fr.ifremer.wao.bean.GlobalIndicatorValue;
import fr.ifremer.wao.bean.GlobalSynthesisParameters;
import fr.ifremer.wao.bean.GlobalSynthesisResult;
import fr.ifremer.wao.bean.ObsProgram;
import fr.ifremer.wao.bean.SynthesisId;
import fr.ifremer.wao.bean.UserRole;
import fr.ifremer.wao.entity.Company;
import fr.ifremer.wao.entity.Indicator;
import fr.ifremer.wao.entity.IndicatorLevel;
import fr.ifremer.wao.service.ServiceSampling;
import fr.ifremer.wao.service.ServiceSynthesis;
import fr.ifremer.wao.ui.components.ContactFilterComponent;
import fr.ifremer.wao.ui.data.ChartUtils;
import fr.ifremer.wao.ui.data.ChartUtils.ChartType;
import fr.ifremer.wao.ui.data.RequiresAuthentication;
import fr.ifremer.wao.ui.services.WaoManager;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.tapestry5.Block;
import org.apache.tapestry5.ComponentResources;
import org.apache.tapestry5.RenderSupport;
import org.apache.tapestry5.annotations.Environmental;
import org.apache.tapestry5.annotations.Import;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.Log;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.corelib.components.Zone;
import org.apache.tapestry5.ioc.Messages;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.jfree.chart.JFreeChart;
import org.slf4j.Logger;

import java.text.DateFormat;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;

import static org.nuiton.i18n.I18n.n_;

/**
 * Synthesis
 *
 * Created: 9 nov. 2009
 *
 * @author fdesbois <fdesbois@codelutin.com>
 */
@RequiresAuthentication(allowedRoles = {UserRole.ADMIN, UserRole.COORDINATOR, UserRole.OBSERVER, UserRole.PROFESSIONAL})
@Import(stylesheet = "context:css/synthesis.css")
public class Synthesis {

    @Environmental
    private RenderSupport renderSupport;

    @Inject
    private ServiceSampling serviceSampling;

    @Inject
    private WaoManager manager;

    @Inject
    private ServiceSynthesis serviceSynthesis;

    @Inject
    private Logger log;

    @InjectComponent
    private ContactFilterComponent filterComponent;

    @SessionState
    @Property
    private ConnectedUser user;

    @Persist
    private SynthesisId activeSynthesis;

    @Persist
    private Indicator activeIndicator;

    @Persist
    private IndicatorLevel activeIndicatorLevel;

    @Inject
    private ComponentResources resources;

    @InjectComponent
    private Zone delegator;

    @Inject
    private Messages messages;

    public ServiceSampling getServiceSampling() {
        return serviceSampling;
    }

    void setupRender() {
        // add a special CSS to make global synthesis page printable without
        // menu, filters, etc.
        renderSupport.addStylesheetLink("css/synthesis-print.css", "print");
    }

    public SynthesisId getActiveSynthesis() {
        if (activeSynthesis == null) {
            // fix the synthesis to show according to role
            if (user.isObsDeb()) {
                activeSynthesis = SynthesisId.OBSERVATION_HOUR;
            } else {
                activeSynthesis = SynthesisId.GRAPH_SAMPLING;
            }
        }
        return activeSynthesis;
    }

    private GlobalSynthesisParameters globalSynthesisParameters;

    /** permit lazy call to serviceSynthesis to set globalSynthesisParameters field. */
    public GlobalSynthesisParameters getGlobalSynthesisParameters() {
        if (globalSynthesisParameters == null) {
            globalSynthesisParameters =
                                getGlobalSynthesisResult().getGlobalSynthesisParameters();
        }
        return globalSynthesisParameters;
    }

    private GlobalSynthesisResult globalSynthesisResult;

    public GlobalSynthesisResult getGlobalSynthesisResult() {
        if (globalSynthesisResult == null) {
            globalSynthesisResult = serviceSynthesis.getGlobalSynthesisResult(getFilter());
        }
        return globalSynthesisResult;
    }

    /** update activeIndicator to the indicator corresponding to activeSynthesis.
     * @return the value of activeIndicator field, after modification
     */
    public Indicator getActiveIndicator() {
        SynthesisId activeSynthesis = getActiveSynthesis();
        activeIndicator = getGlobalSynthesisParameters().getParameter(activeSynthesis);
        return activeIndicator;
    }

    public void setActiveIndicator(Indicator activeIndicator) {
        this.activeIndicator = activeIndicator;
        this.activeSynthesis = activeIndicator.getSynthesisId();
    }

    public IndicatorLevel getActiveIndicatorLevel() {
        activeIndicatorLevel = getGlobalSynthesisResult().getLevelForIndicator(getActiveSynthesis());
        return activeIndicatorLevel;
    }

    public Block getActiveBlock() throws WaoException {

        Block activeBlock;
        if (BooleanUtils.isTrue(showGlobalSynthesis)) {
            activeBlock = resources.getBlock("globalSynthesis");
        } else {
            activeBlock = resources.getBlock("ind" + getActiveSynthesis().ordinal());
        }

        if (log.isDebugEnabled()) {
            log.debug("activeBlock = " + activeBlock);
        }

        return activeBlock;
    }

    public String getActionSynthesisId() {
        return SynthesisId.class.getName();
    }

    public String getActionChartType() {
        return ChartType.class.getName();
    }

    @Persist
    private Boolean showGlobalSynthesis;

    public String getActionShowGlobalSynthesis() {
        return "showGlobalSynthesis";
    }

    public NumberFormat getPercentFormat() {
        return NumberFormat.getPercentInstance();
    }

    public DateFormat getDateFormat() {
        return new SimpleDateFormat("dd/MM/yyyy");
    }

    public NumberFormat getNumberFormat() {
        NumberFormat result = NumberFormat.getNumberInstance();
        result.setMaximumFractionDigits(1);
        return result;
    }

    public boolean showObsMer() {
        return user.getProfile().getObsProgram() == ObsProgram.OBSMER;
    }

    @Log
    Object onAction(String actionType, String actionValue) {

        showGlobalSynthesis = false;

        if (log.isDebugEnabled()) {
            log.debug("actionType = " + actionType);
        }

        // Delegator action (menu)
        if (actionType.equals(SynthesisId.class.getName())) {
            activeSynthesis = SynthesisId.valueOf(actionValue);
            return delegator.getBody();
            // Change chart type for dataSampling
        } else if (actionType.equals(ChartType.class.getName())) {
            dataSamplingChartType = ChartType.valueOf(actionValue);
        } else if (actionType.equals("showGlobalSynthesis")) {
            showGlobalSynthesis = true;
            return delegator.getBody();
        }
        return this;
    }

    /********************* FILTERS ********************************************/

    public ContactFilter getFilter() throws WaoException {
        return filterComponent.getFilter();
    }

    /********************* DYNAMICAL GRAPH : DATA SAMPLING ********************/

    @Persist
    private ChartType dataSamplingChartType;

    private boolean reset;

    public JFreeChart getDataSamplingChart() throws WaoException {
        Pair<SortedMap<Date, Integer>, SortedMap<Date, Integer>> res = serviceSynthesis.getDataSampling(getFilter());
        Map<String, Map<?, Integer>> data = new LinkedHashMap<String, Map<?, Integer>>();
        data.put(messages.get(n_("wao.ui.samplingPlan.expected")), res.getLeft());
        String realText =
                getFilter().getEstimatedTides() ? messages.get(n_("wao.ui.samplingPlan.estimated"))
                                                : messages.get(n_("wao.ui.samplingPlan.actual"));
        data.put(realText, res.getRight());
        String title = manager.getDataSamplingTitle(getFilter().getSamplingFilter());
        return ChartUtils.createCategoryChart(title, messages.get(n_("wao.ui.chart.numberObservations")),
                      messages.get(n_("wao.ui.samplingPlan.months")), getDataSamplingChartType(), data);
    }

    public ChartType getDataSamplingChartType() {
        // Default type is BAR
        if (dataSamplingChartType == null) {
            dataSamplingChartType = ChartType.BAR;
        }
        if (log.isDebugEnabled()) {
            log.debug("ChartType : " + dataSamplingChartType);
        }
        return dataSamplingChartType;
    }

    /********************* STATIC GRAPH : BOARDING BOAT ***********************/

    private BoardingResult boardingResult;

    /*public JFreeChart getBoardingBoatsChart() throws WaoException {
        Map<String, Map<?, Integer>> data =
                new HashMap<String, Map<?, Integer>>();
        data.put("Navires", getBoardingResult().getBoardings());

        String title = "Sollicitations des navires depuis le " +
                getDateFormat().format(getFilter().getPeriod().getFromDate());

        Company companyForBoarding = getFilter().getCompany();
        if (companyForBoarding != null) {
            title += "\nSociété " + companyForBoarding.getName();
        }
        
        return ChartUtils.createCategoryChart(title, "Nb navires",
                "Nb embarquements", ChartType.BAR, data);
    }*/

    public JFreeChart getBoardingBoatsChart() throws WaoException {
        Map<String, Map<?, Integer>> data =
                new LinkedHashMap<String, Map<?, Integer>>();

        data.put(messages.get(n_("wao.ui.chart.numberOfBoatsWithBoardings")), getBoardingResult().getBoardings());

        String title = String.format(messages.get(n_("wao.ui.chart.boardingsSince")),
                getDateFormat().format(getFilter().getPeriod().getFromDate()));

        Company companyForBoarding = getFilter().getSamplingFilter().getCompany();
        if (companyForBoarding != null) {
            title += "\n" + messages.get("wao.ui.entity.Company") + " " + companyForBoarding.getName();
        }

        return ChartUtils.createCategoryChart(title,
                                              messages.get(n_("wao.ui.chart.numberBoats")),
                                              messages.get(n_("wao.ui.chart.numberObservations")),
                                              ChartType.BAR, data);
    }

    public BoardingResult getBoardingResult() throws WaoException {
        if (boardingResult == null) {
            boardingResult = serviceSynthesis.getBoardingBoats(getFilter());
        }
        return boardingResult;
    }

    /********************* INDICATOR : COMPLIANCE BOARDING ********************/

    private Map<String, Double> complianceBoarding;

    @Property
    private Map.Entry<String, Double> complianceBoardingEntry;

    /**
     * Data for nonComplianceBoarding indicator. The key is the company 
     * considered and the value is the indicator value. The map will contains
     * only one row, if the current user is an observer.
     *
     * @return a Map with companies and there values for nonComplianceBoarding
     * @throws WaoException 
     */
    public Map<String, Double> getComplianceBoarding() throws WaoException {
        if (complianceBoarding == null) {
            complianceBoarding =
                    serviceSynthesis.getComplianceBoardingIndicator(getFilter());
        }
        return complianceBoarding;
    }

    /**
     * Return the value of nonComplianceBoarding indicator. Only useful for
     * observer user.
     *
     * @return the value of the indicator
     * @throws WaoException
     */
    public Double getComplianceBoardingValue() throws WaoException {
        return getComplianceBoarding().get(user.getCompany().getName());
    }

    /********************* INDICATOR : CONTACT STATE ******************/

    private Collection<ContactStateStatistics> contactStateStatistics;

    @Property
    private ContactStateStatistics contactStateStats;

    @Property
    private ContactState contactState;

    public List<ContactState> getContactStates() {
        // we should print only stats for states available for current program
        return ContactState.getAllowedStates(getFilter().getObsProgram());
    }

    /**
     * Get all contactStateStatistics from serviceSynthesis. There will be
     * only one result if the current user is an observer.
     *
     * @return all contactStateStatistics depends on company and selected period
     * @throws WaoException
     */
    public Collection<ContactStateStatistics> getContactStateStatistics()
            throws WaoException {
        if (contactStateStatistics == null) {
            contactStateStatistics = 
                    serviceSynthesis.getContactStateStatistics(getFilter());
        }
        return contactStateStatistics;
    }

    /**
     * Get the value for current contactStateStats in table row (by company)
     * and current contactState in table column.
     *
     * @return the number of contact states for the current contactState
     */
    public int getNbContactStates() {
        return contactStateStats.getData().get(contactState);
    }

    /**
     * Get the percent value for the current contactStateStats in table row
     * (by company) and current contactState in table column.
     *
     * @return the percent value of contact states for the current contactState
     */
    public double getPercentContactStates() {
        if (log.isDebugEnabled())  {
            log.debug("nbContactStates : " + getNbContactStates());
            log.debug("total : " + contactStateStats.getTotal());
        }
        return (double)getNbContactStates() / (double)contactStateStats.getTotal();
    }

    /********************* INDICATOR : ALLEGRO REACTIVITY ******************/

    private Collection<ContactAverageReactivity> allegroReactivity;

    @Property
    private ContactAverageReactivity allegroReactivityEntry;

    public Collection<ContactAverageReactivity> getAllegroReactivity()
            throws WaoException {
        if (allegroReactivity == null) {
            allegroReactivity =
                    serviceSynthesis.getContactDataInputDateReactivity(getFilter());
        }
        return allegroReactivity;
    }

    public Double getAllegroReactivityValue() throws WaoException {

        // FIXME bleny 20101021 in this case, what occurs in indicator ?

        if (!getAllegroReactivity().isEmpty()) {
            ContactAverageReactivity entry = (ContactAverageReactivity)
                    CollectionUtils.get(getAllegroReactivity(), 0);
            return entry.getResult();
        } else {
            return 0.;
        }
    }

    public String getAllegroReactivityStyle() throws WaoException {
        double value = user.isAdmin() ?
            allegroReactivityEntry.getResult() : getAllegroReactivityValue();
        if (value >= 15.) {
            return " warn";
        }
        return "";
    }

    /********************* INDICATOR : DATA RELIABILITY ***********************/

    private Map<String, Double> dataReliability;

    /* variable used in template */
    @Property
    private Map.Entry<String, Double> dataReliabilityEntry;

    public Map<String, Double> getDataReliability() {
        if (dataReliability == null) {
            dataReliability = serviceSynthesis.getDataReliability(getFilter());
        }
        return dataReliability;
    }

    public Double getDataReliabilityValue() throws WaoException {
        return getDataReliability().get(user.getCompany());
    }

    /********************* INDICATOR : OBSERVATION HOUR ***********************/

    public JFreeChart getObservationHoursChart() throws WaoException {
        Map result = serviceSynthesis.getObservationHours(getFilter());
        String title = messages.get(n_("wao.ui.synthesis.observationHours.title"));
        String axisName = messages.get(n_("wao.ui.synthesis.observationHours.axisName"));
        String categoryName = messages.get(n_("wao.ui.synthesis.observationHours.categoryName"));
        return ChartUtils.createCategoryChart(title, axisName, categoryName, ChartType.BAR, result);
    }

    /********************* INDICATOR : OBSERVATION HOUR ***********************/

    public JFreeChart getDistinctBoatsCountsChart() throws WaoException {
        Map result = serviceSynthesis.getDistinctBoatsCounts(getFilter());
        String title = messages.get(n_("wao.ui.synthesis.distinctBoatsCounts.title"));
        String axisName = messages.get(n_("wao.ui.synthesis.distinctBoatsCounts.axisName"));
        String categoryName = messages.get(n_("wao.ui.synthesis.distinctBoatsCounts.categoryName"));
        return ChartUtils.createCategoryChart(title, axisName, categoryName, ChartType.BAR, result);
    }

    /********************* GLOBAL SYNTHESIS ***********************************/

    @Property
    private GlobalIndicatorValue globalLevels[] = GlobalIndicatorValue.values();

    /* variable used in template */
    @Property
    private GlobalIndicatorValue globalLevel;

    public boolean highlightGlobalLevel() {
        return globalLevel == getGlobalSynthesisResult().getGlobalLevel();
    }

}