/*
 * #%L
 * Wao :: Web Interface
 * 
 * $Id: GenericSelectModel.java 650 2010-10-03 15:14:48Z fdesbois $
 * $HeadURL: http://svn.forge.codelutin.com/svn/wao/tags/wao-3.2.2/wao-ui/src/main/java/fr/ifremer/wao/ui/data/GenericSelectModel.java $
 * %%
 * Copyright (C) 2009 - 2010 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package fr.ifremer.wao.ui.data;

import org.apache.tapestry5.OptionGroupModel;
import org.apache.tapestry5.OptionModel;
import org.apache.tapestry5.internal.OptionGroupModelImpl;
import org.apache.tapestry5.internal.OptionModelImpl;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.ioc.services.PropertyAccess;
import org.apache.tapestry5.ioc.services.PropertyAdapter;
import org.apache.tapestry5.util.AbstractSelectModel;
import org.slf4j.Logger;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * GenericSelectModel.java
 *
 * Based on IdSelectModel from <http://jumpstart.doublenegative.com.au:8080/jumpstart/examples/select/easyid>
 * A generic selection model whose value is the id of the chosen object, not the object itself.
 * 
 * @param <T>
 * @author fdesbois <fdesbois@codelutin.com>
 */
public class GenericSelectModel<T> extends AbstractSelectModel {

    private List<T> list;
    private Map<String, List<T>> map;
    private PropertyAdapter labelFieldAdapter;
    private PropertyAdapter idFieldAdapter;
    
    @Inject
    private Logger log;

    /**
     * @param list the list of objects you want modeled in a Select component. These objects MUST implement
     *        equals(Object obj) and hashCode(). If the objects are JPA entities, ensure their implementations of
     *        equals(Object obj) and hashCode() return the same thing for different instances of the same detached
     *        entity.
     * @param clazz the class of objects in the list.
     * @param labelField the name of the field you want displayed as the label in the selection list, eg. "name".
     * @param idField the name of the field which is the unique identifier of each object in the list, eg. "id". This is
     *        used in the value property of the Select component.
     * @param access Declare a PropertyAccess injected into your page (eg. Inject private PropertyAccess _access) then pass it in here.
     *
     */
    public GenericSelectModel(List<T> list, Class<T> clazz, String labelField, String idField, PropertyAccess access) {
        init(clazz, labelField, idField, access);

        this.list = list;
    }

    public GenericSelectModel(Map<String, List<T>> map, Class<T> clazz, String labelField, String idField, PropertyAccess access) {
        init(clazz, labelField, idField, access);

        this.map = map;
    }

    private void init(Class<T> clazz, String labelField, String idField, PropertyAccess access) {
        if (clazz == null) {
            throw new IllegalArgumentException("clazz is required.");
        }
        if (idField == null) {
            throw new IllegalArgumentException("idField is required.");
        }
        if (labelField == null) {
            throw new IllegalArgumentException("labelField is required.");
        }

        this.idFieldAdapter = access.getAdapter(clazz).getPropertyAdapter(idField);
        this.labelFieldAdapter = access.getAdapter(clazz).getPropertyAdapter(labelField);

        if (idFieldAdapter == null) {
            throw new IllegalArgumentException("idField " + idField + " does not exist in class " + clazz + ".");
        }
        if (labelFieldAdapter == null) {
            throw new IllegalArgumentException("labelField " + idField + " does not exist in class " + clazz + ".");
        }
    }

    @Override
    public List<OptionGroupModel> getOptionGroups() {
        if (list != null) {
            return null;
        }
        List<OptionGroupModel> optionGroupModelList = new ArrayList<OptionGroupModel>();
        for (String key : map.keySet()) {
            optionGroupModelList.add(new OptionGroupModelImpl(key, false, getOptions(map.get(key))));
        }
        return optionGroupModelList;
    }

    @Override
    public List<OptionModel> getOptions() {
        if (map != null) {
            return null;
        }
        return getOptions(list);
    }

    private List<OptionModel> getOptions(List<T> list) {
        List<OptionModel> optionModelList = new ArrayList<OptionModel>();
        for (T obj : list) {
            optionModelList.add(new OptionModelImpl(nvl(labelFieldAdapter.get(obj)), idFieldAdapter.get(obj)));
        }
        return optionModelList;
    }

    public List<T> getList() {
        List<T> results;
        if (map != null) {
            results = new ArrayList<T>();
            for (String key : map.keySet()) {
                results.addAll(map.get(key));
            }
        } else {
            results = list;
        }
        return results;
    }

    public T findObject(String id) {
        if (id != null) {
            for (T current : getList()) {
                Object currentId = idFieldAdapter.get(current);
                if (currentId.equals(id)) {
                    return current;
                }
            }
        }
        return null;
    }

    private String nvl(Object o) {
        return o == null ? "" : o.toString();
    }
}

