/*
 * #%L
 * IsisFish
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2010 Ifremer, CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.ui.input.model;

import static org.nuiton.i18n.I18n.t;

import java.awt.Component;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableCellRenderer;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import fr.ifremer.isisfish.entities.Equation;
import fr.ifremer.isisfish.entities.Gear;
import fr.ifremer.isisfish.entities.Selectivity;
import fr.ifremer.isisfish.ui.sensitivity.SensitivityTableModel;

/**
 * Table model for {@link Gear}#{@link Selectivity}.
 * 
 * Columns :
 * <ul>
 * <li>selectivity population name</li>
 * <li>selectivity equation</li>
 * </ul>
 *
 * @author chatellier
 * @version $Revision: 3969 $
 * 
 * Last update : $Date: 2014-04-17 18:48:13 +0200 (Thu, 17 Apr 2014) $
 * By : $Author: echatellier $
 */
public class GearPopulationSelectivityModel extends AbstractTableModel implements TableCellRenderer, SensitivityTableModel {

    /** Log. */
    private static Log log = LogFactory.getLog(GearPopulationSelectivityModel.class);
    
    /** serialVersionUID. */
    private static final long serialVersionUID = 3169786638868209920L;

    /** Columns names. */
    public final static String[] COLUMN_NAMES = {
            t("isisfish.common.population"),
            t("isisfish.common.equation")
    };

    protected List<Selectivity> selectivities;

    /**
     * Empty constructor.
     */
    public GearPopulationSelectivityModel() {
        this(null);
    }

    /**
     * Constructor with data.
     *  
     * @param selectivities initial selectivities
     */
    public GearPopulationSelectivityModel(
            List<Selectivity> selectivities) {
        this.selectivities = selectivities;
    }

    /**
     * Set target species list.
     * 
     * @param selectivities the selectivities to set
     */
    public void setSelectivities(List<Selectivity> selectivities) {
        this.selectivities = selectivities;
    }

    /**
     * Get selectivity list.
     * 
     * @return selectivity list
     */
    public List<Selectivity> getSelectivities() {
        return this.selectivities;
    }
    
    /*
     * @see javax.swing.table.TableModel#getColumnCount()
     */
    @Override
    public int getColumnCount() {
        return COLUMN_NAMES.length;
    }

    /*
     * @see javax.swing.table.TableModel#getRowCount()
     */
    @Override
    public int getRowCount() {
        int rows = 0;
        if (selectivities != null) {
            rows = selectivities.size();
        }
        return rows;
    }

    /*
     * @see javax.swing.table.TableModel#getValueAt(int, int)
     */
    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {

        Object result = null;

        Selectivity selectivity = selectivities.get(rowIndex);
        switch (columnIndex) {
        case 0:
            result = selectivity.getPopulation().getName();
            break;
        case 1:
            result = selectivity.getEquation();
            break;
        default:
            throw new IndexOutOfBoundsException("No such column " + columnIndex);
        }

        return result;
    }

    /*
     * @see javax.swing.table.TableModel#getColumnClass(int)
     */
    @Override
    public Class<?> getColumnClass(int columnIndex) {

        Class<?> result = null;
        
        switch (columnIndex) {
        case 0:
            result = String.class;
            break;
        case 1:
            result = Equation.class;
            break;
        default:
            throw new IndexOutOfBoundsException("No such column " + columnIndex);
        }
        
        return result;
    }

    /*
     * @see javax.swing.table.TableModel#getColumnName(int)
     */
    @Override
    public String getColumnName(int columnIndex) {
        return COLUMN_NAMES[columnIndex];
    }

    /*
     * @see javax.swing.table.TableModel#isCellEditable(int, int)
     */
    @Override
    public boolean isCellEditable(int rowIndex, int columnIndex) {
        return columnIndex > 0;
    }

    /*
     * @see javax.swing.table.TableModel#setValueAt(java.lang.Object, int, int)
     */
    @Override
    public void setValueAt(Object value, int rowIndex, int columnIndex) {
        
        if (log.isDebugEnabled()) {
            log.debug("Cell edition (column " + columnIndex + ") = " + value);
        }
        
        Selectivity selectivity = selectivities.get(rowIndex);
        switch (columnIndex) {
        case 1:
            Equation eq = (Equation)value;
            // two events for event to be fired
            selectivity.setEquation(null);
            selectivity.setEquation(eq);
            break;
        default:
            throw new IndexOutOfBoundsException("Can't edit column " + columnIndex);
        }

    }

    /*
     * @see javax.swing.table.TableCellRenderer#getTableCellRendererComponent(javax.swing.JTable, java.lang.Object, boolean, boolean, int, int)
     */
    @Override
    public Component getTableCellRendererComponent(JTable table, Object value,
            boolean isSelected, boolean hasFocus, int row, int column) {

        Component c = null;
        switch (column) {
        case 0:
            c = new JLabel(value.toString());
            break;
        case 1:
            Equation equation = (Equation)value;
            c = new JButton(equation.getName() + "(" + equation.getCategory() + ")");
            break;
        default:
            throw new IndexOutOfBoundsException("No such column " + column);
        }
        return c;
    }

    /*
     * @see fr.ifremer.isisfish.ui.sensitivity.SensitivityTableModel#getPropertyAtColumn(int)
     */
    @Override
    public String getPropertyAtColumn(int column) {
        String result = null;
        if (column == 1) {
            result = "equation";
        }
        return result;
    }
    
    /*
     * @see fr.ifremer.isisfish.ui.sensitivity.SensitivityTableModel#getBeanAtRow(int)
     */
    @Override
    public Object getBeanAtRow(int rowIndex) {
        Object result = null;
        result = selectivities.get(rowIndex);
        return result;
    }
}
