/*
 * #%L
 * IsisFish
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2002 - 2012 Ifremer, Code Lutin, Benjamin Poussin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish;

import static org.nuiton.i18n.I18n._;
import static org.nuiton.i18n.I18n.n_;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.apache.commons.beanutils.ConvertUtils;
import org.apache.commons.lang3.time.DurationFormatUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.util.StringUtil;
import org.nuiton.util.Version;

import fr.ifremer.isisfish.actions.ExportAction;
import fr.ifremer.isisfish.actions.ImportAction;
import fr.ifremer.isisfish.actions.OtherAction;
import fr.ifremer.isisfish.actions.SimulationAction;
import fr.ifremer.isisfish.actions.VCSAction;
import fr.ifremer.isisfish.simulator.SimulationContext;
import fr.ifremer.isisfish.simulator.SimulationControl;
import fr.ifremer.isisfish.simulator.launcher.InProcessSimulatorLauncher;
import fr.ifremer.isisfish.simulator.launcher.SSHSimulatorLauncher;
import fr.ifremer.isisfish.simulator.launcher.SimulationService;
import fr.ifremer.isisfish.simulator.launcher.SubProcessSimulationLauncher;
import fr.ifremer.isisfish.vcs.VCS;

/**
 * Isis fish configuration.
 * 
 * @author poussin
 * @version $Revision: 1310 $
 *
 * Last update: $Date: 2008-08-23 00:46:00 +0200 (Sat, 23 Aug 2008) $
 * by : $Author: bpoussin $
 */
public class IsisConfig extends ApplicationConfig {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static Log log = LogFactory.getLog(IsisConfig.class);
    
    /**
     * Config step after init.
     *
     * @see org.nuiton.util.ApplicationConfig.Action.Step
     */
    public static final int STEP_AFTER_INIT = 0;
    
    /**
     * Config step after init vcs.
     * 
     * @see org.nuiton.util.ApplicationConfig.Action.Step
     */
    public static final int STEP_AFTER_INIT_VCS = 1;
    
    /**
     * Config step after ui.
     * 
     * @see org.nuiton.util.ApplicationConfig.Action.Step
     */
    public static final int STEP_AFTER_UI = 2;
    
    /**
     * Config step before exit.
     * 
     * @see org.nuiton.util.ApplicationConfig.Action.Step
     */
    public static final int STEP_BEFORE_EXIT = 3;
    
    /**
     * La version du logiciel constitue de l.d.a.r
     * <li>l: le numero de version du logiciel
     * <li>d: le numero de version du schema de la base de donnees
     * <li>a: le numero de version de l'api des scripts
     * <li>r: le numero de version de de l'interface graphique ou autre modif mineur
     * <p>
     * lors de l'increment de l, d, a et r sont remis a 0
     * lors de l'increment de d, a et r sont remis a 0
     * lors de l'increment de a, r est remis a 0
     * <p>
     * Un changement d'UI ne modifie jamais le numero de version de database
     * Un changement de schema de base pour lequel on ne peut pas faire de
     * migration de donnees demande automatiquement un changement de version
     * d'application.
     */
    protected final static Version version = new Version(4, 2, 0, 1);

    protected final static Version majorVersion = new Version(version.getNumber(0));
    protected final static Version databaseVersion = new Version(
            version.getNumber(0), version.getNumber(1));
    protected final static Version apiVersion = new Version(
            version.getNumber(0), version.getNumber(1), version.getNumber(2));
            
    public static Version getVersionNumber() {
        return version;
    }
    
    /**
     * Le nombre global ex: 3.2.0.0
     * 
     * @return full version
     */
    static public String getVersion() {
        String result = version.toString();
        return result;
    }

    /**
     * La version de la base ex: 3.2
     * 
     * @return database version
     */
    public static Version getDatabaseVersion() {
        return databaseVersion;
    }

    /**
     * La version de l'api de programmation ex: 3.2.0
     * 
     * @return api version
     */
    public static Version getApiVersion() {
        return apiVersion;
    }
    
    static final public String COPYRIGHT_TEXT = "Version " + getVersion() + " IFREMER-MAERHA © 2000-2011";
    static final public String CONFIG_FILENAME = "isis-config-" + majorVersion.toString();
    
    /** separateur de liste */
    static final public String SEP = ",";
            
    protected transient File backupSessionDirectory = null;
    
    /**
     * Mémorise une unité de temps en millisecondes.
     * 
     * Auparavant, c'etait {@link System#nanoTime()}, mais cette unité de temps
     * n'est pas lié a une date.
     * Par exemple, ca permet juste de mesuré avec une precision de 10-9 le temps
     * ecoulé, par l'heure qu'il est à cette unité de temps.
     */
    protected long startingTime = System.currentTimeMillis();
    
    public IsisConfig() {
        
        for (Option o : Option.values()) {
            setDefaultOption(o.key, o.defaultValue);
        }
        
        for (Action a : Action.values()) {
            for (String alias : a.aliases) {
                addActionAlias(alias, a.action);                
            }
        }

    }
    
    
    //////////////////////////////////////////////////
    // Methode d'acces aux options
    //////////////////////////////////////////////////

    /**
     * Retourne le repertoire racine de toutes les donnees (script, simulation
     * region, ...)
     * 
     * @return database directory
     */
    public File getDatabaseDirectory() {
        File result = getOptionAsFile(Option.DATABASE_DIRECTORY.key);
        return result;
    }
    
    /**
     * Retourne le repertoire racine de toutes les donnees communauté (commit utilisateur).
     * 
     * @return community database directory
     */
    public File getCommunityDatabaseDirectory() {
        File result = getOptionAsFile(Option.COMMUNITY_DATABASE_DIRECTORY.key);
        return result;
    }

    /**
     * Get database directory to use for script.
     * 
     * Defaut to {@link #getDatabaseDirectory()} value, but to
     * {@link SimulationContext#getScriptDirectory()} during a simulation.
     * 
     * @return script database directory
     */
    public File getContextDatabaseDirectory() {

        // add a subdirectory for simulation context
        SimulationContext simContext = SimulationContext.get();
        File scriptDirectory = simContext.getScriptDirectory();

        if (scriptDirectory == null) {
            // if null, not in simulation 
            scriptDirectory = getDatabaseDirectory();
        }

        return scriptDirectory;
    }

    /**
     * Retourne le lock a utiliser pour la base h2.
     * 
     * Par defaut la base utilise 'file'.
     * 
     * Les valeurs acceptées sont :
     *  - file
     *  - socket
     *  - no
     * 
     * http://www.h2database.com/html/features.html#database_file_locking
     * 
     * @return h2 db lock mode
     */
    public String getDatabaseLockMode() {
        String result = getOption(Option.DATABASE_LOCK_MODE.key);
        return result;
    }

    /**
     * Retourne le repertoire ou sont stockes les scripts compiles.
     * 
     * Create directory if not exists.
     * 
     * Make a specific build directory for running simulation.
     * 
     * @return compilation directory
     */
    public File getCompileDirectory() {
        File result = getOptionAsFile(Option.COMPILATION_DIRECTORY.key);

        // add a subdirectory for simulation context
        SimulationContext simContext = SimulationContext.get();
        SimulationControl control = simContext.getSimulationControl();
        if (control != null) {
            result = new File(result, control.getId());
        }
        
        if (!result.exists()) {
            result.mkdirs();
        }

        return result;
    }

    /**
     * Retourne le repertoire ou sont stockes les informations relatives
     * aux simulations en cours.
     * 
     * @return monitoring directory
     */
    public File getMonitoringDirectory() {
        File result = getOptionAsFile(Option.MONITORING_DIRECTORY.key);

        if (!result.exists()) {
            result.mkdirs();
        }

        return result;
    }

    /**
     * Get javadoc directory.
     * 
     * Create directory if not exists.
     * 
     * @return javadoc directory
     */
    public File getJavadocDirectory() {
        File result = getOptionAsFile(Option.JAVADOC_DIRECTORY.key);
        if (!result.exists()) {
            result.mkdirs();
        }
        return result;
    }

    /**
     * Retourne l'objet {@link Locale} a utilise pour la langue.
     * 
     * @return application {@link Locale}
     */
    public Locale getLocale() {
        String value = getOption(Option.LOCALE.key);
        Locale result = (Locale)ConvertUtils.convert(value, Locale.class);
        return result;
    }

    /**
     * Change application locale.
     * (used during application config command line parse)
     * 
     * @param locale new locale
     */
    public void setLocale(String locale) {
        setOption(Option.LOCALE.key, locale);
    }

    /**
     * Retourne l'encoding a utiliser pour les fichiers textes.
     * 
     * @return encoding to use
     */
    public String getEncoding() {
        String result = getOption(Option.ENCODING.key);
        return result;
    }

    /**
     * Retourne le serveur SMTP a utiliser pour l'envoie de mail.
     * 
     * @return smtp server address
     */
    public String getSmtpServer() {
        String result = getOption(Option.SMTP_SERVER.key);
        return result;
    }

    /**
     * Retourne le nom usuel de l'utilisateur.
     * 
     * @return username
     */
    public String getUserName() {
        String result = getOption(Option.USER_NAME.key);
        return result;
    }
    
    /**
     * Retourne l'email de l'utilisateur.
     * 
     * @return user email
     */
    public String getUserMail() {
        String result = getOption(Option.USER_MAIL.key);
        return result;
    }
    
    public String getSimulatorClassfile() {
        String result = getOption(Option.SIMULATOR_CLASSFILE.key);
        return result;
    }
    
    public void setSimulatorClassfile(String value) {
        setOption(Option.SIMULATOR_CLASSFILE.key, value);
    }

    /**
     * Retourne le nombre maximum de thread à utiliser en in process.
     * 
     * @return number of thread to use
     */
    public int getSimulatorInMaxThreads() {
        int result = getOptionAsInt(Option.SIMULATOR_IN_MAXTHREADS.key);
        return result;
    }

    /**
     * Retourne le nombre de sous processus maximum a creer.
     * 
     * @return number of sub process to create
     */
    public int getSimulatorSubMaxProcess() {
        int result = getOptionAsInt(Option.SIMULATOR_SUB_MAXPROCESS.key);
        return result;
    }
    
    /**
     * Retourne la quantité de ram (Xmx) a alloue au sous processus.
     * 
     * @return memory amount to use for sub process
     */
    public String getSimulatorSubMaxMemory() {
        String result = getOption(Option.SIMULATOR_SUB_MAXMEMORY.key);
        return result;
    }

    /**
     * Retourne l'url du serveur de simulation accessible via SSH.
     * 
     * @return simulator server
     */
    public String getSimulatorSshServer() {
        String result = getOption(Option.SIMULATOR_SSH_SERVER.key);
        return result;
    }

    /**
     * Change ssh server url value.
     * 
     * @param sshServer
     */
    public void setSimulatorSshServer(String sshServer) {
        setOption(Option.SIMULATOR_SSH_SERVER.key, sshServer);
    }

    /**
     * Retourne le login pour acceder au serveur de simulation accessible via SSH.
     * 
     * @return simulator username
     */
    public String getSimulatorSshUsername() {
        String result = getOption(Option.SIMULATOR_SSH_USER_NAME.key);
        return result;
    }

    /**
     * Change ssh username.
     * 
     * @param username username
     */
    public void setSimulatorSshUsername(String username) {
        setOption(Option.SIMULATOR_SSH_USER_NAME.key, username);
    }

    /**
     * Retourne le chemin relatif ou abs du repertoire root des données d'isis.
     * 
     * @return simulator data path
     */
    public String getSimulatorSshDataPath() {
        String result = getOption(Option.SIMULATOR_SSH_DATAPATH.key);
        return result;
    }

    /**
     * Change remote datapath.
     * 
     * @param datapath datapath
     */
    public void setSimulatorSshDataPath(String datapath) {
        setOption(Option.SIMULATOR_SSH_DATAPATH.key, datapath);
    }

    /**
     * Get remote user home directory.
     * 
     * @return remote user home
     */
    public String getSimulatorSshUserHome() {
        String result = getOption(Option.SIMULATOR_SSH_USER_HOME.key);
        return result;
    }

    /**
     * Change isis remote home directory.
     * 
     * @param userhome new remote home
     */
    public void setSimulatorSshUserHome(String userhome) {
        setOption(Option.SIMULATOR_SSH_USER_HOME.key, userhome);
    }

    /**
     * Retourne le chemin distant ou est installé isis.
     * 
     * @return remote isis home
     */
    public String getSimulatorSshIsisHome() {
        String result = getOption(Option.SIMULATOR_SSH_ISIS_HOME.key);
        return result;
    }

    /**
     * Change isis home on ssh server.
     * 
     * @param isishome isis home
     */
    public void setSimulatorSshIsisHome(String isishome) {
        setOption(Option.SIMULATOR_SSH_ISIS_HOME.key, isishome);
    }

    /**
     * Retourne l'emplacement du dossier temporaire distant.
     * 
     * Ce dossier doit être ABSOLUT.
     * 
     * @return tmp dir
     */
    public String getSimulatorSshTmpPath() {
        String result = getOption(Option.SIMULATOR_SSH_TMPPATH.key);
        return result;
    }

    /**
     * Change ssh temp path.
     * 
     * @param sshtemppath temp path
     */
    public void setSimulatorSshTmpPath(String sshtemppath) {
        setOption(Option.SIMULATOR_SSH_TMPPATH.key, sshtemppath);
    }

    /**
     * PBS bin path directory.
     * 
     * @return path
     */
    public String getSimulatorSshPbsBinPath() {
        String result = getOption(Option.SIMULATOR_SSH_PBSBINPATH.key);
        return result;
    }

    /**
     * Change PBS bin path.
     * 
     * @param path new path
     */
    public void setSimulatorSshPbsBinPath(String path) {
        setOption(Option.SIMULATOR_SSH_PBSBINPATH.key, path);
    }
    
    /**
     * PBS qsub options (command line).
     * 
     * See man qsub for available options.
     * 
     * @return options
     */
    public String getSimulatorSshPbsQsubOptions() {
        String result = getOption(Option.SIMULATOR_SSH_PBSQSUBOPTIONS.key);
        return result;
    }

    /**
     * Change PBS qsub options (command line).
     * 
     * @param options new options
     */
    public void setSimulatorSshPbsQsubOptions(String options) {
        setOption(Option.SIMULATOR_SSH_PBSQSUBOPTIONS.key, options);
    }
    
    /**
     * Java path directory.
     * 
     * @return path
     */
    public String getSimulatorSshJavaPath() {
        String result = getOption(Option.SIMULATOR_SSH_JAVAPATH.key);
        return result;
    }

    /**
     * Change Java bin path.
     * 
     * @param path new path
     */
    public void setSimulatorSshJavaPath(String path) {
        setOption(Option.SIMULATOR_SSH_JAVAPATH.key, path);
    }

    /**
     * Retourne l'interval de temps a utiliser
     * pour recuperer le fichier de control.
     * 
     * @return time (in seconds)
     */
    public int getSimulatorSshControlCheckInterval() {
        int result = getOptionAsInt(Option.SIMULATOR_SSH_CONTROLCHECKINTERVAL.key);
        return result;
    }

    /**
     * Change control check interval.
     * 
     * @param interval interval
     */
    public void setSimulatorSshControlCheckInterval(int interval) {
        setOption(Option.SIMULATOR_SSH_CONTROLCHECKINTERVAL.key, String.valueOf(interval));
    }
    
    /**
     * Retourne le nombre de thread simultane a utiliser par le pool de thread
     * du launcher SSH.
     * 
     * @return number of thread to use
     */
    public int getSimulatorSshMaxThreads() {
        int result = getOptionAsInt(Option.SIMULATOR_SSH_MAXTHREADS.key);
        return result;
    }

    /**
     * Set number of ssh simulation thread to use.
     * 
     * @param max max
     */
    public void setSimulatorSshMaxThreads(int max) {
        setOption(Option.SIMULATOR_SSH_MAXTHREADS.key, String.valueOf(max));
    }

    /**
     * Retourne la quantité de ram (Xmx) a alloue au processus sur caparmor.
     * 
     * @return number of thread to use
     */
    public String getSimulatorSshMaxMemory() {
        String result = getOption(Option.SIMULATOR_SSH_MAXMEMORY.key);
        return result;
    }

    /**
     * Set ssh simulation process memory to use
     * 
     * @param memory memory
     */
    public void setSimulatorSshMaxMemory(String memory) {
        setOption(Option.SIMULATOR_SSH_MAXMEMORY.key, memory);
    }

    /**
     * Retourne la clé privée de l'utilisteur courant.
     * 
     * @return private ssh key path
     */
    public File getSSHPrivateKeyFilePath() {
        File result = getOptionAsFile(Option.SSH_KEY_FILE.key);
        return result;
    }

    /**
     * Set ssh private key file path.
     * 
     * @param sshFile new value
     */
    public void setSSHPrivateKeyFilePath(File sshFile) {
        setOption(Option.SSH_KEY_FILE.key, sshFile.getAbsolutePath());
    }

    /*public boolean isUseVCS() {
        String value = getOption(Option.VCS_TYPE.key);
        boolean result = !VCSNone.TYPE_NONE.equals(value);
        return result;
    }*/

    /**
     * Get launch.ui option value.
     * 
     * @return launch ui option value
     */
    public boolean isLaunchUI() {
        boolean result = getOptionAsBoolean(Option.LAUNCH_UI.key);
        return result;
    }

    /**
     * Get perform migration option value.
     * 
     * @return perform migration option value
     */
    public boolean isPerformMigration() {
        boolean result = getOptionAsBoolean(Option.PERFORM_MIGRATION.key);
        return result;
    }

    /**
     * Get perform vcs update option value.
     * 
     * @return perform vcs update option value
     */
    public boolean isPerformVcsUpdate() {
        boolean result = getOptionAsBoolean(Option.PERFORM_VCS_UPDATE.key);
        return result;
    }

    /**
     * Get perform cron option value.
     * 
     * @return perform cron option value
     */
    public boolean isPerformCron() {
        boolean result = getOptionAsBoolean(Option.PERFORM_CRON.key);
        return result;
    }

    /**
     * Get script auto configuration option value.
     * 
     * @return autoconfiguration
     */
    public boolean isScriptAutoconfig() {
        boolean result = getOptionAsBoolean(Option.SIMULATION_SCRIPT_AUTOCONFIG.key);
        return result;
    }

    public void setSimulatorLauncher(String value) {
        setOption(Option.SIMULATOR_LAUNCHER.key, value);
    }

    public boolean isSimulationShowOnlyError() {
        Boolean result = getOptionAsBoolean(Option.SIMULATION_SHOW_ONLY_ERROR.key);
        return result;
    }

    public String getDefaultExportNames() {
        String result = getOption(Option.DEFAULT_EXPORT_NAMES.key);
        return result;
    }

    public void setDefaultExportNames(List<String> exportNames) {
        StringBuilder sb = new StringBuilder();
        for (String exportName : exportNames) {
            sb.append(SEP).append(exportName);
        }
        String value = sb.toString();
        // remove first SEP
        if (value.length() > 0) {
            value = value.substring(1);
        }
        setOption(Option.DEFAULT_EXPORT_NAMES.key, value);
        saveForUser();
    }

    /**
     * @return la liste des noms d'exports par defaut sous forme de liste,
     *         a partir de la propriete {@link Option#DEFAULT_EXPORT_NAMES}
     *         ou null si ils n'ont jamais ete sauves par l'utilisateur.
     *         by user.
     * @see Option#DEFAULT_EXPORT_NAMES
     */
    public List<String> getDefaultExportNamesAsList() {
        List<String> result = null;
        String exportNamesList = getDefaultExportNames();
        if (exportNamesList != null) {
            result = new ArrayList<String>();
            String[] exportNames = StringUtil.split(exportNamesList, ",");
            result.addAll(Arrays.asList(exportNames));
        }
        return result;
    }

   public String getDefaultMapFilename() {
        String result = getOption(Option.DEFAULT_MAP_FILENAME.key);
        return result;
    }

   public String getDefaultResultNames() {
        String result = getOption(Option.DEFAULT_RESULT_NAMES.key);
        return result;
    }

    public void setDefaultResultNames(List<String> resultNames) {
        StringBuilder sb = new StringBuilder();
        for (String resultName : resultNames) {
            sb.append(SEP).append(resultName);
        }
        String value = sb.toString();
        // remove first SEP
        if (value.length() > 0) {
            value = value.substring(1);
        }
        setOption(Option.DEFAULT_RESULT_NAMES.key, value);
        saveForUser();
    }

    /**
     * @return les resultats par defaut d'une simulation sous forme de liste
     *         a partir de la propriete {@link Option#DEFAULT_RESULT_NAMES}
     * @see Option#DEFAULT_RESULT_NAMES
     */
    public List<String> getDefaultResultNamesAsList() {
        List<String> result = null;
        String resultNamesList = getDefaultResultNames();
        if (resultNamesList != null) {
            result = new ArrayList<String>();
            String[] resultNames = StringUtil.split(resultNamesList, ",");
            result.addAll(Arrays.asList(resultNames));
        }
        return result;
    }
    
    public String getDefaultTagValue() {
        String result = getOption(Option.DEFAULT_TAG_VALUE.key);
        return result;
    }
    
    public void setDefaultTagValues(Map<String, String> tagValues) {
        StringBuilder sb = new StringBuilder(" ");
        for (Map.Entry<String, String> entry : tagValues.entrySet()) {
            sb.append(SEP).append('"').append(entry.getKey()).append("\":\"").append(entry.getValue()).append('"');
        }
        String value = sb.toString().trim();
        setOption(Option.DEFAULT_TAG_VALUE.key, value.isEmpty() ? "" : value.substring(1));
        saveForUser();
    }


    
    /**
     * @return le dictionnaire des tags par defaut d'une simulation a partir
     *         de la propriete  {@link Option#DEFAULT_TAG_VALUE}
     * @see Option#DEFAULT_TAG_VALUE
     */
    public Map<String, String> getDefaultTagValueAsMap() {
        Map<String, String> result = new HashMap<String, String>();
        String tagValuesList = getDefaultTagValue();
        if (tagValuesList != null) {
            String[] tagValues = StringUtil.split(tagValuesList, ",");
            for (String tagValue : tagValues) {
                String[] tagAndValue = StringUtil.split(tagValue, ":");

                String tag = tagAndValue[0].trim();
                tag = tag.substring(1, tag.length() - 1); // remove "..."

                String value = tagAndValue[1].trim();
                value = value.substring(1, value.length() - 1); // remove "..."

                result.put(tag, value);
            }
        }
        return result;
    }
    
    public String getJavadocURL() {
        String result = getOption(Option.JAVADOC_URL.key);
        return result;
    }

    public File getBackupDirectory() {
        File result = getOptionAsFile(Option.BACKUP_DIRECTORY.key);
        return result;
    }

    /**
     * Retourne le temps de départ de l'application en millisecondes.
     *
     * @return application start millis time
     */
    public long getStartingTime() {
        return startingTime;
    }
    
    /**
     * Retourne le temps écoulé depuis de debut de l'application (en secondes).
     * 
     * @return le temps écoulé en seconde sous forme de chaine
     */
    public String getElapsedTimeAsString() {
        long diff = System.currentTimeMillis() - getStartingTime();
        String result = DurationFormatUtils.formatDuration(diff, "s'.'S");
        return result;
    }
            
    public File getBackupSessionDirectory() {
        if (backupSessionDirectory == null) {
            long time = getStartingTime();
            // creation de l'unique répertoire de backup pour la session
            String path = new SimpleDateFormat("yyyy-MM-dd-HH-mm-ss").format(new java.util.Date(time));
            backupSessionDirectory = new File(getBackupDirectory(), path);
            if (!backupSessionDirectory.exists()) {
                backupSessionDirectory.mkdirs();
            }
        }
        return backupSessionDirectory;
    }

    /**
     * Retourne un class loader contenant le repertoire de compilation
     * Il permet alors de charger des classes qui viennent d'etre compilees
     * dans isis
     * If current thread is in simulation then return specific simulation
     * compilation directory, else default compilation directory
     *
     * @return the class loader adequate
     */
    public ClassLoader getScriptClassLoader() {
        SimulationContext simContext = SimulationContext.get();
        ClassLoader result = simContext.getClassLoader();
        if (result == null) {
            // on est pas dans une simulation, il faut retourner un nouveau
            // a chaque fois. on force la creation d'un nouveau classloader
            // a chaque fois pour
            // que l'ancienne class compiler et charg<E9> ne soit pas presente
            File f = getCompileDirectory();
            try {
                URL[] cp = new URL[]{f.toURI().toURL()};
                // il faut prendre le ClassLoader du thread courant comme parent
                // car pour les simulations il a ete modifi<E9>, et il faut
                // que les classes de script soit recherch<E9> dedans avant
                // la recherche dans le getCompileDirectory().
                // ce qui est le cas avec les URLClassLoader
                ClassLoader parent = Thread.currentThread().getContextClassLoader();
                result = new URLClassLoader(cp, parent);
            } catch (MalformedURLException eee) {
                throw new IsisFishRuntimeException(_("isisfish.error.load.classloader", f, eee.getMessage()), eee);
            }
        }
        return result;
    }

    //////////////////////////////////////////////////
    // Toutes les options disponibles
    //////////////////////////////////////////////////

    /**
     * Options.
     * 
     * Set it protected to force getOption() call.
     */
    public static enum Option implements OptionDef {

        CONFIG_FILE(CONFIG_FILE_NAME, n_("isisfish.config.main.configFileName.description"), CONFIG_FILENAME),
        ISIS_HOME_DIRECTORY("isis.home.directory", n_("isisfish.config.main.compileDirectory.description"), getUserHome() + File.separator + "isis-fish-" + majorVersion.toString()),
        COMPILATION_DIRECTORY("compilation.directory", n_("isisfish.config.main.compileDirectory.description"), "${isis.home.directory}" + File.separator + "isis-build"),
        MONITORING_DIRECTORY("monitoring.directory", n_("isisfish.config.main.monitoringDirectory.description"), "${isis.home.directory}" + File.separator + "isis-monitoring"),
        JAVADOC_DIRECTORY("javadoc.directory", n_("isisfish.config.main.javadocDirectory.description"), "${isis.home.directory}" + File.separator + "isis-docs"),
        BACKUP_DIRECTORY("backup.directory", n_("isisfish.config.main.defaultBackupDirectory.description"), "${isis.home.directory}" + File.separator + "isis-backup"),

        DEFAULT_EXPORT_NAMES("default.export.names", n_("isisfish.config.main.defaultExportNames.description"), ""),
        DEFAULT_RESULT_NAMES("default.result.names", n_("isisfish.config.main.defaultResultNames.description"), ""),
        DEFAULT_MAP_FILENAME("default.map.filename", n_("isisfish.config.main.defaultMapFile.description"), "maps/vmap_area_thin.shp"),
        DEFAULT_TAG_VALUE("default.tagvalue", n_("isisfish.config.main.defaultTagValue.description"), ""),
        ENCODING("encoding", n_("isisfish.config.main.encoding.description"), "UTF-8"),
        JAVADOC_URL("javadoc.url", n_("isisfish.config.main.javadocURL.description"), "http://www.isis-fish.org/apidocs/"),
        
        SIMULATOR_CLASSFILE("simulator.classfile", n_("isisfish.config.main.defaultSimulator.description"), "DefaultSimulator.java"),
        /** prevu pour l'architecture de lancement en plugin: local, isis-server, caparmor, ... */
        SIMULATOR_LAUNCHER(SimulationService.SIMULATION_LAUNCHER + ".3", n_("isisfish.config.main.localSimulator.description"), InProcessSimulatorLauncher.class.getName()),
        SIMULATOR_LAUNCHER2(SimulationService.SIMULATION_LAUNCHER + ".1", n_("isisfish.config.main.subSimulator.description"), SubProcessSimulationLauncher.class.getName()),
        SIMULATOR_LAUNCHER_REMOTE(SimulationService.SIMULATION_LAUNCHER + ".2", n_("isisfish.config.main.remoteCaparmor.description"), SSHSimulatorLauncher.class.getName()),

        /** Nombre maximum de thread de simulation in process. */
        SIMULATOR_IN_MAXTHREADS("simulation.in.max.threads", n_("isisfish.config.main.simulation.in.max.threads.description"), "1"),

        /** Nombre max de sous processus à lancer (-1 = pas de limite). */
        SIMULATOR_SUB_MAXPROCESS("simulation.sub.max.process", n_("isisfish.config.main.simulation.sub.max.process.description"), "-1"),
        /** Mémoire max (Xmx) allouée aux sous processus. */
        SIMULATOR_SUB_MAXMEMORY("simulation.sub.max.memory", n_("isisfish.config.main.simulation.sub.max.memory.description"), "1024M"),

        /** Serveur accessible par ssh : address */
        SIMULATOR_SSH_SERVER("simulation.ssh.server", n_("isisfish.config.main.simulation.ssh.server.description"), "caparmor-sftp"),
        /** Serveur accessible par ssh : login */
        SIMULATOR_SSH_USER_NAME("simulation.ssh.username", n_("isisfish.config.main.simulation.ssh.username.description"), ""),
        /** Serveur accessible par ssh : user home directory */
        SIMULATOR_SSH_USER_HOME("simulation.ssh.userhome", n_("isisfish.config.main.simulation.ssh.userhome.description"), ""),
        /** Serveur accessible par ssh : remote data path */
        SIMULATOR_SSH_DATAPATH("simulation.ssh.datapath", n_("isisfish.config.main.simulation.ssh.datapath.description"), "isis-fish-" + majorVersion.toString() + "/isis-database"),
        /** Serveur accessible par ssh : remote isis home install */
        SIMULATOR_SSH_ISIS_HOME("simulation.ssh.isis.home", n_("isisfish.config.main.simulation.ssh.isis.home.description"), "/home3/caparmor/poussin/isis-fish-" + IsisConfig.getApiVersion()),
        /** Serveur accessible par ssh : remote tmp path */
        SIMULATOR_SSH_TMPPATH("simulation.ssh.tmppath", n_("isisfish.config.main.simulation.ssh.tmppath.description"), "isis-tmp"),
        /** Serveur accessible par SSH : emplacement de Java (full path) */
        SIMULATOR_SSH_JAVAPATH("simulation.ssh.javapath64", n_("isisfish.config.main.simulation.ssh.javapath.description"), "/home3/caparmor/poussin/jdk64/bin/java"),
        /** Serveur accessible par SSH : emplacement des executables pbs */
        SIMULATOR_SSH_PBSBINPATH("simulation.ssh.pbsbinpath", n_("isisfish.config.main.simulation.ssh.pbsbinpath.description"), "/usr/pbs/bin"),
        /** Serveur accessible par SSH : option de l'executable qsub (defaut to -m n = no mail) */
        SIMULATOR_SSH_PBSQSUBOPTIONS("simulation.ssh.pbsqsuboptions", n_("isisfish.config.main.simulation.ssh.pbsqsuboptions.description"), "-m n"),
        /** Serveur accessible par SSH : interval de check du fichier de control */
        SIMULATOR_SSH_CONTROLCHECKINTERVAL("simulation.ssh.control.check.interval", n_("isisfish.config.main.simulation.ssh.control.check.interval.description"), "120"),
        /** Serveur accessible par SSH : nombre de thread au maximum a utilise simultanement */
        SIMULATOR_SSH_MAXTHREADS("simulation.ssh.max.threads", n_("isisfish.config.main.simulation.ssh.max.threads.description"), "1"),
        /** Serveur accessible par SSH : mémoire (Xmx) allouée pour les process java sur caparmor */
        SIMULATOR_SSH_MAXMEMORY("simulation.ssh.max.memory", n_("isisfish.config.main.simulation.ssh.max.memory.description"), "2000M"),

        /** Application locale (for i18n init). */
        LOCALE("locale", n_("isisfish.config.main.locale.description"), "fr_FR"),
        SIMULATION_SHOW_ONLY_ERROR("simulationShowOnlyError", n_("isisfish.config.main.simulationShowOnlyError.description"), "false"),

        /** Perform ui launch option. */
        LAUNCH_UI("launch.ui", n_("isisfish.config.main.launchUI.description"), "true"),
        /** Perform data migration option. */
        PERFORM_MIGRATION("perform.migration", n_("isisfish.config.main.performmigration.description"), "true"),
        /** Perform vcs update option. */
        PERFORM_VCS_UPDATE("perform.vcsupdate", n_("isisfish.config.main.performvcsupdate.description"), "true"),
        /** Perform cron option. */
        PERFORM_CRON("perform.cron", n_("isisfish.config.main.performcron.description"), "true"),
        /** Auto configure script parameter when addind it */
        SIMULATION_SCRIPT_AUTOCONFIG("script.autoconfig", n_("isisfish.config.main.script.autoconfig.description"), "true"),

        /** Database lock mode (h2).*/
        DATABASE_LOCK_MODE("database.lockmode", n_("isisfish.config.database.lockmode.description"), "file"),

        // SSH (global, for both VCS and caparmor)
        SSH_KEY_FILE("ssh.key.file", n_("isisfish.config.ssh.key.file.description"), getUserHome() + File.separator + ".ssh" + File.separator + "isis_rsa"),

        DATABASE_DIRECTORY("database.directory", n_("isisfish.config.vcs.localDatabasePath.description"), "${isis.home.directory}" + File.separator + "isis-database"),
        VCS_TYPE(VCS.VCS_TYPE, n_("isisfish.config.vcs.type.description"), VCS.TYPE_SVN),
        VCS_PROTOCOL(VCS.VCS_PROTOCOL, n_("isisfish.config.vcs.protocol.description"), "http"),
        VCS_USER_NAME(VCS.VCS_USER_NAME, n_("isisfish.config.vcs.userName.description"), ""),
        VCS_USER_PASSWORD(VCS.VCS_USER_PASSWORD, n_("isisfish.config.vcs.userPassword.description"), ""),
        VCS_HOST_NAME(VCS.VCS_HOST_NAME, n_("isisfish.config.vcs.hostName.description"), "svn.forge.codelutin.com"),
        VCS_PATH(VCS.VCS_PATH, n_("isisfish.config.vcs.remotePath.description"), "/svn/isis-fish-data"),

        // community vsc
        COMMUNITY_DATABASE_DIRECTORY("community.database.directory", n_("isisfish.config.vcs.localCommunityDatabasePath.description"), "${isis.home.directory}" + File.separator + "isis-community-database"),
        VCS_COMMUNITY_TYPE("vcs.community.type", n_("isisfish.config.vcs.community.type.description"), VCS.TYPE_SVN),
        VCS_COMMUNITY_PROTOCOL("vcs.community.protocole", n_("isisfish.config.vcs.community.protocol.description"), "http"),
        VCS_COMMUNITY_USER_NAME("vcs.community.username", n_("isisfish.config.vcs.community.username.description"), ""),
        VCS_COMMUNITY_PASSWORD("vcs.community.password", n_("isisfish.config.vcs.community.password.description"), ""),
        VCS_COMMUNITY_HOST_NAME("vcs.community.hostname", n_("isisfish.config.vcs.community.hostname.description"), "svn.forge.codelutin.com"),
        VCS_COMMUNITY_PATH("vcs.community.path", n_("isisfish.config.vcs.community.remotepath.description"), "/svn/isis-fish-community"),

        // misc options
        USER_NAME("user.name", n_("isisfish.config.main.userName.description"), System.getProperty("user.name")),
        SMTP_SERVER("smtpServer", n_("isisfish.config.main.smtpServer.description"), "smtp"),
        USER_MAIL("userMail", n_("isisfish.config.main.userMail.description"), USER_NAME.key + "@" + VCS_HOST_NAME.key);

        public String key;
        public String description;
        public String defaultValue;

        private Option(String key, String description, String defaultValue) {
            this.key = key;
            this.description = description;
            this.defaultValue = defaultValue;
        }

        /*
         * @see org.nuiton.util.ApplicationConfig.OptionDef#getDefaultValue()
         */
        @Override
        public String getDefaultValue() {
            return defaultValue;
        }

        /*
         * @see org.nuiton.util.ApplicationConfig.OptionDef#getDescription()
         */
        @Override
        public String getDescription() {
            return _(description);
        }

        /*
         * @see org.nuiton.util.ApplicationConfig.OptionDef#getKey()
         */
        @Override
        public String getKey() {
            return key;
        }

        /*
         * @see org.nuiton.util.ApplicationConfig.OptionDef#getType()
         */
        @Override
        public Class<?> getType() {
            return String.class;
        }

        /*
         * @see org.nuiton.util.ApplicationConfig.OptionDef#isFinal()
         */
        @Override
        public boolean isFinal() {
            return false;
        }

        /*
         * @see org.nuiton.util.ApplicationConfig.OptionDef#isTransient()
         */
        @Override
        public boolean isTransient() {
            return false;
        }

        /*
         * @see org.nuiton.util.ApplicationConfig.OptionDef#setDefaultValue(java.lang.String)
         */
        @Override
        public void setDefaultValue(String defaultValue) {
            this.defaultValue = defaultValue;
        }

        /*
         * @see org.nuiton.util.ApplicationConfig.OptionDef#setTransient(boolean)
         */
        @Override
        public void setTransient(boolean isTransient) {
            
        }

        /*
         * @see org.nuiton.util.ApplicationConfig.OptionDef#setFinal(boolean)
         */
        @Override
        public void setFinal(boolean isFinal) {
            
        }
    }
    
    //////////////////////////////////////////////////
    // Toutes les actions disponibles
    //////////////////////////////////////////////////
    
    public static enum Action {
        HELP(n_("Show help"), OtherAction.class.getName() + "#help", "-h", "--help"),
        VERSION(n_("Show version"), OtherAction.class.getName() + "#version", "-v", "--version"),

        IMPORT_EXPORT(n_(""), ImportAction.class.getName() + "#importExport", "--importExport"),
        IMPORT_RULE(n_(""), ImportAction.class.getName() + "#importRule", "--importRule"),
        IMPORT_SCRIPT(n_("Import one java file script source"), ImportAction.class.getName() + "#importScript", "--importScript"),
        IMPORT_SIMULATION_PLAN(n_(""), ImportAction.class.getName() + "#importSimulationPlan", "--importSimulationPlan"),
        IMPORT_SIMULATOR(n_(""), ImportAction.class.getName() + "#importSimulator", "--importSimulator"),
        IMPORT_FORMULA(n_(""), ImportAction.class.getName() + "#importFormula", "--importFormula"),
        IMPORT_REGION(n_(""), ImportAction.class.getName() + "#importRegion", "--importRegion"),
        IMPORT_REGION_AND_RENAME(n_(""), ImportAction.class.getName() + "#importRegionAndRename", "--importRegionAndRename"),
        IMPORT_SIMULATION(n_(""), ImportAction.class.getName() + "#importSimulation", "--importSimulation"),
        IMPORT_SCRIPT_MODULE(n_("Import zipped file containing all scripts directory structure"), ImportAction.class.getName() + "#importScriptModule", "--importScriptModule"),

        LIST_EXPORT(n_(""), ExportAction.class.getName() + "#listExport", "--listExport"),
        LIST_RULE(n_(""), ExportAction.class.getName() + "#listRule", "--listRule"),
        LIST_SCRIPT(n_(""), ExportAction.class.getName() + "#listScript", "--listScript"),
        LIST_SIMULATION_PLAN(n_(""), ExportAction.class.getName() + "#listSimulationPlan", "--listSimulationPlan"),
        LIST_SIMULATOR(n_(""), ExportAction.class.getName() + "#listSimulator", "--listSimulator"),
        LIST_FORMULA(n_(""), ExportAction.class.getName() + "#listFormula", "--listFormula"),
        LIST_REGION(n_(""), ExportAction.class.getName() + "#listRegion", "--listRegion"),
        LIST_SIMULATION(n_(""), ExportAction.class.getName() + "#listSimulation", "--listSimulation"),

        EXPORT_EXPORT(n_(""), ExportAction.class.getName() + "#exportExport", "--exportExport"),
        EXPORT_RULE(n_(""), ExportAction.class.getName() + "#exportRule", "--exportRule"),
        EXPORT_SCRIPT(n_(""), ExportAction.class.getName() + "#exportScript", "--exportScript"),
        EXPORT_SIMULATION_PLAN(n_(""), ExportAction.class.getName() + "#exportSimulationPlan", "--exportSimulationPlan"),
        EXPORT_SIMULATOR(n_(""), ExportAction.class.getName() + "#exportSimulator", "--exportSimulator"),
        EXPORT_FORMULA(n_(""), ExportAction.class.getName() + "#exportFormula", "--exportFormula"),
        EXPORT_REGION(n_(""), ExportAction.class.getName() + "#exportRegion", "--exportRegion"),
        EXPORT_SIMULATION(n_(""), ExportAction.class.getName() + "#exportSimulation", "--exportSimulation"),
                
        VCS_SSH_CREATE_KEY(n_(""), VCSAction.class.getName() + "#sshCreateKey", "--sshCreateKey"),
        VCS_UPDATE(n_(""), VCSAction.class.getName() + "#vcsUpdate", "--vcsUpdate"),
        VCS_ADD(n_(""), VCSAction.class.getName() + "#vcsAdd", "--vcsAdd"),
        VCS_REMOVE(n_(""), VCSAction.class.getName() + "#vcsRemove", "--vcsRemove"),
        VCS_COMMIT(n_(""), VCSAction.class.getName() + "#vcsCommit", "--vcsCommit"),

        SIMULATE_WITH_REGION(n_(""), SimulationAction.class.getName() + "#simulateWithRegion", "--simulateWithRegion"),
        SIMULATE_WITH_SIMULATION(n_(""), SimulationAction.class.getName() + "#simulateWithSimulation", "--simulateWithSimulation"),
        SIMULATE_WITH_SIMULATION_AND_SCRIPT(n_(""), SimulationAction.class.getName() + "#simulateWithSimulationAndScript", "--simulateWithSimulationAndScript"),
        SIMULATE_REMOTELLY(n_(""), SimulationAction.class.getName() + "#simulateRemotelly", "--simulateRemotelly"),
        SIMULATE_REMOTELLY_WITH_PRESCRIPT(n_(""), SimulationAction.class.getName() + "#simulateRemotellyWithPreScript", "--simulateRemotellyWithPreScript");
        
        protected String description;
        protected String action;
        protected String[] aliases;

        private Action(String description, String action, String ... aliases) {
            this.description = description;
            this.action = action;
            this.aliases = aliases;
        }
        
        public String getDescription() {
            return _(description);
        }
        
        public String getAction() {
            return action;
        }
        
        public String[] getAliases() {
            return aliases;
        }
    }
}
