/* *##%
 *  Copyright (C) 2002-2009 Ifremer, Code Lutin, Benjamin Poussin
 * 
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 *##%*/

package fr.ifremer.isisfish.util;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Stack;
import org.apache.commons.lang.time.DurationFormatUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Permet de tracer les appels aux methodes utilisateur ainsi que l'execution
 * a ces methodes. La difference entre les deux est lors de l'utilisation du
 * cache les appels seront superieur a l'execution car certaine valeur seront
 * reutilisé dans le cache.
 *
 * @author poussin
 * @version $Revision: 2599 $
 *
 * Last update: $Date: 2009-09-11 12:13:35 +0200 (ven., 11 sept. 2009) $
 * by : $Author: chatellier $
 */
public class Trace {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(Trace.class);

    /** nombre d'appel Cache inclus*/
    final static public int STAT_CALL = 0;
    /** nombre d'appel reel qui a fait le calcul*/
    final static public int STAT_COMPUTATION = 1;
    /** nombre d'appel vers une autre e depuis cette methode */
    final static public int STAT_CALL_NEST_METHOD = 2;
    /** temps mini d'execution de cette methode (sans le temps des autres methodes) */
    final static public int STAT_TIME_MIN = 3;
    /** temps max d'execution de cette methode (sans le temps des autres methodes) */
    final static public int STAT_TIME_MAX = 4;    
    /** temps total d'execution de cette methode (sans le temps des autres methodes) */
    final static public int STAT_TIME_TOTAL = 5;    
    /** temps total d'execution de cette methode (avec le temps des autres methodes) */
    final static public int STAT_TIME_TOTAL_NEST_METHOD = 6;    

    /** nombre d'appel vers une autre e depuis cette methode */
    final static private int STACK_CALL_NEST_METHOD = 0;    
    /** heure de depart de l'appel a la methode (sans le temps des autres methodes) */
    final static private int STACK_TIME_START = 1;    
    /** heure de depart de l'appel a la methode (avec le temps des autres methodes) */
    final static private int STACK_TIME_START_NEST_METHOD = 2;    
    
    
    protected String name = "";
    
    /** array : [call's numbers, call do, min time, max time, total time, total time with child]*/
    protected Map<String, long[]> statistics = new LinkedHashMap<String, long[]>();
    
    /** array : [nest e call, start time, start time with child] */
    protected Stack<long[]> callStack = new Stack<long[]>(); 
    
    public Trace(String name) {
        this.name = name;
    }
    
    public long[] getStatistics(Object e) {
        String key = String.valueOf(e);
        long [] result = statistics.get(key);
        if (result == null) {
            result = new long[]{0, 0, 0, 0, 0, 0, 0};
            statistics.put(key, result);
        }
        return result;
    }
    
    public void traceBefore () {
        // ajout dans le stack 
        long current = System.nanoTime();
        long [] stackItem = new long[]{0, current, current};
        callStack.push(stackItem);
    }
        
    public void traceAfterCall(Object e) {
        traceAfter(e, false);
    }
    public void traceAfterComputation(Object e) {
        traceAfter(e, true);
    }
     
    protected void traceAfter(Object e, boolean computation) {
        long current = System.nanoTime();
        
        if (callStack.isEmpty()) {
            log.warn("Empty stack in after for " + e);
        } else {
            long [] stackItem = callStack.pop();
            long timeSpent = current - stackItem[STACK_TIME_START];
            long timeSpentNestMethod = current - stackItem[STACK_TIME_START_NEST_METHOD];
            
            long [] stat = getStatistics(e);
            if (computation) {
                stat[STAT_COMPUTATION]++; // add +1 to computation number
            } else {
                // on incremente pas tout le temps STAT_CALL, car le plus
                // souvent lors de l'utilisation de l'objet Trace dans un cache
                // on a deja compte l'appel et reappelle lui meme computation
                // ce qui ferait 2 appel alors qu'il n'y en a qu'un en realite
                stat[STAT_CALL]++; // add +1 to call number
            }
            stat[STAT_CALL_NEST_METHOD] += stackItem[STACK_CALL_NEST_METHOD];
            stat[STAT_TIME_TOTAL] += timeSpent;
            stat[STAT_TIME_TOTAL_NEST_METHOD] += timeSpentNestMethod;
            if (stat[STAT_TIME_MIN] > timeSpent) {
                stat[STAT_TIME_MIN] = timeSpent;
            }
            if (stat[STAT_TIME_MAX] < timeSpent) {
                stat[STAT_TIME_MAX] = timeSpent;
            }
        
            if (!callStack.isEmpty()) {
                long [] parent = callStack.peek();
                parent[STACK_CALL_NEST_METHOD]++;  // add +1 to call number nest e
                parent[STACK_TIME_START] += timeSpentNestMethod; // remove to time all time spent in nest e (yes + to remove :)
            }
        }
        
        // ajouter le delta de temps dans le temps passé dans la méthod
        
        // il faud garder le temps passé dans l'appel d'autre methode de la stack
        // --> A
        //   =========
        //   --> B
        //   <-- B
        //   =========
        //   --> C
        //     --> D
        //     <-- D
        //   <-- C  
        //   =========
        // <-- A
        
        // le temps reellement passé dans A est representé par les =====
    }

    public String printStatisticAndClear() {
        long call = 0;
        long computation = 0;
        StringBuffer result = new StringBuffer();
        result.append("--- " + name + " Statistics ---\n");
        for (String e : statistics.keySet()) {
            long[] stat = getStatistics(e);

            // fix / by zero
            // if Cache aspect has not been selected
            if (stat[STAT_COMPUTATION] == 0) {
                stat[STAT_COMPUTATION] = stat[STAT_CALL];
            }

            long meanTime = stat[STAT_TIME_TOTAL] / stat[STAT_COMPUTATION];
            call += stat[STAT_CALL];
            computation += stat[STAT_COMPUTATION];
            result.append(
                    e +
                    " call: " + stat[STAT_CALL] +
                    " computation: " + stat[STAT_COMPUTATION] +
                    // usage du cache
                    "(" + (100*stat[STAT_COMPUTATION]/stat[STAT_CALL]) + "%)" +
                    // Time is in nano not millis, we must divide by 1000000
                    " min: " + DurationFormatUtils.formatDuration(stat[STAT_TIME_MIN] / 1000000, "s'.'S") +
                    " mean: " + DurationFormatUtils.formatDuration(meanTime / 1000000, "s'.'S") +
                    " max: " + DurationFormatUtils.formatDuration(stat[STAT_TIME_MAX] / 1000000, "s'.'S") +
                    " total: " + DurationFormatUtils.formatDuration(stat[STAT_TIME_TOTAL] / 1000000, "s'.'S") +
                    " call_nest: " + stat[STAT_CALL_NEST_METHOD] +
                    " total_with_nest: " + DurationFormatUtils.formatDuration(stat[STAT_TIME_TOTAL_NEST_METHOD] / 1000000, "s'.'S") +
                    "\n");
        }
        result.append("--------------------\n");
        result.append("Total call: " + call + "\n");
        result.append("Total computation: " + computation + "\n");
        result.append("Cache usage: " + (100 * (call-computation) / (call+1/*+1 pour la / par 0 */) ) + "%" + "\n");
        result.append("--------------------\n");
        callStack.clear();
        statistics.clear();
        
        System.out.println(result.toString());
        return result.toString();
    }

}
