/* *##%
 * Copyright (C) 2006 - 2009
 *     Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package fr.ifremer.isisfish.util;

import static org.nuiton.i18n.I18n._;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.FileUtil;

import fr.ifremer.isisfish.IsisFishRuntimeException;
import fr.ifremer.isisfish.IsisFish;


/**
 * Permet d'evaluer les equations ecritent en Java
 * 
 * Created: 3 juil. 2006 23:44:48
 *
 * @author poussin
 * @version $Revision: 2690 $
 *
 * Last update: $Date: 2009-10-26 15:56:42 +0100 (lun., 26 oct. 2009) $
 * by : $Author: chatellier $
 */
public class EvaluatorHelper {

    /** Logger for this class. */
    private static Log log = LogFactory.getLog(EvaluatorHelper.class);

    protected static String normalizeClassName(String name) {
        StringBuffer result = new StringBuffer(name);        
        for (int i=0; i<result.length(); i++) {
            char c = result.charAt(i);
            if (!Character.isJavaIdentifierPart(c)) {
                result.setCharAt(i, '_');
            }
        }
        return result.toString();
    }

    /**
     * Verifie si une equation est syntaxiquement correcte.
     * 
     * @param javaInterface
     * @param script
     * @param out output writer (can be null for non output)
     * @return 0 si ok
     */
    public static int check(Class javaInterface, String script, PrintWriter out) {
        try {
            File src = File.createTempFile("check", "equation");
            src.deleteOnExit();
            String packageName = null;
            String className = normalizeClassName(src.getName());
            
            src = new File(src.getParentFile(), className + ".java");
            src.deleteOnExit();
            
            // recherche la methode de l'interface
            Method [] methods = javaInterface.getDeclaredMethods();
            Method interfaceMethod = methods[0];
            
            String content = generateContent(packageName, className, interfaceMethod, script);
            
            FileUtil.writeString(src, content);

            int compileResult = CompileHelper.compile(src.getParentFile(), src, src.getParentFile(), out);
            File dest = new File(src.getParentFile(), className + ".class");
            dest.deleteOnExit();

            return compileResult;

        } catch (Exception eee) {
            log.warn("Can't check equation", eee);
            return -10000;
        }
    }

    /**
     * Evalue une equation.
     * 
     * @param packageName le nom de package de la classe
     * @param className le nom de la classe
     * @param javaInterface l'interface que la classe doit etendre,
     *        cette interface n'a qu'un methode
     * @param script le code de la methode
     * @param args les arguments a utiliser pour l'appel de la methode
     * @return la valeur retourné par la methode
     */
    public static Object evaluate(String packageName, String className,
            Class javaInterface, String script, Map<String, Object> args) {
        className = normalizeClassName(className);

        Object result = null;
        Class clazz = null;

        // recherche la methode de l'interface
        Method [] methods = javaInterface.getDeclaredMethods();
        Method interfaceMethod = methods[0];

        String classname = packageName + "." + className;

        File fileRootSrc = IsisFish.config.getCompileDirectory();
        File fileCheckSum = new File(fileRootSrc, packageName + File.separator + className + ".hashCode");
        File fileSrc = new File(fileRootSrc, packageName + File.separator + className + ".java");
        File fileDest = new File(fileRootSrc, packageName + File.separator + className + ".class");

        boolean checkSumEquals = false;

        // if equation's Java file exists, check the checksum 
        if (fileSrc.exists() && fileCheckSum.exists()) {
            String oldCheckSum = "";
            try {
                oldCheckSum = FileUtil.readAsString(fileCheckSum);
            } catch (IOException eee) {
                log.info("Can't read old checkSum:  " + fileCheckSum, eee);
            }
            String newCheckSum = Integer.toString(script.hashCode());
            checkSumEquals = newCheckSum.equals(oldCheckSum);
        }

        // if Java file's checkSum is not equals to script's checkSum
        // generate new Java file
        if (!checkSumEquals) {
            String content = generateContent(packageName, className, interfaceMethod, script);
            
            try {
                // force writing to UTF-8
                // fix compilation issue : unmappable characters
                FileUtil.writeString(fileSrc, content, "utf-8");
            } catch (IOException zzz) {
                throw new IsisFishRuntimeException(_("isisfish.error.save.script.compilation", fileSrc), zzz);
            }
            try {
                FileUtil.writeString(fileCheckSum, Integer.toString(script.hashCode()));
            } catch (IOException zzz) {
                throw new IsisFishRuntimeException(_("isisfish.error.save.checkSum.compilation", fileSrc), zzz);
            }
        }

        // if Java file is newer than class file, compile java file
        if (FileUtil.isNewer(fileSrc, fileDest)) {
            try {
                // does'nt contains isisdatabase-3 directory
                //int compileResult = CompileHelper.compile(fileRootSrc, fileSrc, fileRootSrc, null);

                List<File> classpath = new ArrayList<File>();
                classpath.add(fileRootSrc.getAbsoluteFile());
                classpath.add(IsisFish.config.getDatabaseDirectory().getAbsoluteFile());
                int compileResult = CompileHelper.compile(classpath, Collections.singletonList(fileSrc), fileRootSrc, null);

                if (compileResult != 0) {
                    throw new IsisFishRuntimeException(_("isisfish.error.compile.script", compileResult, fileSrc));
                }
            } catch (Exception zzz) {
                throw new IsisFishRuntimeException(_("isisfish.error.compile.script", fileSrc), zzz);
            }
        }

        // try to load class
        try {
            ClassLoader cl = IsisFish.config.getScriptClassLoader();
            clazz = cl.loadClass(classname);                
        } catch (Exception zzz) {
            throw new IsisFishRuntimeException(_("isisfish.error.load.class", classname), zzz);
        }

        result = invoke(clazz, interfaceMethod, args);

        return result;
    }

    /**
     * Generate equation content.
     * 
     * Warning, content are always on a unique single line (without \n) for
     * debuging purpose.
     * 
     * @param packageName
     * @param className
     * @param interfaceMethod
     * @param script
     * @return equation return
     */
    protected static String generateContent(String packageName, String className, Method interfaceMethod, String script) {
        String content = "";
        if (packageName != null && !"".equals(packageName)) {
            content += "package " + packageName + ";";
        }
        content += "import fr.ifremer.isisfish.entities.*;";
        content += "import org.nuiton.math.matrix.*;";
        content += "import fr.ifremer.isisfish.types.Date;";
        content += "import fr.ifremer.isisfish.types.Month;";
        content += "import fr.ifremer.isisfish.types.RangeOfValues;";
        content += "import fr.ifremer.isisfish.types.TimeUnit;";
        content += "import org.apache.commons.logging.Log;";
        content += "import org.apache.commons.logging.LogFactory;";
        content += "import java.util.*;";
        content += "public class " + className + " implements " + interfaceMethod.getDeclaringClass().getName() + " {";
        content += "private static Log log = LogFactory.getLog(" + className + ".class);";
        content += "public " + interfaceMethod.getReturnType().getName() +  " " + interfaceMethod.getName() + "(";

        Args args = interfaceMethod.getAnnotation(Args.class);
        String [] names = args.value();
        
        String [] stringTypes;
        ArgTypes argTypes = interfaceMethod.getAnnotation(ArgTypes.class);
        if (argTypes != null) {
            stringTypes = argTypes.value();
        } else {
            stringTypes = new String[names.length];
            Class [] types = interfaceMethod.getParameterTypes();
            for (int i=0; i<types.length; i++) {
                stringTypes[i] = types[i].getName();
            }
        }

        for (int i=0; i<names.length; i++) {
            content += stringTypes[i] + " " + names[i];
            if (i+1<names.length) {
                content += ", ";
            }
        }
        content += ") throws Exception {";
        content += script;
        content += "\n}\n}\n";

        return content;
    }

    protected static Object invoke(Class clazz, Method interfaceMethod, Map<String, Object> args) {
        try {
            Method method = clazz.getDeclaredMethod(interfaceMethod.getName(),
                    interfaceMethod.getParameterTypes());
            
            List<Object> params = new ArrayList<Object>();
            for (String name : interfaceMethod.getAnnotation(Args.class).value()) {
                Object arg = args.get(name);
                params.add(arg);
            }
            Object eq = clazz.newInstance();
            Object result;
            result = method.invoke(eq, params.toArray());
            return result;
        } catch (Exception eee) {
            throw new IsisFishRuntimeException(_("isisfish.error.invoke.method", interfaceMethod, clazz.getName()), eee);
        }
    }
}


