package fr.ifremer.isisfish.ui.simulator.filter;

import fr.ifremer.isisfish.simulator.SimulationMeta;
import fr.ifremer.isisfish.simulator.SimulationProperties;
import fr.ifremer.isisfish.ui.widget.filter.FilterModel;
import fr.ifremer.isisfish.ui.widget.filter.FilterParamModel;

import static org.nuiton.i18n.I18n._;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.IOException;
import java.text.ParseException;
import java.util.ArrayList;
import static java.util.Collections.emptyList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

/**
 * This class offers methods to select or filter simulation name via an ui
 *
 * @see #selectSimulation(List)
 * @see #selectSimulation(FilterModel)
 * @see #filterSimulation(List)
 * @see #filterSimulation(FilterModel)
 */
public class SimulationFilterUtil {

    static private Log log = LogFactory.getLog(SimulationFilterUtil.class);

    /**
     * Open the filter simulation filter dialog using list of simualtion names
     * to create FilterModel, and waits for user to select a simualtion in a non
     * empty filter result.
     * <p/>
     * If user cancels, it returns <code>null</code>
     *
     * @param names the list of simulation names to used
     * @return name of the simulation choosed by user or null if it cancel
     *         operation
     * @throws java.text.ParseException if problem while init dialogModel
     * @throws java.io.IOException todo
     */
    public static String selectSimulation(List<String> names)
            throws ParseException, IOException {

        // no simulation found : so nothing to be done
        if (names.isEmpty()) {
            return null;
        }
        FilterModel<SimulationProperties, String> model = createFilterModel(names);
        return selectSimulation(model);
    }

    /**
     * Open the filter simulation filter dialog using directly a given
     * FilterModel, and waits for user to select a simualtion in a non
     * empty filter result.
     *
     * @param model model to used
     * @return name of the simulation choosed by user or null if it cancel
     *         operation
     * @throws java.text.ParseException if problem while init dialogModel
     * @throws java.io.IOException todo
     */
    public static String selectSimulation(FilterModel<SimulationProperties, String> model)
            throws ParseException, IOException {

        // no simulation found : so nothing to be done
        if (model == null) {
            return null;
        }
        SimulationFilterDialog ui = new SimulationFilterDialog(model,
                _("isisfish.filter.select.simulation"), _("isisfish.filter.select"));
        log.info("ui used " + ui.getClass());

        // obtain exactly one simulation name or null if none taken
        String result = ui.obtainSelectResult();
        log.info("simulation selected " + result);
        return result;
    }

    /**
     * Open the filter simulation dialog and waits for user to accept a
     * non empty filter result.
     * It used a list of simulation names to create FilterModel.
     * <p/>
     * If user cancels, it returns an empty list.
     *
     * @param names the list of simulation names to used
     * @return list of simulation's names selected by user
     * @throws java.text.ParseException if problem while init model
     * @throws java.io.IOException todo
     */
    public static List<String> filterSimulation(List<String> names)
            throws ParseException, IOException {

        // no simulation found : so nothing to be done        
        if (names.isEmpty()) return emptyList();

        FilterModel<SimulationProperties, String> model = createFilterModel(names);
        return filterSimulation(model);
    }

    /**
     * Open the filter simulation dialog and waits for user to accept a
     * non empty filter result.
     * It used directly the given FilterModel.
     * <p/>
     * If user cancels, it returns the previous result list.
     *
     * @param model filterModel to use
     * @return list of simulation's names selected by user
     * @throws java.text.ParseException if problem while init model
     * @throws java.io.IOException todo
     */
    @SuppressWarnings({"unchecked"})
    public static List<String> filterSimulation(FilterModel<SimulationProperties, String> model)
            throws ParseException, IOException {

        // no simulation found : so nothing to be done
        if (model == null) return emptyList();
        // keep old result list
        List<SimulationProperties> oldResult = new ArrayList<SimulationProperties>(model.getFiltered());

        SimulationFilterDialog ui = new SimulationFilterDialog(model,_("isisfish.filter.simulation"),_("isisfish.filter.apply"));        
        log.info("ui used " + ui.getClass());

        // obtain list of filtered simulation names
        List<String> result = ui.obtainFilterResult();

        if (result.isEmpty()) {
            model.setFiltered(oldResult);
            result = model.getFilteredResult();
            log.info("simulation filtered no result, push back old result " + result.size());
        } else {
            log.info("simulation filtered " + result.size());
        }
        return result;
    }

    /**
     * Create a new filter dialogModel for simulation
     *
     * @param names list of simulation's names to use
     * @return the filter dialogModel
     * @throws ParseException if any problem while init dialogModel
     * @throws java.io.IOException todo
     */
    public static FilterModel<SimulationProperties, String> createFilterModel(
            List<String> names) throws ParseException, IOException {
        // compute filter model
        List<SimulationProperties> original = new ArrayList<SimulationProperties>();
        // create filter model datas (says here SimulationProperties for each
        // simulation found)
        for (String name : names) {
            // get associated properties
            SimulationProperties properties = new SimulationProperties(name);
            original.add(properties);
        }
        // build FilterParamModels
        Map<String, FilterParamModel<SimulationProperties, ?>> paramModels = new TreeMap<String, FilterParamModel<SimulationProperties, ?>>();
        for (SimulationMeta meta : SimulationMeta.values()) {
            paramModels.put(meta.name(), meta.createParamModel());
        }
        // instanciate a new FilterModel
        return new FilterModel<SimulationProperties, String>(original, paramModels) {
            /**
             * @param item simulation properties to use
             * @return the result name for the given simulation properties
             * (here the simulation name)
             */
            public String convertToResult(SimulationProperties item) {
                return item.getName();
            }
        };
    }

    protected SimulationFilterUtil() {
        // don't instianciate me
    }


}

