/* *##%
 * Copyright (C) 2005 - 2010
 *     Ifremer, Code Lutin, Cedric Pineau, Benjamin Poussin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package fr.ifremer.isisfish.ui.simulator;

import static org.nuiton.i18n.I18n._;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.Serializable;
import java.lang.reflect.Field;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.SortedMap;
import java.util.TreeMap;

import javax.swing.JComponent;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.math.matrix.MatrixND;
import org.nuiton.topia.TopiaContext;

import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.IsisFishException;
import fr.ifremer.isisfish.datastore.AnalysePlanStorage;
import fr.ifremer.isisfish.datastore.ExportStorage;
import fr.ifremer.isisfish.datastore.RegionStorage;
import fr.ifremer.isisfish.datastore.RuleStorage;
import fr.ifremer.isisfish.datastore.ScriptStorage;
import fr.ifremer.isisfish.datastore.SensitivityExportStorage;
import fr.ifremer.isisfish.datastore.SensitivityStorage;
import fr.ifremer.isisfish.datastore.SimulationStorage;
import fr.ifremer.isisfish.datastore.SimulatorStorage;
import fr.ifremer.isisfish.entities.Population;
import fr.ifremer.isisfish.entities.Species;
import fr.ifremer.isisfish.entities.Strategy;
import fr.ifremer.isisfish.export.Export;
import fr.ifremer.isisfish.export.SensitivityExport;
import fr.ifremer.isisfish.mexico.MexicoHelper;
import fr.ifremer.isisfish.rule.Rule;
import fr.ifremer.isisfish.simulator.AnalysePlan;
import fr.ifremer.isisfish.simulator.SimulationParameter;
import fr.ifremer.isisfish.simulator.SimulationParameterImpl;
import fr.ifremer.isisfish.simulator.launcher.SimulationService;
import fr.ifremer.isisfish.simulator.launcher.SimulatorLauncher;
import fr.ifremer.isisfish.simulator.sensitivity.DesignPlan;
import fr.ifremer.isisfish.simulator.sensitivity.Factor;
import fr.ifremer.isisfish.simulator.sensitivity.SensitivityCalculator;
import fr.ifremer.isisfish.simulator.sensitivity.SensitivityException;
import fr.ifremer.isisfish.simulator.sensitivity.domain.ContinuousDomain;
import fr.ifremer.isisfish.simulator.sensitivity.domain.DiscreteDomain;
import fr.ifremer.isisfish.simulator.sensitivity.domain.EquationContinuousDomain;
import fr.ifremer.isisfish.simulator.sensitivity.domain.MatrixContinuousDomain;
import fr.ifremer.isisfish.ui.util.ErrorHelper;

/**
 * SimulAction.
 *
 * Created: 1 aout 2005 18:37:25 CEST
 *
 * @author Benjamin POUSSIN <poussin@codelutin.com>
 * @version $Revision: 3034 $
 *
 * Last update: $Date: 2010-04-29 18:14:25 +0200 (jeu., 29 avril 2010) $
 * by : $Author: chatellier $
 */
public class SimulAction {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static Log log = LogFactory.getLog(SimulAction.class);

    private static final SimpleDateFormat DATEFORMAT = new SimpleDateFormat("yyyy-MM-dd-HH-mm");
    
    protected SimulationParameter param = null;
    protected RegionStorage regionStorage = null;
    protected SimulationStorage simulStorage = null;
    
    protected List<String> oldSimulNames = null;
    protected String simulName = null;
    protected SensitivityStorage sensitivityStorage = null;
    protected Map<String, Factor> factors;
    protected Map<String, JComponent> factorComponent = new HashMap<String, JComponent>();
    protected DesignPlan designPlan = new DesignPlan();

    /*
     ************
     * Init
     ************
     */

    public SimulAction() {
        init();
    }

    /**
     * cree et initialise param avec les valeurs par defaut de l'utilisateur
     */
    public void init() {
        if (log.isDebugEnabled()) {
            log.debug("Init ");
        }
        try {
            param = new SimulationParameterImpl();
            List<String> resultNames = getResultNames();
            // put default value in param
            param.setSimulatorName(IsisFish.config.getSimulatorClassfile());
            param.setTagValue(IsisFish.config.getDefaultTagValueAsMap());

            param.setExportNames(IsisFish.config.getDefaultExportNamesAsList());
            List<String> defaultResultNames = IsisFish.config
                    .getDefaultResultNamesAsList();
            if (defaultResultNames != null) {
                param.setResultEnabled(defaultResultNames);
            } else {
                param.setResultEnabled(resultNames);
            }
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't init SimulationParameter", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.simulation.initsimulaction"), eee);
        }
        
        // factors has been requested to be sorted
        // TODO add factor sorting
        factors = new HashMap<String, Factor>();
    }

    protected void setName(String name) {
        simulName = name;
    }

    /**
     * Load simulation parameter file.
     * 
     * @param f
     * 
     * TODO public just for sensitivity
     */
    public void importSimulation(File f) {
        FileInputStream fos = null;
        try {
            fos = new FileInputStream(f);
            Properties proper = new Properties();
            proper.load(fos);
            param.fromProperties(proper);
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't import simulation", e);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.simulation.importparameter"), e);
        } finally {
            try {
                fos.close();
            } catch (IOException e) {
                if (log.isErrorEnabled()) {
                    log.error("Can't import simulation", e);
                }
                ErrorHelper.showErrorDialog(_("isisfish.error.simulation.importparameter"), e);
            }
        }
    }

    /**
     * Save current simulation to given file.
     * 
     * @param f file to save simulation to
     * 
     * TODO public just for sensitivity
     */
    public void saveSimulation(File f) {
        FileOutputStream fos = null;
        try {
            fos = new FileOutputStream(f);
            param.toProperties().store(fos, simulName);
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't save simulation", e);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.simulation.savesimulation"), e);
        } finally {
            try {
                if (fos != null) {
                    fos.close();
                }
            } catch (IOException e) {
                if (log.isErrorEnabled()) {
                    log.error("Can't save simulation", e);
                }
                ErrorHelper.showErrorDialog(_("isisfish.error.simulation.savesimulation"), e);
            }
        }
    }

    /**
     * Looking for ResultName.java script and read all fields and fill
     * list of field value and doc
     * @return the list of result names found in ResultName script
     */
    protected List<String> getResultNames() {
        List<String> result = new ArrayList<String>();
        try {
            ScriptStorage storage = ScriptStorage.getScript("ResultName.java");
            Class<?> resultNameClass = storage.getCodeClass();
            // TODO: if we can't find class, perhaps try to parser Java source ?
            Field[] fields = resultNameClass.getFields();
            for (Field f : fields) {
                String fieldName = (String) f.get(null);
                result.add(fieldName);
            }
        } catch (Exception eee) {
            if (log.isWarnEnabled()) {
                log.warn("Can't find result name script", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.simulation.getresultname"), eee);
        }
        return result;
    }

    /**
     * Change region in simulation launcher
     *
     * @param regionName region name
     */
    public void regionChange(String regionName) {
        try {
            regionStorage = RegionStorage.getRegion(regionName);
            
            // chatellier, on ne peut pas le reinitialiser, on
            // perd toutes les info apres un rechergement d'une anciennes simulation
            //init(); // reinitialise param pour le vider
            param.setRegionName(regionName);

            // poussin 20090519 quel est l'interet de faire ca ? et encore plus maintenant qu'on reinitialise
            // for (Rule r : param.getRules()) {
            //     rules.put(r, RuleStorage.getName(r));
            // }
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't change region", e);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.simulation.openregion"), e);
        }
    }

    /**
     * Load old simulation.
     * 
     * Reset some field to empty default values:
     * <ul>
     *  <li>params</li>
     *  <li>analyse plans</li>
     *  <li>factors list</li>
     * <ul>
     * 
     * Open old simulation:
     * <ul>
     *  <li>params copy</li>
     *  <li>factors</li>
     * </ul>
     * 
     * @param simulName name of simulation to load
     */
    public void loadOldSimulation(String simulName) {
        if (log.isDebugEnabled()) {
            log.debug("call loadOldSimulation: " + simulName);
        }
        try {
            this.simulName = simulName;
            simulStorage = SimulationStorage.getSimulation(simulName);
            param = simulStorage.getParameter().copy();
            // all time reset number after load
            param.setAnalysePlanNumber(-1);
            regionStorage = param.getRegion();

            // Chargement des exports de sensibilites
            /* not used anymore
             * sensitivityExports.clear();
            for (SensitivityExport ex : param.getSensitivityExport()){
                sensitivityExports.put(ex, ex.getExportFilename());
            }*/

            // Chargement des facteurs
            // clear list even if mexico file doesn't exists
            factors.clear();
            File f = SimulationStorage.getMexicoDesignPlan(SimulationStorage.getSimulationDirectory(simulName));
            if (f != null && f.canRead()) {
                if (log.isInfoEnabled()) {
                    log.info("Import design plan from : " + f.getAbsolutePath());
                }
                TopiaContext topiaContext = regionStorage.getStorage();
                designPlan = MexicoHelper.getDesignPlanFromXML(f, topiaContext);
                for (Factor factor : designPlan.getFactors()) {
                    if (log.isDebugEnabled()) {
                        log.debug("Find factor : " + factor.getName());
                    }
                    factors.put(factor.getPath() + factor.getName(), factor);
                }
            }
            else {
                if (log.isInfoEnabled()) {
                    log.info("No xml design plan file found");
                }
            }
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't load old Simulation: " + simulName, eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.simulation.loadoldsimulation"), eee);
        }
    }

    /*
     ************
     * ParamsUI
     ************
     */

    /**
     * Get simulation parameters rule list.
     * 
     * @return rules
     */
    public List<Rule> getRules() {
        return param.getRules();
    }
    
    /**
     * Add new rule.
     * 
     * Get a new instance of ruleItem, and add it to rule list.
     * 
     * @param ruleName rule to get copy
     */
    public void addNewRule(String ruleName) {
        try {
            RuleStorage ruleStorage = RuleStorage.getRule(ruleName);
            Rule ruleTmp = ruleStorage.getNewRuleInstance();
            param.addRule(ruleTmp);
        } catch (IsisFishException e) {
            if (log.isErrorEnabled()) {
                log.error("Can't add rule", e);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.simulation.addrule"), e);
        }
    }

    /**
     * Retire une regle.
     * 
     * @param rule rule to removes
     * @return <tt>true</tt> if this list contained the specified element
     */
    public boolean removeRule(Rule rule) {
        return param.removeRule(rule);
    }

    /**
     * Retourne une liste d'instance de toutes les regles disponible.
     * 
     * Instancié (utile pour les modeles).
     * 
     * @return instance rule list
     */
    public List<String> getAvailableRuleNames() {
        List<String> result = new ArrayList<String>();
        for (String ruleName : RuleStorage.getRuleNames()) {
            // remove last .java
            String shortRuleName = ruleName.substring(0, ruleName.length() - 5);
            result.add(shortRuleName);
        }
        return result;
    }

    // Strategies

    public List<Strategy> getStrategies() {
        List<Strategy> result = new ArrayList<Strategy>();
        try {
            TopiaContext tx = param.getRegion().getStorage().beginTransaction();
            result = RegionStorage.getFisheryRegion(tx).getStrategy();
            tx.rollbackTransaction();
            tx.closeContext();
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't get strategies", e);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.simulation.liststrategies"), e);
        }
        return result;
    }

    public void setStrategies(Object[] strategies) {
        List<Strategy> result = new ArrayList<Strategy>();
        for (Object o : strategies) {
            result.add((Strategy) o);
        }
        param.setStrategies(result);
    }

    // Population

    public List<Population> getPopulations() {

        List<Population> result = new ArrayList<Population>();
        try {
            TopiaContext tx = param.getRegion().getStorage().beginTransaction();
            List<Species> species = RegionStorage.getFisheryRegion(tx)
                    .getSpecies();
            for (Species s : species) {
                Collection<Population> populations = s.getPopulation();

                // FIXME initialiaze lazy hibernate collection
                for (Population p : populations) {
                    p.getPopulationGroup().size();
                    p.getPopulationZone().size();
                }

                result.addAll(populations);
            }
            tx.rollbackTransaction();
            tx.closeContext();
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't get population", e);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.simulation.listpopulation"), e);
        }
        return result;
    }

    public void setPopulations(Object[] populations) {
        List<Population> result = new ArrayList<Population>();
        for (Object o : populations) {
            result.add((Population) o);
        }
        param.setPopulations(result);
    }

    // Years

    public int getNumberOfYear() {
        return param.getNumberOfYear();
    }
    
    public void setNumberOfYear(String years) {
        param.setNumberOfYear(Integer.parseInt(years));
    }

    public List<String> getSimulatorNames() {
        return SimulatorStorage.getSimulatorNames();
    }

    /**
     * Return old simulations.
     *
     * @return old simulations
     */
    public List<String> getOldSimulationItem() {

        //try {
            oldSimulNames = new ArrayList<String>();

            //FilterModel<SimulationProperties, String> filterModel;
            // keep in context list of old simulation names (for filter process)
            oldSimulNames.addAll(SimulationStorage.getSimulationNames());
            // create filter model
            //filterModel = SimulationFilterUtil.createFilterModel(oldSimulNames);
            // to used directly model.getFilteredResult() in xml
            // we must fill filterModel result with original items
            //filterModel.selectAll();

            // chatellier : Ajout d'une entrée vide pour forcer la sélection
            // et que le changement lance un evenement sur la liste
            // Add empty item after, otherwise, un " " directory
            // will be created
            oldSimulNames.add(0, " ");

            return oldSimulNames;
        /*} catch (ParseException e) {
            if (log.isErrorEnabled()) {
                log.error("Can't get old simulation item", e);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.simulation.listoldsimulation"), e);
        } catch (IOException e) {
            if (log.isErrorEnabled()) {
                log.error("Can't get old simulation item", e);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.simulation.listoldsimulation"), e);
        }
        return null;*/
    }

    public List<String> getFilteredOldSimulatorNames(boolean force) {
        if (oldSimulNames == null || force) {
            oldSimulNames = getOldSimulationItem();
        }
        return oldSimulNames;
    }

    public void setOldSimulatorNames(List<String> sn) {
        oldSimulNames = sn;
    }

    public void resetOldSimulatorNames() {
        oldSimulNames = getOldSimulationItem();
    }

    /*
     ************
     * AdvancedParameterUI
     ************
     */

    // Gestion des TagValues
    public void addTagValue(String tag, String value) {
        param.getTagValue().put(tag, value);
    }

    public void removeTagValue(String tag) {
        log.debug("removeTagValue: " + tag);
        param.getTagValue().remove(tag);
    }

    public void saveTagValue(String simulatorName) {
        Map<String, String> tagValues = param.getTagValue();
        log.debug("call saveTagValue: " + tagValues);
        IsisFish.config.setDefaultTagValues(tagValues);

        IsisFish.config.setSimulatorClassfile(simulatorName);
    }

    /*
     * ExportUI
     */

    public List<String> getExportNames() {
        List<String> result = new ArrayList<String>();
        List<String> exportNames = ExportStorage.getExportNames();
        for (String export : exportNames) {
            result.add(export);
        }
        return result;
    }

    /**
     * Save current parameters exports names in configuration.
     */
    public void saveExport() {

        if (param.getExportNames() != null) {
            
            if (log.isDebugEnabled()) {
                log.debug("Set exports in configuration : " + param.getExportNames());
            }
            
            IsisFish.config.setDefaultExportNames(param.getExportNames());
        }
    }

    /*
     * ResultUI
     */

    public void saveParametersResultNames(Object[] resultNames) {
        List<String> resultNamesString = new ArrayList<String>();
        for (Object o : resultNames) {
            resultNamesString.add(o.toString());
        }
        param.setResultEnabled(resultNamesString);
        
        if (log.isDebugEnabled()) {
            log.debug("Set simulation result names : " + resultNamesString);
        }
    }

    public void saveConfigurationResultNames(Object[] resultNames) {
        List<String> resultNamesString = new ArrayList<String>();
        for (Object o : resultNames) {
            resultNamesString.add(o.toString());
        }
        IsisFish.config.setDefaultResultNames(resultNamesString);
        
        if (log.isDebugEnabled()) {
            log.debug("Set configuration result names : " + resultNamesString);
        }
    }

    public List<String> getDefaultResultNames() {
        return IsisFish.config.getDefaultResultNamesAsList();
    }

    /**
     * Return Analyse plan names list.
     * 
     * @return Analyse plan names
     */
    public List<String> getAnalysePlanNames() {
        return AnalysePlanStorage.getAnalysePlanNames();
    }

    public List<AnalysePlan> getParamAnalysePlans() {
        return param.getAnalysePlans();
    }

    public void addAnalysePlan(String name) {
        try {
            AnalysePlan ap = AnalysePlanStorage.getAnalysePlan(name)
                    .getNewAnalysePlanInstance();
            getSimulationParameter().addAnalysePlan(ap);
        } catch (IsisFishException e) {
            if (log.isErrorEnabled()) {
                log.error("Can't add analyse plan", e);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.simulation.addanalyseplan"), e);
        }
    }

    public void removeAnalysePlan(AnalysePlan ap) {
        getSimulationParameter().removeAnalysePlan(ap);
    }

    public void clearAnalysePlans() {
        getSimulationParameter().clearPlans();
    }

    public Map<String, Class<?>> getAnalysePlanParameterName(AnalysePlan ap) {
        Map<String, Class<?>> result = null;
        if (ap != null) {
            result = AnalysePlanStorage.getParameterNames(ap);
        }
        return result;
    }

    public Object getAnalysePlanParameterValue(String paramName, AnalysePlan ap) {
        Object result = null;
        if (ap != null) {
            try {
                result = AnalysePlanStorage.getParameterValue(ap, paramName);
            } catch (IsisFishException e) {
                if (log.isErrorEnabled()) {
                    log.error("Can't get analyse plan param value", e);
                }
                ErrorHelper.showErrorDialog(_("isisfish.error.simulation.getparametervalue"), e);
            }
        }
        return result;
    }

    public void setAnalysePlanParameterValue(String paramName, AnalysePlan ap,
            Object value) {
        if (log.isDebugEnabled()) {
            log.debug("paramName : " + paramName + " analysePlanName : " + ap
                    + " value : " + value);
        }
        try {
            AnalysePlanStorage.setParameterValue(ap, paramName, value);
        } catch (IsisFishException e) {
            if (log.isErrorEnabled()) {
                log.error("Can't set analyse plan param value", e);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.simulation.setparametervalue"), e);
        }
    }

    /*
     * Sensitivity
     */
    public List<String> getSensitivityExportNames() {
        List<String> result = new ArrayList<String>();
        List<String> exportNames = SensitivityExportStorage.getSensitivityExportNames();
        for (String export : exportNames) {
            result.add(export);
        }
        return result;
    }

    public List<SensitivityExport> getSensitivityExports() {
        List<SensitivityExport> result = param.getSensitivityExport();
        return result;
    }

    public void addSensitivityExport(String name) {
        try {
            SensitivityExportStorage storage = SensitivityExportStorage.getSensitivityExport(name);
            SensitivityExport sensitivityExport = storage.getNewSensitivityExportInstance();
            param.getSensitivityExport().add(sensitivityExport);
        } catch (IsisFishException e) {
            if (log.isErrorEnabled()) {
                log.error("Can't add sensitivity export", e);
            }
        }
    }

    /**
     * Remove an export.
     * 
     * @param export export to remove
     */
    public void removeSensitivityExport(SensitivityExport export) {
        param.getSensitivityExport().remove(export);
    }

    public void clearSensitivityExport() {
        param.getSensitivityExport().clear();
    }

    public Map<String, Class<?>> getSensitivityExportParameterNames(
            SensitivityExport export) {
        return SensitivityExportStorage.getParameterNames(export);
    }

    public void setSensitivityExportParameterValue(Export export,
            String paramName, Object value) {
        if (value != null) {
            try {
                SensitivityExportStorage.setParameterValue(export, paramName, value);
            } catch (IsisFishException e) {
                if (log.isErrorEnabled()) {
                    log.error("Can't set sensitivity export param value", e);
                }
            }
        }
    }

    public Object getSensitivityExportParameterValue(SensitivityExport sensitivityExport, String paramName) {
        Object result = null;
        try {
            result = SensitivityExportStorage.getParameterValue(sensitivityExport, paramName);
        } catch (IsisFishException e) {
            if (log.isErrorEnabled()) {
                log.error("Can't get sensitivity export param value", e);
            }
        }
        return result;
    }
    
    /**
     * Return sensitivity calculator name without .java extension.
     * 
     * @return sensitivity calculator names list
     */
    public List<String> getSensitivityNames() {
        List<String> result = new ArrayList<String>();
        for (String r : SensitivityStorage.getSensitivityNames()) {
            // there is some non java files in sensitivity directory
            if (r.endsWith(".java")) {
                // Remove .java extention
                // for example SensitivityStorage.getRuleName(String)
                result.add(r.substring(0, r.length() - 5));
            }
        }
        return result;
    }

    /**
     * Get current sensitivity calculator instance.
     * @return sensitivity calculator
     */
    public SensitivityCalculator getSensitivityCalculator() {
        return param.getSensitivityCalculator();
    }

    /**
     * Build a new sensitivity calculator instance by his name.
     * 
     * @param name calculator name
     * @return instance
     */
    public SensitivityCalculator getSensivitityCalculatorInstance(String name) {
        SensitivityCalculator sensitivityCalculator = null;
        try {
            sensitivityStorage = SensitivityStorage.getSensitivity(name);
            sensitivityCalculator = sensitivityStorage
                    .getNewSensitivityInstance();
        } catch (IsisFishException e) {
            if (log.isErrorEnabled()) {
                log.error("Can't set sensitivity", e);
            }
        }
        return sensitivityCalculator;
    }

    /**
     * Set calculator instance to use.
     * 
     * @param sensitivityCalculator new instance
     */
    public void setSensitivityCalculator(SensitivityCalculator sensitivityCalculator) {
        param.setSensitivityCalculator(sensitivityCalculator);
    }

    /**
     * Get current simulation factor list.
     * 
     * @return factor list
     */
    public List<Factor<?, ?>> getFactors() {
        List<Factor<?, ?>> result = new ArrayList<Factor<?, ?>>();
        for (Map.Entry<?, ?> entry : factors.entrySet()) {
            result.add((Factor<?, ?>) entry.getValue());
        }
        
        // sort factor on name
        Collections.sort(result, new Comparator<Factor<?, ?>> (){
            @Override
            public int compare(Factor<?, ?> o1, Factor<?, ?> o2) {
                return o1.getName().compareTo(o2.getName());
            }
        });

        return result;
    }

    public void addFactor(Factor<?, ?> f, JComponent c) {
        if (log.isDebugEnabled()) {
            log.debug("Add factor (" + f.getName() + ") : " +f.getPath());
        }
        factors.put(f.getPath() + f.getName(), f);
        factorComponent.put(f.getPath() + f.getName(), c);
        designPlan.setFactors(getFactors());
    }

    public void addContinuousFactor(String name, String comment, String path, Double min,
            Double max, JComponent c, boolean exist) {
        Factor<Double, Double> f = new Factor<Double, Double>(name);
        ContinuousDomain<Double, Double> domain = new ContinuousDomain<Double, Double>();
        domain.setMinBound(min);
        domain.setMaxBound(max);
        f.setDomain(domain);
        f.setComment(comment);
        f.setPath(path);
        if (exist) {
            removeFactor(f);
        }
        addFactor(f, c);
    }

    public void addDiscreteFactor(String name, String comment, String path,
            List<Serializable> values, JComponent c, boolean exist) {
        Factor<Serializable, Serializable> f = new Factor<Serializable, Serializable>(name);
        DiscreteDomain<Serializable, Serializable> domain = new DiscreteDomain<Serializable, Serializable>();
        int i = 0;
        for (Serializable value : values) {
            // FIXME test when integer
            // Don't work with String ;(
            domain.getValues().put(i, value);
            // and start at 0
            i++;
        }
        f.setDomain(domain);
        f.setComment(comment);
        f.setPath(path);
        if (exist) {
            removeFactor(f);
        }
        addFactor(f, c);
    }

    public void addContinuousEquationFactor(String name, String comment, String path,
            EquationContinuousDomain domain, JComponent c,
            boolean exist) {
        // factor name need to be composed
        Factor<Double, Double> f = new Factor<Double, Double>(name + "." + domain.getVariableName());
        f.setDomain(domain);
        f.setComment(comment);
        // don't modify path, need to be valid identifier
        //f.setPath(path + domain.getVariableName());
        f.setPath(path);
        if (exist) {
            removeFactor(f);
        }
        addFactor(f, c);
    }

    public void addContinuousMatrixFactor(String name, String comment, String path,
            Double coef, MatrixND matrix,
            String operateur, JComponent c, boolean exist) {
        Factor<MatrixND, Double> f = new Factor<MatrixND, Double>(name);
        MatrixContinuousDomain domain = new MatrixContinuousDomain();
        domain.setMatrix(matrix);
        domain.setOperator(operateur);
        domain.setCoefficient(coef);
        f.setComment(comment);
        f.setDomain(domain);
        f.setPath(path);
        if (exist) {
            removeFactor(f);
        }
        addFactor(f, c);
    }

    public JComponent getFactorComponent(Factor<?, ?> f) {
        return factorComponent.get(f.getPath() + f.getName());
    }

    public void removeFactor(Factor<?, ?> f) {
        factors.remove(f.getPath() + f.getName());
        factorComponent.remove(f.getPath() + f.getName());
        designPlan.setFactors(getFactors());
    }

    /*
     ************
     * General
     ************
     */

    public SimulationParameter getSimulationParameter() {
        return param;
    }

    public void setSimulationParameter(SimulationParameter p) {
        param = p;
    }

    public SimulationStorage getSimulationStorage() {
        return simulStorage;
    }

    public RegionStorage getRegionStorage() {
        return regionStorage;
    }

    public List<SimulatorLauncher> getSimulationLauncher() {
        return SimulationService.getService().getSimulationLaunchers();
    }

    /**
     * Launch automatically the simulation, when is possible (no other simulation)
     * or wait for the last automatically simulation ended.
     *
     * @param simulationId id of the simulation to simulate
     * @param launcher launcher to use
     * 
     * @see SimulatorLauncher
     */
    public void launchSimulation(String simulationId, SimulatorLauncher launcher) {

        String fullSimulationId = "sim_" + simulationId + "_"
                + DATEFORMAT.format(new java.util.Date());

        // log
        if (log.isDebugEnabled()) {
            log.debug("Launch simulation with custom launcher "
                    + launcher.toString());
        }

        try {
            if ("".equals(fullSimulationId)
                    || SimulationStorage.localyExists(fullSimulationId)
                    || SimulationService.getService().exists(fullSimulationId)) {
                ErrorHelper.showErrorDialog(_("isisfish.simulator.simulaction.badid",
                        fullSimulationId), null);
            } else {
                SimulationService.getService().submit(fullSimulationId, param,
                        launcher, 0);
            }
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't start simulation", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.simulation.launchsimulation"), eee);
        }
    }

    /**
     * Launch simulation with factors variations parameters.
     * 
     * @param simulationId id of the simulation to simulate
     * @param launcher launcher to use
     * @param sensitivityCalculator sensitivity calculator to use
     * @param designPlan design plan for this simulation
     * 
     * @see SimulatorLauncher
     * @see SensitivityCalculator
     * @see DesignPlan
     */
    public void launchSimulation(String simulationId,
            SimulatorLauncher launcher,
            SensitivityCalculator sensitivityCalculator, DesignPlan designPlan) {

        String fullSimulationId = "as_" + simulationId + "_"
                + DATEFORMAT.format(new java.util.Date());

        // log
        if (log.isDebugEnabled()) {
            log.debug("Launch factor simulation with custom launcher "
                    + launcher.toString());
            log.debug("Using sensitivityCalculator : "
                    + sensitivityCalculator.getDescription());
        }

        try {
            if ("".equals(fullSimulationId)
                    || SimulationStorage.localyExists(fullSimulationId)
                    || SimulationService.getService().exists(fullSimulationId)) {
                ErrorHelper.showErrorDialog(_("isisfish.simulator.simulaction.badid",
                        fullSimulationId), null);
            } else {
                SimulationService.getService().submit(fullSimulationId, param,
                        launcher, 0, sensitivityCalculator, designPlan);
            }
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't start simulation", eee);
            }
            ErrorHelper.showErrorDialog(_("isisfish.error.simulation.launchsimulation"), eee);
        }
    }

    public void launchSimulationWithSensibility(String simulationId,
            SimulatorLauncher launcher) {
        launchSimulation(simulationId, launcher, getSensitivityCalculator(),
                designPlan);
    }

    /**
     * Lance la second passe d'une analyse de sensibilité.
     * 
     * @param masterSensitivitySimulationName nom du prefix de toutes les simulations (without _)
     */
    public void runSensitivitySecondPass(String masterSensitivitySimulationName) {

        // calculator found
        SensitivityCalculator sensitivityCalculator = null;
        //List<SimulationStorage> simulationStorageForAnalyze = new ArrayList<SimulationStorage>();
        SortedMap<Integer, SimulationStorage> simulationStorageForAnalyze = new TreeMap<Integer, SimulationStorage>();
        File simuationDirectory = SimulationStorage.getSimulationDirectory();
        for (File simuation : simuationDirectory.listFiles()) {
            if (simuation.isDirectory()
                    && simuation.getName().startsWith(
                            masterSensitivitySimulationName + "_")) {
                SimulationStorage storage = SimulationStorage
                        .getSimulation(simuation.getName());
                String suffix = simuation.getName().substring(
                        simuation.getName().lastIndexOf("_") + 1);
                simulationStorageForAnalyze.put(Integer.valueOf(suffix),
                        storage);

                if (sensitivityCalculator == null) {
                    // try to find find calculator name in one storage
                    SimulationParameter params = storage.getParameter();
                    sensitivityCalculator = params.getSensitivityCalculator();
                }
            }
        }

        if (sensitivityCalculator != null) {
            try {
                // build master sensitivity export directory
                File masterExportDirectory = new File(SimulationStorage.getSensitivityResultsDirectory(),
                        masterSensitivitySimulationName);
                if (!masterExportDirectory.isDirectory()) {
                    masterExportDirectory.mkdirs();
                }
                List<SimulationStorage> simulationStorageForAnalyzeList = new ArrayList<SimulationStorage>(
                        simulationStorageForAnalyze.values());
                sensitivityCalculator.analyzeResult(
                        simulationStorageForAnalyzeList, masterExportDirectory);
            } catch (SensitivityException e) {
                if (log.isErrorEnabled()) {
                    log.error("Can't call analyse result", e);
                }
            }
        } else {
            if (log.isWarnEnabled()) {
                log.warn("Can't run sensitivity second pass");
            }
        }
    }

    /**
     * Renvoie les resultats de la seconde pass.
     * 
     * @param masterSensitivitySimulationName nom du prefix de toutes les simulations (without _)
     * 
     * @return les fichiers genere lors de la seconde passe (seulement ceux qui ne commence pas par ".")
     */
    public List<File> getSensitivitySecondPassResults(
            String masterSensitivitySimulationName) {

        List<File> result = new ArrayList<File>();

        // build master sensitivity export directory
        File masterExportDirectory = new File(SimulationStorage.getSensitivityResultsDirectory(), masterSensitivitySimulationName);
        if (masterExportDirectory.isDirectory()) {
            for (File exportFile : masterExportDirectory.listFiles()) {
                if (exportFile.isFile()
                        && !exportFile.getName().startsWith(".")) {
                    result.add(exportFile);
                }
            }
        }

        return result;
    }
}
