/* *##%
 * Copyright (C) 2009-2010 Ifremer, Code Lutin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package fr.ifremer.isisfish.ui.sensitivity;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JList;
import javax.swing.JRadioButton;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.text.JTextComponent;

import jaxx.runtime.JAXXInitialContext;
import jaxx.runtime.swing.editor.NumberEditor;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.math.matrix.MatrixND;
import org.nuiton.math.matrix.gui.MatrixEditor;
import org.nuiton.math.matrix.gui.MatrixPanelEditor;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.persistence.TopiaEntity;

import fr.ifremer.isisfish.entities.Equation;
import fr.ifremer.isisfish.entities.Formule;
import fr.ifremer.isisfish.entities.Selectivity;
import fr.ifremer.isisfish.entities.TargetSpecies;
import fr.ifremer.isisfish.simulator.sensitivity.Factor;
import fr.ifremer.isisfish.simulator.sensitivity.domain.ContinuousDomain;
import fr.ifremer.isisfish.simulator.sensitivity.domain.EquationContinuousDomain;
import fr.ifremer.isisfish.simulator.sensitivity.domain.MatrixContinuousDomain;
import fr.ifremer.isisfish.types.Month;
import fr.ifremer.isisfish.types.RangeOfValues;
import fr.ifremer.isisfish.ui.input.InputAction;
import fr.ifremer.isisfish.ui.input.InputOneEquationUI;
import fr.ifremer.isisfish.ui.input.InputSaveVerifier;
import fr.ifremer.isisfish.ui.input.MetierSeasonInfoSpeciesEditorUI;
import fr.ifremer.isisfish.ui.input.PopulationMigrationEmigrationUI;
import fr.ifremer.isisfish.ui.input.PopulationMigrationImmigrationUI;
import fr.ifremer.isisfish.ui.input.PopulationMigrationMigrationUI;
import fr.ifremer.isisfish.ui.input.PopulationZonesEditorUI;
import fr.ifremer.isisfish.ui.input.RangeOfValuesUI;
import fr.ifremer.isisfish.ui.input.SelectivityEditorUI;
import fr.ifremer.isisfish.ui.input.SpeciesStructuredUI;
import fr.ifremer.isisfish.ui.widget.IntervalPanel;

/**
 * Helper for sensitivity Swing component manipulation.
 * 
 * @author letellier
 * @version $Revision: 2937 $
 *
 * Last update: $Date: 2010-01-22 16:39:11 +0100 (ven., 22 janv. 2010) $
 * by : $Author: chatellier $
 */
public class EditorHelper {

    /** Class logger. */
    private static Log log = LogFactory.getLog(EditorHelper.class);

    /**
     * Get "method" property value for wing component.
     * 
     * @param component component
     * @return property value
     */
    public static String getMethod(JComponent component) {
        String method = (String)component.getClientProperty("method");
        return method;
    }

    /**
     * Get copy of component with original entity value as default value.
     * 
     * @param component component
     * @param entity entity
     * @return component copy
     */
    public static JComponent getEditor(JComponent component, TopiaEntity entity) {
        JComponent result = null;
        try {
            String fieldName = getMethod(component);
            result = component.getClass().newInstance();

            if (result instanceof JTextField) {
                ((JTextComponent) result).setText(String.valueOf(getMethodValue(entity, fieldName)));

            } else if (result instanceof NumberEditor) {
                NumberEditor ui = (NumberEditor) component;
                JTextField numEditor = new JTextField();
                numEditor.setText(String.valueOf(ui.getModel()));
                result = numEditor;

            } else if (result instanceof MatrixEditor) {
                MatrixEditor ui = (MatrixEditor) component;
                MatrixND m = (MatrixND) getMethodValue(entity, fieldName);
                MatrixPanelEditor matrix = new MatrixPanelEditor(false);
                if (m != null) {
                    matrix.setMatrix(m.clone());
                }
                matrix.setLinearModel(ui.isLinearModel());
                matrix.setLinearModelShowDefault(ui.isLinearModelShowDefault());
                result = matrix;

            } else if (result instanceof JTable) {
                ((JTable) result).setModel(((JTable) component).getModel());

            } else if (result instanceof JComboBox) {
                JComboBox combo = (JComboBox) component;
                JComboBox comboResult = new JComboBox();
                comboResult.setModel(combo.getModel());
                comboResult.setSelectedIndex(combo.getSelectedIndex());
                result = comboResult;

            } else if (result instanceof JList) {
                JList list = (JList) component;
                JList listResult = new JList();
                listResult.setModel(list.getModel());
                listResult.setSelectedIndices(list.getSelectedIndices());
                result = listResult;
            } else if (result instanceof JCheckBox) {
                ((JCheckBox) result).setSelected((Boolean) getMethodValue(entity,
                        fieldName));
                JCheckBox cOrigin = (JCheckBox) component;
                ((JCheckBox) result).setText(cOrigin.getText());
            } else if (result instanceof JRadioButton) {
                ((JRadioButton) result).setSelected((Boolean) getMethodValue(entity,
                        fieldName));
                JRadioButton cOrigin = (JRadioButton) component;
                ((JRadioButton) result).setText(cOrigin.getText());
            } else if (result instanceof SpeciesStructuredUI) {
                SpeciesStructuredUI ui = new SpeciesStructuredUI();
                ui.setActif(true);
                result = ui;
            } else if (result instanceof InputOneEquationUI) {
                InputOneEquationUI origine = (InputOneEquationUI) component;
                InputOneEquationUI ui = new InputOneEquationUI(
                        new JAXXInitialContext().add(new InputAction()));
                ui.setSelectedEquation(origine.getSelectedEquation());
                ui.setText(origine.getText());
                ui.setFormuleCategory(origine.getFormuleCategory());
                ui.setClazz(origine.getClazz());
                //ui.setMethodGet(origine.getMethodGet());
                //ui.setMethodSet(origine.getMethodSet());
                ui.setBeanProperty(origine.getBeanProperty());
                ui.setBean(entity);
                ui.setActif(true);
                ui.refresh();
                result = ui;
            } else if (result instanceof IntervalPanel) {
                IntervalPanel ipResult = new IntervalPanel();
                IntervalPanel ip = (IntervalPanel) component;
                ipResult.setLabelRenderer(Month.MONTH);
                ipResult.setModel(ip.getModel().clone());
                result = ipResult;
            } else if (result instanceof RangeOfValuesUI) {
                result = new JTextField("");
            } else if (result instanceof SelectivityEditorUI) {
                SelectivityEditorUI origine = (SelectivityEditorUI) component;
                SelectivityEditorUI ui = new SelectivityEditorUI(
                        new JAXXInitialContext().add(new InputAction()).add(
                                origine.getRegion()).add(
                                new InputSaveVerifier()));
                ui.setBean(origine.getBean());
                ui.setActif(true);
                ui.refresh();
                ui.setActif(true);
                result = ui;
            } else if (result instanceof MetierSeasonInfoSpeciesEditorUI) {
                MetierSeasonInfoSpeciesEditorUI origine = (MetierSeasonInfoSpeciesEditorUI) component;
                MetierSeasonInfoSpeciesEditorUI ui = new MetierSeasonInfoSpeciesEditorUI(
                        new JAXXInitialContext().add(new InputAction()).add(
                                origine.getRegion()).add(
                                new InputSaveVerifier()));
                ui.setBean(origine.getBean());
                if (origine.getMetierSeasonInfo() != null) {
                    ui.setMetierSeasonInfo(origine.getMetierSeasonInfo());
                    ui.setMetierSeasonSelected(true);
                    ui.setActif(true);
                }
                ui.refresh();
                result = ui;
            } else if (result instanceof PopulationZonesEditorUI) {
                PopulationZonesEditorUI origine = (PopulationZonesEditorUI) component;
                PopulationZonesEditorUI ui = new PopulationZonesEditorUI(new JAXXInitialContext().add(new InputAction()).add(origine.getRegion()));
                ui.setBean(origine.getBean());
                ui.setActif(true);
                ui.refresh();
                result = ui;
            } else if (result instanceof PopulationMigrationMigrationUI) {
                PopulationMigrationMigrationUI origine = (PopulationMigrationMigrationUI) component;
                PopulationMigrationMigrationUI ui = new PopulationMigrationMigrationUI(
                        new JAXXInitialContext().add(new InputAction()));
                ui.setBean(origine.getBean());
                ui.setPopInfo(origine.getPopInfo());
                ui.setActif(true);
                ui.refreshPanel();
                result = ui;
            } else if (result instanceof PopulationMigrationEmigrationUI) {
                PopulationMigrationEmigrationUI origine = (PopulationMigrationEmigrationUI) component;
                PopulationMigrationEmigrationUI ui = new PopulationMigrationEmigrationUI(new JAXXInitialContext().add(new InputAction()));
                ui.setBean(origine.getBean());
                ui.setPopInfo(origine.getPopInfo());
                ui.setActif(true);
                ui.refreshPanel();
                result = ui;
            } else if (result instanceof PopulationMigrationImmigrationUI) {
                PopulationMigrationImmigrationUI origine = (PopulationMigrationImmigrationUI) component;
                PopulationMigrationImmigrationUI ui = new PopulationMigrationImmigrationUI(new JAXXInitialContext().add(new InputAction()));
                ui.setBean(origine.getBean());
                ui.setPopInfo(origine.getPopInfo());
                ui.setActif(true);
                ui.refreshPanel();
                result = ui;
            }
        } catch (IllegalAccessException ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't make new instance of " + component, ex);
            }
        } catch (InstantiationException ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't make new instance of " + component, ex);
            }
        }
        return result;
    }

    /**
     * Get copy of component with original entity value as default value.
     * 
     * @param component component
     * @param entity entity
     * @return component copy
     */
    public static ContinuousPanelContainerUI getContinuousPanel(JComponent component, TopiaEntity entity) {
        ContinuousPanelContainerUI result = null;
        String fieldName = getMethod(component);
        DefaultContinuousPanelUI continuousPanel = null;
        if (component instanceof InputOneEquationUI) {
            InputOneEquationUI origine = (InputOneEquationUI) component;
            EquationContinuousPanelUI ui = new EquationContinuousPanelUI(
                    new JAXXInitialContext().add(new InputAction()));
            ui.setSelectedEquation(origine.getSelectedEquation());
            ui.setText(origine.getText());
            ui.setFormuleCategory(origine.getFormuleCategory());
            ui.setClazz(origine.getClazz());
            //ui.setMethodGet(origine.getMethodGet());
            //ui.setMethodSet(origine.getMethodSet());
            ui.setBeanProperty(origine.getBeanProperty());
            ui.setBean(entity);
            result = ui;
        } else if (component instanceof MatrixEditor) {
            MatrixContinuousPanelUI matrixPanel = new MatrixContinuousPanelUI();
            MatrixND m = (MatrixND) getMethodValue(entity, fieldName);
            if (m != null) {
                matrixPanel.setMatrix(m.clone());
            }
            result = matrixPanel;
        } else {
            continuousPanel = new DefaultContinuousPanelUI();
            result = continuousPanel;
        }
        if (continuousPanel != null) {
            continuousPanel.init(getNewBoundEditor(component, entity), getNewBoundEditor(component, entity));
        }
        return result;
    }

    /**
     * Get new text component with current entity default value.
     * 
     * @param component component
     * @param entity entity
     * @return bound editor
     */
    protected static JTextField getNewBoundEditor(JComponent component, TopiaEntity entity) {
        String fieldName = getMethod(component);
        Object value = getMethodValue(entity, fieldName);
        if (component instanceof JTextComponent) {
            return new JTextField(String.valueOf(value));
        }
        return new JTextField();
    }

    /**
     * Get continuous editor for component with given value.
     * 
     * @param component component
     * @param domain domain
     * @param factor factor
     * @return component copy
     */
    public static ContinuousPanelContainerUI getContinuousPanelWithValue(
            JComponent component, ContinuousDomain<?, ?> domain, Factor<?, ?> factor) {
        ContinuousPanelContainerUI result = null;

        DefaultContinuousPanelUI continuousPanel = null;
        if (component instanceof InputOneEquationUI) {
            InputOneEquationUI origine = (InputOneEquationUI) component;
            EquationContinuousPanelUI ui = new EquationContinuousPanelUI(
                    new JAXXInitialContext().add(new InputAction()));
            ui.setText(origine.getText());
            ui.setFormuleCategory(origine.getFormuleCategory());
            ui.getEditor().getEditor().setText((String) factor.getValue());
            ui.setClazz(origine.getClazz());
            //ui.setMethodGet(origine.getMethodGet());
            //ui.setMethodSet(origine.getMethodSet());
            ui.setBeanProperty(origine.getBeanProperty());
            ui.setBean(origine.getBean());

            EquationContinuousDomain equationDomain = (EquationContinuousDomain) domain;
            ui.addDomain(equationDomain);
            ui.setTable();

            result = ui;
        } else if (component instanceof MatrixEditor) {
            MatrixContinuousPanelUI panel = new MatrixContinuousPanelUI();
            MatrixContinuousDomain matrixDomain = (MatrixContinuousDomain) domain;
            MatrixND m = matrixDomain.getMatrix();
            if (m != null) {
                panel.setMatrix(m.clone());
            }
            panel.getOperator().setSelectedItem(matrixDomain.getOperator());
            panel.getCoef().setText(
                    String.valueOf(matrixDomain.getCoefficient() * 100));
            result = panel;
        } else {
            continuousPanel = new DefaultContinuousPanelUI();
            result = continuousPanel;
        }
        if (continuousPanel != null) {
            continuousPanel.init(new JTextField(String.valueOf(domain
                    .getMinBound())), new JTextField(String.valueOf(domain
                    .getMaxBound())));
        }

        return result;
    }

    /**
     * Get new instance of component filled with original swing component value.
     * 
     * @param component component
     * @return new component instance
     */
    public static JComponent getEditorWithValue(JComponent component) {
        Object value = getValue(component);
        return getEditorWithValue(component, value);
    }

    /**
     * Get new instance of component filled with specified value depending
     * of component type.
     * 
     * @param component component
     * @param value value
     * @return new component instance
     */
    public static JComponent getEditorWithValue(JComponent component, Object value) {
        JComponent result = null;

        if (log.isDebugEnabled()) {
            log.debug("Get editor for " + component + " with value = " + value);
        }

        try {
            result = component.getClass().newInstance();

            if (result instanceof JTextComponent) {
                ((JTextComponent) result).setText(String.valueOf(value));
            } else if (result instanceof NumberEditor) {
                JTextField numEditor = new JTextField();
                numEditor.setText(String.valueOf(value));
                result = numEditor;
            } else if (result instanceof RangeOfValuesUI) {
                result = new JTextField(value.toString());
            } else if (result instanceof MatrixEditor) {
                MatrixEditor ui = (MatrixEditor) component;
                MatrixND m = (MatrixND) value;
                MatrixPanelEditor matrix = new MatrixPanelEditor(false);
                if (m != null) {
                    matrix.setMatrix(m.clone());
                }
                matrix.setLinearModel(ui.isLinearModel());
                matrix.setLinearModelShowDefault(ui.isLinearModelShowDefault());
                result = matrix;
            } else if (result instanceof JComboBox) {
                JComboBox combo = (JComboBox) component;
                JComboBox comboResult = new JComboBox();
                comboResult.setModel(combo.getModel());
                comboResult.setSelectedItem(value);
                result = comboResult;
            } else if (result instanceof JList) {
                JList list = (JList) component;
                JList listResult = new JList();
                listResult.setModel(list.getModel());
                //listResult.setSelectedValues((Object[]) value);
                listResult.setSelectedIndices(list.getSelectedIndices());
                result = listResult;
            } else if (result instanceof JCheckBox) {
                ((JCheckBox) result).setSelected((Boolean) value);
                JCheckBox ch = (JCheckBox) component;
                ((JCheckBox) result).setText(ch.getText());

            } else if (result instanceof JRadioButton) {
                ((JRadioButton) result).setSelected((Boolean) value);
                JRadioButton cOrigin = (JRadioButton) component;
                ((JRadioButton) result).setText(cOrigin.getText());
            } else if (result instanceof SpeciesStructuredUI) {
                //SpeciesStructuredUI origine = (SpeciesStructuredUI) component;
                SpeciesStructuredUI ui = new SpeciesStructuredUI();
                ui.getFieldSpeciesDynamicAge().setSelected((Boolean) value);
                ui.setActif(true);
                result = ui;
            } else if (result instanceof InputOneEquationUI) {
                InputOneEquationUI origine = (InputOneEquationUI) component;
                InputOneEquationUI ui = new InputOneEquationUI(new JAXXInitialContext().add(new InputAction()));
                ui.getEditor().setText((String) value);
                ui.setText(origine.getText());
                ui.setFormuleCategory(origine.getFormuleCategory());
                ui.setClazz(origine.getClazz());
                /// ??? was not set before
                ui.setBeanProperty(origine.getBeanProperty());
                ui.setFormule((Formule) value);
                ui.setActif(true);
                ui.refresh();
                result = ui;
            } else if (result instanceof IntervalPanel) {
                IntervalPanel ipResult = new IntervalPanel();
                IntervalPanel ip = (IntervalPanel) component;
                ipResult.setLabelRenderer(Month.MONTH);
                ipResult.setModel(ip.getModel().clone());
                result = ipResult;
            } else if (result instanceof SelectivityEditorUI) {
                SelectivityEditorUI origine = (SelectivityEditorUI) component;
                SelectivityEditorUI ui = new SelectivityEditorUI(
                        new JAXXInitialContext().add(new InputAction()).add(
                                origine.getRegion()).add(new InputSaveVerifier()));
                ui.setBean(origine.getBean());
                ui.getBean().setPopulationSelectivity((Collection<Selectivity>) value);
                ui.setActif(true);
                ui.refresh();
                ui.setActif(true);
                result = ui;
            } else if (result instanceof MetierSeasonInfoSpeciesEditorUI) {
                MetierSeasonInfoSpeciesEditorUI origine = (MetierSeasonInfoSpeciesEditorUI) component;
                MetierSeasonInfoSpeciesEditorUI ui = new MetierSeasonInfoSpeciesEditorUI(
                        new JAXXInitialContext().add(new InputAction()).add(
                                origine.getRegion()).add(
                                new InputSaveVerifier()));
                ui.setBean(origine.getBean());
                if (origine.getMetierSeasonInfo() != null) {
                    ui.setMetierSeasonInfo(origine.getMetierSeasonInfo());
                    ui.getMetierSeasonInfo().setSpeciesTargetSpecies((Collection<TargetSpecies>) value);
                    ui.setMetierSeasonSelected(true);
                    ui.setActif(true);
                }
                ui.refresh();
                result = ui;
            } else if (result instanceof PopulationZonesEditorUI) {
                PopulationZonesEditorUI origine = (PopulationZonesEditorUI) component;
                PopulationZonesEditorUI ui = new PopulationZonesEditorUI(
                        new JAXXInitialContext().add(new InputAction()).add(origine.getRegion()));
                ui.setBean(origine.getBean());
                MatrixND m = (MatrixND) value;
                ui.getFieldPopulationMappingZoneReproZoneRecru().setMatrix(m == null ? null : m.clone());
                ui.setActif(true);
                ui.refresh();
                result = ui;
            } else if (result instanceof PopulationMigrationMigrationUI) {
                PopulationMigrationMigrationUI origine = (PopulationMigrationMigrationUI) component;
                PopulationMigrationMigrationUI ui = new PopulationMigrationMigrationUI(
                        new JAXXInitialContext().add(new InputAction()));
                ui.setBean(origine.getBean());
                ui.setPopInfo(origine.getPopInfo());
                MatrixND m = (MatrixND) value;
                ui.getPopInfo().setMigrationMatrix(m == null ? null : m.clone());
                ui.setActif(true);
                ui.refreshPanel();
                result = ui;
            } else if (result instanceof PopulationMigrationEmigrationUI) {
                PopulationMigrationEmigrationUI origine = (PopulationMigrationEmigrationUI) component;
                PopulationMigrationEmigrationUI ui = new PopulationMigrationEmigrationUI(
                        new JAXXInitialContext().add(new InputAction()));
                ui.setBean(origine.getBean());
                ui.setPopInfo(origine.getPopInfo());
                MatrixND m = (MatrixND) value;
                ui.getPopInfo().setEmigrationMatrix(m == null ? null : m.clone());
                ui.setActif(true);
                ui.refreshPanel();
                result = ui;
            } else if (result instanceof PopulationMigrationImmigrationUI) {
                PopulationMigrationImmigrationUI origine = (PopulationMigrationImmigrationUI) component;
                PopulationMigrationImmigrationUI ui = new PopulationMigrationImmigrationUI(
                        new JAXXInitialContext().add(new InputAction()));
                ui.setBean(origine.getBean());
                ui.setPopInfo(origine.getPopInfo());
                MatrixND m = (MatrixND) value;
                ui.getPopInfo().setImmigrationMatrix(m == null ? null : m.clone());
                ui.setActif(true);
                ui.refreshPanel();
                result = ui;
            }
        } catch (IllegalAccessException ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't make new instance of " + component, ex);
            }
        } catch (InstantiationException ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't make new instance of " + component, ex);
            }
        }
        return result;
    }

    /**
     * Return value in swing component that could be next used into factor.
     * 
     * @param component component
     * @return factor value
     */
    public static Object getValue(JComponent component) {
        Object result = null;
        if (component instanceof JTextComponent) {
            result = ((JTextComponent) component).getText();
        } else if (component instanceof NumberEditor) {
            result = ((NumberEditor) component).getModel();
        } else if (component instanceof PopulationZonesEditorUI) {
            result = ((PopulationZonesEditorUI) component)
                    .getFieldPopulationMappingZoneReproZoneRecru().getMatrix();
        } else if (component instanceof MatrixPanelEditor) {
            result = ((MatrixPanelEditor) component).getMatrix();

        } else if (component instanceof JComboBox) {
            result = ((JComboBox) component).getSelectedItem();

        } else if (component instanceof JList) {
            result = ((JList) component).getSelectedValues();

        } else if (component instanceof JCheckBox) {
            result = ((JCheckBox) component).isSelected();

        } else if (component instanceof JRadioButton) {
            result = ((JRadioButton) component).isSelected();

        } else if (component instanceof InputOneEquationUI) {
            result = ((InputOneEquationUI) component).getEditor().getText();

        } else if (component instanceof SpeciesStructuredUI) {
            result = ((SpeciesStructuredUI) component)
                    .getFieldSpeciesDynamicAge().isSelected();

        } else if (component instanceof SelectivityEditorUI) {
            result = ((SelectivityEditorUI) component).getBean()
                    .getPopulationSelectivity();

        } else if (component instanceof MetierSeasonInfoSpeciesEditorUI) {
            result = ((MetierSeasonInfoSpeciesEditorUI) component)
                    .getMetierSeasonInfo().getSpeciesTargetSpecies();

        } else if (component instanceof PopulationZonesEditorUI) {
            result = ((PopulationZonesEditorUI) component)
                    .getFieldPopulationMappingZoneReproZoneRecru().getMatrix();

        } /*else if (component instanceof RangeOfValuesUI) {

        } */else if (component instanceof PopulationMigrationMigrationUI) {
            result = ((PopulationMigrationMigrationUI) component).getPopInfo()
                    .getMigrationMatrix();

        } else if (component instanceof PopulationMigrationEmigrationUI) {
            result = ((PopulationMigrationEmigrationUI) component).getPopInfo()
                    .getEmigrationMatrix();

        } else if (component instanceof PopulationMigrationImmigrationUI) {
            result = ((PopulationMigrationImmigrationUI) component)
                    .getPopInfo().getImmigrationMatrix();
        }
        return result;
    }

    /**
     * Return if component is instanceof {@link RangeOfValuesUI}.
     * @param component component
     * @return {@code true} if component is intance of {@link RangeOfValuesUI}
     */
    public static boolean isRangeOfValue(JComponent component) {
        if (component instanceof RangeOfValuesUI) {
            return true;
        }
        return false;
    }

    /**
     * Return if component is is continue factor enable.
     * 
     * @param component component
     * @return {@code true} if component is is continue factor enable
     */
    public static boolean isContinue(JComponent component) {
        if (component instanceof RangeOfValuesUI) {
            RangeOfValuesUI ui = (RangeOfValuesUI) component;
            if (ui.getFieldGearParamType().getSelectedItem().equals("Float")) {
                String value = ui.getFieldGearParamPossibleValue().getText();
                if (value.matches("^\\ *[0-9]*\\ *\\-\\ *[0-9]*\\ *$")) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * Return true if component or entity can be defined in continuous factor.
     * 
     * @param component component
     * @param entity entity
     * @return continuous enabled
     */
    public static boolean canBeContinue(JComponent component,
            TopiaEntity entity) {
        String fieldName = (String) component.getClientProperty("method");
        if (component instanceof JTextComponent) {
            Object o = getMethodValue(entity, fieldName);
            return canBeContinue(component, o);
        } else if (component instanceof NumberEditor) {
            return true;
        } else if (component instanceof InputOneEquationUI) {
            return true;
        } else if (component instanceof MatrixEditor) {
            return true;
        } else if (component instanceof RangeOfValuesUI) {
            RangeOfValuesUI ui = (RangeOfValuesUI) component;
            if (ui.getFieldGearParamType().getSelectedItem().equals("Float")) {
                return true;
            }
        }
        return false;
    }

    /**
     * Return true if component or value can be defined in continuous factor.
     * 
     * @param component component
     * @param value value
     * @return continuous enabled
     */
    public static boolean canBeContinue(JComponent component, Object value) {
        boolean result = false;

        if (component instanceof JTextComponent) {
            if (value instanceof Double) {
                result = true;
            } else if (value instanceof Long) {
                result = true;
            }
        }
        else if (component instanceof InputOneEquationUI) {
            result = true;
        }
        else if (component instanceof MatrixPanelEditor) {
            result = true;
        }
        // Metier.gearPossibleValue case
        else if (value instanceof RangeOfValues) {
            RangeOfValues rangeOfValues = (RangeOfValues)value;
            if (rangeOfValues.getType().equals("Float")) {
                result = true;
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Component " + component.getClass().getSimpleName() +
                    " with value " + value + "(" + value.getClass().getSimpleName() + ") can be continuous : " + result);
        }
        return result;
    }

    /**
     * Get value for fieldName in entity.
     * 
     * @param entity
     * @param fieldName
     * @return method return value
     */
    protected static Object getMethodValue(Object entity, String fieldName) {
        Object result = null;
        try {
            // fieldName maybe be sometime lower case
            String getMethod = "get" + StringUtils.capitalize(fieldName);

            Method m = entity.getClass().getMethod(getMethod);
            result = m.invoke(entity);
        } catch (Exception ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't get entity value", ex);
            }
        }
        return result;
    }

    // RangeOfValue
    public static ContinuousPanelContainerUI getContinuousRangeOfValuePanel(
            JComponent cOrigine, TopiaEntity bean) {
        DefaultContinuousPanelUI continuousPanel = new DefaultContinuousPanelUI();
        RangeOfValuesUI ui = (RangeOfValuesUI) cOrigine;
        String value = ui.getFieldGearParamPossibleValue().getText();
        String min = "0";
        String max = "0";
        if (value.matches("^\\ *[0-9]*\\ *\\-\\ *[0-9]*\\ *$")) {
            int first = 0;
            if (value.startsWith("-")) {
                first = 1;
            }
            first = value.indexOf("-");
            if (first != -1) {
                min = value.substring(0, first);
                max = value.substring(first + 1);
            }
        }
        continuousPanel.init(new JTextField(min), new JTextField(max));
        return continuousPanel;
    }

    public static List<String> getDiscretRangeOfValueValues(
            JComponent cOrigine, TopiaEntity bean) {
        List<String> result = new ArrayList<String>();
        RangeOfValuesUI ui = (RangeOfValuesUI) cOrigine;
        String value = ui.getFieldGearParamPossibleValue().getText();
        if (value.matches("^\\ *[0-9]*\\ *\\-\\ *[0-9]*\\ *$")) {
            int first = 0;
            if (value.startsWith("-")) {
                first = 1;
            }
            first = value.indexOf("-");
            int min = 0;
            int max = 0;
            if (first != -1) {
                min = Integer.valueOf(value.substring(0, first).trim());
                max = Integer.valueOf(value.substring(first + 1).trim());
            }
            for (int i = min; i <= max; i++) {
                result.add(i + "");
            }
        } else {
            if (value.indexOf(";") != -1) {
                for (String s : value.split("\\;")) {
                    result.add(s);
                }
            } else {
                result.add(value);
            }
        }
        return result;
    }
    
    /**
     * Return the component to edit factor depending on factor path.
     * (inspect the return type of entity denoted by path)
     * 
     * Used to edit reloader factor from old simulation.
     * 
     * @param factor factor
     * @param context context
     * @return component
     */
    public static JComponent getEditorForFactor(Factor<?, ?> factor, TopiaContext context) {
        JComponent result = null;
        
        // we can't do it only with path maybe ?
        // need to check property type with introspection
        String path = factor.getPath();

        if (log.isDebugEnabled()) {
            log.debug("Try to find component for path " + path);
        }

        String topiaId = path.substring(0, path.lastIndexOf('#'));
        String property = path.substring(path.lastIndexOf('#') + 1);

        Class<?> typeClazz = null;
        try {
            TopiaEntity entity = context.findByTopiaId(topiaId);
            
            if (log.isDebugEnabled()) {
                log.debug("Factor " + factor.getName() + " denoted entity " + entity);
            }
            
            String getter = "get" + StringUtils.capitalize(property);
            Method method = entity.getClass().getMethod(getter);
            typeClazz = method.getReturnType();
            
            if (log.isDebugEnabled()) {
                log.debug("Property " + property + " denoted type " + typeClazz.getSimpleName());
            }
            
            result = getEditorForType(typeClazz);
            
        } catch (TopiaException ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't find entity for " + topiaId, ex);
            }
        } catch (SecurityException ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't get type info for " + topiaId, ex);
            }
        } catch (NoSuchMethodException ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't get type info for " + topiaId, ex);
            }
        }
        
        return result;
    }

    /**
     * Get editor for type.
     * 
     * @param typeClazz type to get editor
     * @return component
     */
    public static JComponent getEditorForType(Class<?> typeClazz) {

        JComponent result = null;
        
        if (Double.class.isAssignableFrom(typeClazz)) {
            result = new JTextField();
        }
        else if (double.class.isAssignableFrom(typeClazz)) {
            result = new JTextField();
        }
        else if (MatrixND.class.isAssignableFrom(typeClazz)) {
            result = new MatrixPanelEditor();
        }
        else if (Equation.class.isAssignableFrom(typeClazz)) {
            result = new InputOneEquationUI();
        }

        return result;
    }

    /**
     * Get editor for editing value.
     * 
     * @param bean bean
     * @param  beanProperty bean property
     * @return component
     */
    public static JComponent getEditorForValue(Object bean, String beanProperty) {
        Object basicType = getMethodValue(bean, beanProperty);
        
        JComponent component = null;

        if (basicType instanceof Equation) {
            Equation basicEquation = (Equation)basicType;
            InputOneEquationUI equationComponent = new InputOneEquationUI(
                    new JAXXInitialContext().add(new InputAction()));
            equationComponent.setFormuleCategory(basicEquation.getCategory());
            equationComponent.setBean((TopiaEntity)bean);
            equationComponent.setBeanProperty(beanProperty);
            equationComponent.putClientProperty("method", beanProperty);
            component = equationComponent;
        }

        return component;
    }
}
