/* *##%
 * Copyright (C) 2006 - 2010
 *     Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package fr.ifremer.isisfish.ui.script;

import static org.nuiton.i18n.I18n._;

import java.awt.Color;
import java.awt.Desktop;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.io.Writer;
import java.lang.reflect.Method;
import java.net.URI;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;
import javax.swing.event.CaretEvent;
import javax.swing.event.CaretListener;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.TreePath;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.FileUtil;
import org.nuiton.util.ZipUtil;
import org.nuiton.widget.SwingUtil;

import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.IsisFishRuntimeException;
import fr.ifremer.isisfish.datastore.AnalysePlanStorage;
import fr.ifremer.isisfish.datastore.CodeSourceStorage;
import fr.ifremer.isisfish.datastore.ExportStorage;
import fr.ifremer.isisfish.datastore.FormuleStorage;
import fr.ifremer.isisfish.datastore.JavaSourceStorage;
import fr.ifremer.isisfish.datastore.RuleStorage;
import fr.ifremer.isisfish.datastore.ScriptStorage;
import fr.ifremer.isisfish.datastore.SensitivityExportStorage;
import fr.ifremer.isisfish.datastore.SensitivityStorage;
import fr.ifremer.isisfish.datastore.SimulatorStorage;
import fr.ifremer.isisfish.equation.Language;
import fr.ifremer.isisfish.ui.WelcomePanelUI;
import fr.ifremer.isisfish.ui.script.model.ScriptTreeModel;
import fr.ifremer.isisfish.ui.util.ErrorHelper;
import fr.ifremer.isisfish.ui.vcs.UpdateDialogUI;
import fr.ifremer.isisfish.util.ClasspathTemplateLoader;
import fr.ifremer.isisfish.util.JavadocHelper;
import fr.ifremer.isisfish.vcs.VCSException;
import freemarker.cache.TemplateLoader;
import freemarker.template.Configuration;
import freemarker.template.Template;

class ScriptFileFilter implements FileFilter {
    protected FileFilter filter;

    public ScriptFileFilter(FileFilter filter) {
        this.filter = filter;
    }

    public boolean accept(File dir) {
        return !filter.accept(dir) || dir.getName().equals("data");
    }
}

/**
 * ScriptAction.
 * 
 * Template are now loaded with freemarker.
 * 
 * @author letellier
 * @version $Revision: 1526 $
 *
 * Last update: $Date: 2008-10-07 18:46:13 +0200 (mar 07 oct 2008) $
 * by : $Author: tchemit $
 */
public class ScriptAction implements TreeSelectionListener, CaretListener {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static Log log = LogFactory.getLog(ScriptAction.class);

    /** Couleur de succes (vert leger). */
    protected static final Color COLOR_SUCCESS = new Color(210, 255, 210);

    /** Couleur d'echec (rouge leger). */
    protected static final Color COLOR_FAILURE = new Color(255, 210, 210);

    /** Storage for currently edited file. */
    protected CodeSourceStorage codeStorage;

    /** Freemarke configuration used to create new script (based on templates). */
    protected Configuration freemarkerConfiguration;

    /** UI managed by this action class. */
    protected ScriptUI scriptUI;
    
    /**
     * Constructeur.
     * 
     * Init freemarker.
     * 
     * @param scriptUI managed script UI
     */
    public ScriptAction(ScriptUI scriptUI) {

        this.scriptUI = scriptUI;

        freemarkerConfiguration = new Configuration();
        // needed to overwrite "Defaults to default system encoding."
        // fix encoding issue on some systems
        freemarkerConfiguration.setDefaultEncoding("utf-8");
        // specific template loader to get template from jars (classpath)
        TemplateLoader templateLoader = new ClasspathTemplateLoader();
        freemarkerConfiguration.setTemplateLoader(templateLoader);

    }

    /**
     * Post init, must be called after ui building.
     */
    public void postInit() {
        
        // add listeners
        this.scriptUI.getTree().addTreeSelectionListener(this);
        this.scriptUI.getEditor().addCaretListener(this);
    }

    /**
     * Update script UI component actions buttons.
     */
    protected void setButton() {
    
        File selectedFile = (File)scriptUI.getTree().getLastSelectedPathComponent();

        if (selectedFile != null) {
            scriptUI.setSingleFileSelected(selectedFile.isFile());
            scriptUI.setJavaFileSelected(selectedFile.getName().endsWith(".java"));
        }
        else {
            scriptUI.setSingleFileSelected(false);
            scriptUI.setJavaFileSelected(false);
        }
        
        /*buttonScriptExport.setEnabled(tree.getSelectionValue() != null);
        miExport.setEnabled(tree.getSelectionValue() != null);
        buttonScriptCommit.setEnabled(fileSelected);
        miCommitVCS.setEnabled(fileSelected);
        miDiffVCS.setEnabled(fileSelected);
        //buttonScriptUpdate.setEnabled(fileSelected);
        buttonScriptSave.setEnabled(fileSelected);
        miSave.setEnabled(fileSelected);
        buttonScriptCut.setEnabled(fileSelected);
        miCut.setEnabled(fileSelected);
        buttonScriptCopy.setEnabled(fileSelected);
        miCopy.setEnabled(fileSelected);
        buttonScriptPaste.setEnabled(fileSelected);
        miPaste.setEnabled(fileSelected);
        buttonScriptCheck.setEnabled(isJavaScript);
        miCheckSyntax.setEnabled(isJavaScript);
        buttonScriptEval.setEnabled(isJavaScript);
        miEvaluate.setEnabled(isJavaScript);*/
    }
    
    protected void setStatusMessage(String msg) {
        // FIXME remove all parent container reference
        WelcomePanelUI parentUI = scriptUI.getParentContainer(WelcomePanelUI.class);
        parentUI.setStatusMessage(msg);
    }
    protected void setStatusMessage(String msg, boolean running) {
        // FIXME remove all parent container reference
        WelcomePanelUI parentUI = scriptUI.getParentContainer(WelcomePanelUI.class);
        parentUI.setStatusMessage(msg, running);
    }
    
    /**
     * Make new script, and select it on tree.
     */
    public void newScript() {
        ScriptMapping selectedType = (ScriptMapping) scriptUI.getScriptTypeChoice().getSelectedItem();
        newScript(selectedType);
    }

    /**
     * Make new script, and select it on tree.
     *
     * @param scriptType script to make
     */
    public void newScript(ScriptMapping scriptType) {

        String equationModelType = "";
        String equationModelTypePath = "";

        // specific case for equation model
        if (scriptType.equals(ScriptMapping.EquationModel)) {
            java.util.List<String> values = FormuleStorage.getCategories();
            equationModelType = (String) JOptionPane.showInputDialog(scriptUI,
                    _("isisfish.message.new.formule.category"),
                    _("isisfish.message.new.formule.title"),
                    JOptionPane.PLAIN_MESSAGE, null, values.toArray(), values.get(0));
            equationModelTypePath = equationModelType + File.separator;
        }

        // if user has not choose "cancel"
        if (equationModelType != null) {
            String fileName = JOptionPane.showInputDialog(_("isisfish.message.new.filename"));
            // user cancel
            if (!StringUtils.isEmpty(fileName)) {
                File scriptFile = newScript(equationModelTypePath + fileName, scriptType);
                // creation successful
                if (scriptFile != null) {
                    ScriptTreeModel model = (ScriptTreeModel) scriptUI.getTree().getModel();
                    model.fileAdded(scriptFile);
                    TreePath treePath = model.getTreePathFor(scriptFile);
                    scriptUI.getTree().setSelectionPath(treePath);
                }
            }
        }
    }

    /**
     * Creer un nouveau script, ici un script peut-etre un Script, un Simulator,
     * un Export.
     *
     * @param fileName full filename
     * @param scriptType le type que l'on souhaite Script, Simulator, ou Export.
     * @return created file or {@code null} if any error happen
     */
    protected File newScript(String fileName, ScriptMapping scriptType) {

        if (log.isDebugEnabled()) {
            log.debug("newScript called [" + scriptType + "]");
        }

        File scriptFile = null;

        try {
            // Vérifie qu'il n'y pas de caractères spéciaux. Seul les caractre
            // de a à z (majuscule ou minuscule) ainsi que les nombres sont
            // autorisés. + pour signifier qu'il doit y avoir
            // au moins 1 caractère.
            String realFilename;
            String category;

            int pos = fileName.lastIndexOf('/');
            if (pos != -1) {
                if (scriptType != ScriptMapping.EquationModel) {
                    // interdit pour le moment ?
                    String message = _("isisfish.error.invalid.file.name",
                            fileName);
                    Exception e = new RuntimeException(message);
                    returnError(_("isisfish.error.script.create", fileName, e.getMessage()), e);
                    return null;
                }
                // il y a un sous type à traiter
                if (pos == fileName.length() - 1) {
                    String message = _("isisfish.error.invalid.file.name",
                            fileName);
                    Exception e = new RuntimeException(message);
                    returnError(_("isisfish.error.script.create", fileName, e.getMessage()), e);
                    return null;
                }
                realFilename = fileName
                        .substring(fileName.lastIndexOf('/') + 1);
                category = fileName.substring(0, fileName.lastIndexOf('/'));
            } else {
                realFilename = fileName;
                category = "";
            }
            if (!realFilename.matches("[A-Z0-9_][a-zA-Z0-9_]*")) {
                String message = _("isisfish.error.invalid.file.name", fileName);
                Exception e = new RuntimeException(message);
                returnError(_("isisfish.error.script.create", fileName, e.getMessage()), e);
                return null;
            }
            //TODO do test on category
            CodeSourceStorage script = null;
            switch (scriptType) {
            case EquationModel:
                script = FormuleStorage.createFormule(category, realFilename,
                        Language.JAVA);
                break;
            case AnalysePlan:
                script = AnalysePlanStorage.getAnalysePlan(fileName);
                break;
            case Export:
                script = ExportStorage.getExport(fileName);
                break;
            case Rule:
                script = RuleStorage.getRule(fileName);
                break;
            case Script:
                script = ScriptStorage.getScript(fileName);
                break;
            case Simulator:
                script = SimulatorStorage.getSimulator(fileName);
                break;
            case Sensitivity:
                script = SensitivityStorage.getSensitivity(fileName);
                break;
            case SensitivityExport:
                script = SensitivityExportStorage
                        .getSensitivityExport(fileName);
                break;
            default:
                if (log.isErrorEnabled()) {
                    log.fatal("ScriptType unknown: " + scriptType);
                }
            }

            if (script.exists()) {
                // Message d'erreur si le fichier existe en local.
                String message = _("isisfish.error.file.already.exists",
                        fileName);
                Exception e = new RuntimeException(message);
                returnError(_("isisfish.error.script.create", fileName, e
                        .getMessage()), e);
            }

            String scriptTemplatePath = scriptType.getTemplatePath();

            if (scriptTemplatePath != null) {

                if (log.isDebugEnabled()) {
                    log.debug("Parsing freemarker template located in "
                            + scriptTemplatePath);
                }

                // get template
                Template template = freemarkerConfiguration
                        .getTemplate(scriptTemplatePath);

                // context values
                Map<String, Object> root = new HashMap<String, Object>();
                root.put("name", realFilename);
                root.put("date", new Date());
                root.put("author", IsisFish.config.getUserName());
                root.put("email", IsisFish.config.getUserMail());

                // process template
                Writer out = new StringWriter();
                template.process(root, out);
                out.flush();
                script.setContent(out.toString());
            } else {
                throw new IsisFishRuntimeException("There is no templatePath");
            }

            codeStorage = script;
            scriptFile = script.getFile();
        } catch (Exception eee) {
            returnError(_("isisfish.error.script.create", fileName, eee
                    .getMessage()), eee);
        }

        return scriptFile;
    }

    /**
     * Write error in log and display exception to user.
     * 
     * @param s message
     * @param eee cause
     */
    protected void returnError(String s, Exception eee) {
        if (log.isErrorEnabled()) {
            log.error(s, eee);
        }
        ErrorHelper.showErrorDialog(s, eee);
    }

    /*
     * @see javax.swing.event.TreeSelectionListener#valueChanged(javax.swing.event.TreeSelectionEvent)
     */
    @Override
    public void valueChanged(TreeSelectionEvent e) {
        if (e.getNewLeadSelectionPath() != null) {
            File selectedFile = (File)e.getNewLeadSelectionPath().getLastPathComponent();
            if (selectedFile.isFile()) {
                // load file into current action codeStorage
                loadScript(selectedFile);
                scriptUI.getEditor().open(selectedFile);
                // force refresh
                scriptUI.getEditor().repaint();
                scriptUI.getEditor().validate();
                
                setButton();
            }
            else {
                scriptUI.getEditor().close();
                // force refresh
                scriptUI.getEditor().repaint();
                scriptUI.getEditor().validate();
                
                setButton();
            }
        }
    }
    
    /**
     * Load specified script in current action.
     * 
     * TODO can we change this ?
     * 
     * @param file file to load
     */
    public void loadScript(File file) {
        ScriptMapping mapping = ScriptMapping.getMappingFor(file);
        CodeSourceStorage script = null;

        try {
            switch (mapping) {
            case EquationModel:
                String fullPath = file.getAbsolutePath();
                int lastIndexOf = fullPath.lastIndexOf('/');
                // in string .../aaa/bbb/ccc/ddd.java
                // get ccc
                String category = fullPath.substring(fullPath.lastIndexOf('/',
                        lastIndexOf - 1) + 1, lastIndexOf);
                script = FormuleStorage.getFormule(category, file.getName());
                break;
            case Rule:
                script = RuleStorage.getRule(file.getName());
                break;
            case AnalysePlan:
                script = AnalysePlanStorage.getAnalysePlan(file.getName());
                break;
            case Export:
                script = ExportStorage.getExport(file.getName());
                break;
            case Script:
                script = ScriptStorage.getScript(file.getName());
                break;
            case Simulator:
                script = SimulatorStorage.getSimulator(file.getName());
                break;
            case Sensitivity:
                script = SensitivityStorage.getSensitivity(file.getName());
                break;
            case SensitivityExport:
                script = SensitivityExportStorage.getSensitivityExport(file
                        .getName());
                break;
            default:
                log.fatal("ScriptType unknown: " + file.getName());
            }

            //frame.setInfoText(_("isisfish.message.load.finished"));
        } catch (Exception eee) {
            returnError(_("isisfish.error.script.load", file.getAbsolutePath(),
                    eee.getMessage()), eee);

        } finally {
            codeStorage = script;
        }
    }

    public boolean fileLoaded() {
        return codeStorage != null;
    }

    public boolean isJavaScript() {
        return JavaSourceStorage.class.isInstance(codeStorage);
    }

    /**
     * Save current editor test in current loaded codeStorage.
     */
    public void saveScript() {
        if (log.isDebugEnabled()) {
            log.debug("saveScript called on " + codeStorage.getName());
        }

        try {
            //String content = scriptUI.getEditor().getText();
            //codeStorage.setContent(content);
            // setContent() or scriptUI.getEditor().save()
            // if setContent() only editor ask for t saving
            scriptUI.getEditor().save();
            
            // notify tree to refresh
            ScriptTreeModel model = (ScriptTreeModel) scriptUI.getTree().getModel();
            model.fileModified(codeStorage.getFile());
        } catch (Exception eee) {
            returnError(_("isisfish.error.script.save", codeStorage.getFile(),
                    eee.getMessage()), eee);
        }
        setStatusMessage(_("isisfish.message.save.finished"));
    }

    /**
     * Save script, and display commit UI.
     */
    public void commitScript() {

        if (log.isDebugEnabled()) {
            log.debug("commitScript called for " + codeStorage.getName());
        }

        try {
            // save script before commit
            saveScript();

            String msg = JOptionPane.showInputDialog(_("isisfish.message.script.commit", codeStorage.getName()));
            if (msg == null) {
                setStatusMessage(_("isisfish.message.commit.cancelled"));
            } else {
                codeStorage.commit(msg);
                codeStorage.reload();
                setStatusMessage(_("isisfish.message.commit.finished"));
            }
        } catch (Exception ex) {
            if (log.isErrorEnabled()) {
                log.error("Error on script commit", ex);
            }
            
            // if vcs can't write
            ErrorHelper.showErrorDialog(ex.getMessage(), ex);
        }
    }

    /**
     * Exporte le(s) script(s) sélectionnés dans l'arbre.
     * <br>L'arbre doit avoir au moins un script de selectionnés
     */
    public void exportScript() {
        
        TreePath[] selectedFilesPath = scriptUI.getTree().getSelectionPaths();

        // first step : acquire list of files required 
        int prefixLength = IsisFish.config.getDatabaseDirectory()
                .getAbsolutePath().length() + 1;
        List<String> listFiles = extractFiles(prefixLength, selectedFilesPath);
        
    }

    protected static List<String> extractFiles(int prefixLength,
            TreePath[] selectedPaths) {
        List<String> result = new ArrayList<String>();
        List<File> dirFound = new ArrayList<File>();
        List<File> dirWithFileFound = new ArrayList<File>();

        for (TreePath selectedPath : selectedPaths) {

            DefaultMutableTreeNode node = (DefaultMutableTreeNode) selectedPath
                    .getPathComponent(1);
            String moduleDisplayName = String.valueOf(node.getUserObject());
            File file = ScriptMapping.valueOf(moduleDisplayName).getModule();
            int nbPaths = selectedPath.getPathCount();
            if (nbPaths > 2)
                for (int i = 2; i < nbPaths; i++) {
                    node = (DefaultMutableTreeNode) selectedPath
                            .getPathComponent(i);
                    String pathName = String.valueOf(node.getUserObject());
                    file = new File(file, pathName);
                }
            if (file.isFile()) {
                File parentFile = file.getParentFile();
                if (!dirFound.contains(parentFile)) {
                    dirFound.add(parentFile);
                }
                dirWithFileFound.add(parentFile);
                result.add(file.getAbsolutePath().substring(prefixLength));
            } else {
                // mark the file
                dirFound.add(file);
            }
        }

        // keep only user selected directories
        dirFound.removeAll(dirWithFileFound);
        dirWithFileFound.clear();

        if (!dirFound.isEmpty()) {
            List<File> listF = new ArrayList<File>();
            // there is some directories selected by user
            for (File dir : dirFound) {
                FileFilter filter = new FileFilter() {
                    FileFilter excludeFilter = getScriptFileFilter();

                    public boolean accept(File pathname) {
                        return !excludeFilter.accept(pathname);
                    }
                };
                listF.addAll(FileUtil.getFilteredElements(dir, filter, true));
            }
            for (File file : listF)
                result.add(file.getAbsolutePath().substring(prefixLength));
            listF.clear();
        }
        dirFound.clear();
        return result;
    }

    /**
     * Cut selection in current editor.
     */
    public void cut() {
        scriptUI.getEditor().cut();
    }

    /**
     * Copy selection in current editor.
     */
    public void copy() {
        scriptUI.getEditor().copy();
    }

    /**
     * Paste clipboard content in editor.
     */
    public void paste() {
        scriptUI.getEditor().paste();
    }

    protected static FileFilter scriptFileFilter;

    public static FileFilter getScriptFileFilter() {
        if (scriptFileFilter == null) {
            scriptFileFilter = new ScriptFileFilter(IsisFish.vcs);
        }
        return scriptFileFilter;
    }

    //public static Object updateScript()
    public void updateScript() {
        try {
            log.debug("updateScript called for ");
            //TODO Use VCS UI dialog
            UpdateDialogUI ui = new UpdateDialogUI();
            ui.setVisible(true);
            //TODO Use FileState new mechanism to obtain state
            if (codeStorage != null) {
                codeStorage.update();
                codeStorage.reload();
            }
        } catch (VCSException ex) {
            if (log.isErrorEnabled()) {
                log.error("Error on script update", ex);
            }
        }
    }

    /**
     * Check script content.
     * 
     * @return compilation success flag
     */
    public boolean checkScript() {

        boolean check = false;
        String content = scriptUI.getEditor().getText();

        try {
            if (log.isDebugEnabled()) {
                log.debug("checkScript called");
            }
            // save script before commit
            codeStorage.setContent(content);
            JavaSourceStorage javaCode = (JavaSourceStorage) codeStorage;
            StringWriter result = new StringWriter();
            PrintWriter out = new PrintWriter(result);
            try {
                int compileResult = javaCode.compile(false, out);

                if (compileResult == 0) {
                    check = true;
                }
            } catch (Exception eee) {
                eee.printStackTrace(out);
            }
            out.flush();

            if (check) {
                scriptUI.getActionLogArea().setText(
                        _("isisfish.script.compilation.ok", result.toString()));
                scriptUI.getActionLogArea().setBackground(COLOR_SUCCESS);
            } else {
                scriptUI.getActionLogArea().setText(
                        _("isisfish.script.compilation.failed", result
                                .toString()));
                scriptUI.getActionLogArea().setBackground(COLOR_FAILURE);
            }

        } catch (IOException ex) {
            if (log.isErrorEnabled()) {
                log.error("Error on script check", ex);
            }
        }
        setStatusMessage(_("isisfish.message.check.finished"));
        return check;
    }

    /**
     * Call main method in current cod storage code.
     * Check script before call.
     */
    public void evaluateScript() {

        if (log.isDebugEnabled()) {
            log.debug("evaluateScript called");
        }

        try {
            if (checkScript()) {
                // reset area color
                scriptUI.getActionLogArea().setBackground(null);

                JavaSourceStorage javaCode = (JavaSourceStorage) codeStorage;
                ByteArrayOutputStream result = new ByteArrayOutputStream();
                PrintStream out = new PrintStream(result);
                PrintStream err = new PrintStream(result);
                PrintStream oldOut = System.out;
                PrintStream oldErr = System.err;
                System.setOut(out);
                System.setErr(err);
                Class<?> clazz = javaCode.getCodeClass();
                Method main = clazz.getMethod("main", String[].class);
                //noinspection RedundantArrayCreation
                main.invoke(null, new Object[] { new String[] {} });
                System.setOut(oldOut);
                System.setErr(oldErr);
                scriptUI.getActionLogArea().setText(result.toString());
            }
        } catch (Exception ex) {
            if (log.isDebugEnabled()) {
                log.debug("Error on script evaluation", ex);
            }
            ByteArrayOutputStream result = new ByteArrayOutputStream();
            PrintStream out = new PrintStream(result);
            ex.printStackTrace(out);
            scriptUI.getActionLogArea().setText(result.toString());
        }
        setStatusMessage(_("isisfish.message.evaluation.finished"));
    }

    /**
     * Show a improved script dialog with multiple selection support.
     */
    public void importScript() {

        if (log.isDebugEnabled()) {
            log.debug("importScript ");
        }

        File root = IsisFish.config.getDatabaseDirectory();
        try {
            // ask user form a .zip file
            File file = FileUtil.getFile(".*.zip$",
                    _("isisfish.message.import.scripts.zipped"));
            if (file != null) {

                //frame.setInfoText(_("isisfish.message.import.scripts.file", file));

                // get two list of relative path (one for new files,
                // one for exisiting files)
                List<String>[] explode = scanZip(file, root);
                List<String> newFiles = explode[0];
                List<String> conflictFiles = explode[1];

                // build model for dialog (list of new items), (list of exisiting items)
                FileSelectionTableModel modelNewItems = null;
                FileSelectionTableModel modelItems = null;

                // si'il y a des nouveau fichiers
                //if (!newFiles.isEmpty()) {
                modelNewItems = new FileSelectionTableModel(newFiles);
                //}

                // s'il y a des fichier en conflic
                //if (!conflictFiles.isEmpty()) {
                modelItems = new FileSelectionTableModel(conflictFiles);
                //}

                // create import dialog
                // TODO do a better code
                ImportDialogUI dialog = new ImportDialogUI();
                dialog.setNewFilesTableModel(modelNewItems);
                dialog.setConflictFilesTableModel(modelItems);
                dialog.getConflictFilesTable().setModel(modelItems);
                dialog.getNewFilesTable().setModel(modelNewItems);
                // can't be in ui :(
                dialog.getNewFilesTable().getColumnModel().getColumn(0)
                        .setWidth(30);
                dialog.getConflictFilesTable().getColumnModel().getColumn(0)
                        .setWidth(30);
                dialog.getArchivePath().setText(file.getAbsolutePath());
                dialog.refresh();
                dialog.pack();
                SwingUtil.center(dialog);
                dialog.setVisible(true);
                setStatusMessage(_("isisfish.message.import.scripts.file.done", file));
            } else {
                setStatusMessage(_("isisfish.message.import.scripts.file.cancelled"));
            }
        } catch (Exception eee) {
            returnError(_("isisfish.error.script.import", eee.getMessage()),
                    eee);
        }
    }

    /**
     * Perform import script action called by {@link ImportDialogUI}.
     * 
     * @param sourceScriptArchive source archive file
     * @param selectedFiles selected non conflict files model
     * @param selectedConflictFiles selected conflict files model
     */
    public void performImportScript(File sourceScriptArchive,
            FileSelectionTableModel selectedFiles,
            FileSelectionTableModel selectedConflictFiles) {

        File sourceDirectory = IsisFish.config.getDatabaseDirectory();

        List<String> filesToImports = new ArrayList<String>();
        if (selectedFiles != null) {
            filesToImports.addAll(selectedFiles.getSelectedFiles());
        }

        if (selectedConflictFiles != null) {
            // do backup stuff (only backup conflict files)
            backupScripts(selectedConflictFiles.getSelectedFiles(),
                    sourceDirectory);

            filesToImports.addAll(selectedConflictFiles.getSelectedFiles());
        }

        // unflate files to destination (root)
        try {
            if (log.isInfoEnabled()) {
                for (String newFile : filesToImports) {
                    log.info("unzip " + newFile);
                }
            }
            ZipUtil.uncompress(sourceScriptArchive, sourceDirectory,
                    filesToImports, null, null);

        } catch (IOException e1) {
            throw new RuntimeException(e1);
        }
        if (log.isInfoEnabled()) {
            log.info("Unzip " + filesToImports.size() + " entry(ies) from '"
                    + sourceScriptArchive + "'");
        }
    }

    /**
     * Make a backup a given files to current isis backup directory.
     * 
     * @param paths
     */
    protected void backupScripts(List<String> paths, File destination) {

        File backupDirectory = IsisFish.config.getBackupSessionDirectory();

        if (log.isDebugEnabled()) {
            log.debug("Backup directory is : " + backupDirectory);
        }
        String suffix = "_"
                + IsisFish.config.getBackupSessionDirectory().getName();
        try {
            for (String path : paths) {

                //backup(new File(src, path), path, suffix);

                File sourceFile = new File(destination, path);

                if (!sourceFile.exists()) {
                    return;
                }
                if (sourceFile.isDirectory()) {
                    File dst = new File(backupDirectory, path);
                    dst.mkdirs();
                } else {
                    int extension = path.lastIndexOf(".");
                    if (extension != -1) {
                        path = path.substring(0, extension) + suffix + "."
                                + path.substring(extension + 1);
                    } else {
                        path += suffix;
                    }
                    File dst = new File(destination, path);
                    if (!dst.getParentFile().exists()) {
                        dst.getParentFile().mkdirs();
                    }
                    if (log.isInfoEnabled()) {
                        log.info("Copying " + sourceFile.getName() + " to "
                                + dst);
                    }
                    FileUtil.copy(sourceFile, dst);
                }
            }
        } catch (Exception ee) {
            throw new IsisFishRuntimeException(ee);
        }
    }

    /**
     * Copy a single file to destination directory.
     * 
     * @param src file to copy
     * @param path path of file to copy
     * @param suffix file suffix
     * @throws IOException if copy fail
     */
    protected void backup(File src, String path, String suffix)
            throws IOException {

    }

    /**
     * Scan archive for script, and return an array of new files and conflict files.
     * 
     * @param source archive file
     * @param root database directory
     * @return an array [newFiles, conflictFiles]
     */
    protected static List<String>[] scanZip(File source, File root) {

        List<String> overwrittenFiles = new ArrayList<String>();
        List<String> newFiles = new ArrayList<String>();

        // ontain list of relative paths (to add or overwrite)
        try {
            ZipUtil.scan(source, root, newFiles, overwrittenFiles,
                    getScriptFileFilter(), null, null);
        } catch (IOException e) {
            log.error("Can't scan zip (" + source + ")", e);
            throw new RuntimeException(e);
        }

        return new List[] { newFiles, overwrittenFiles };
    }

    /**
     * Delete a script
     * 
     * @param deleteRemote {@code true} to remove in vcs too
     */
    public void deleteScript(boolean deleteRemote) {

        if (log.isDebugEnabled()) {
            log.debug("DeleteScript called");
        }

        String name = codeStorage.getName();
        int resp = JOptionPane.showConfirmDialog(scriptUI, _(
                "isisfish.message.confirm.remove.script", name), null,
                JOptionPane.YES_NO_OPTION);
        if (resp == JOptionPane.YES_OPTION) {
            // stay in UI even if deleted
            scriptUI.getEditor().close();

            try {
                // TODO change this, need to be called before
                // effective deletion
                ((ScriptTreeModel) scriptUI.getTree().getModel())
                        .fileDeleted(codeStorage.getFile());

                //TODO desactive editor
                //TODO Review this because after delete fi file saw previously
                //TODO modified, it ask if we want to save, and then we have
                //TODO again the file in panel but not in tree panel ?
                codeStorage.delete(deleteRemote);
                if (codeStorage.getFile().exists()) {
                    ErrorHelper.showErrorDialog(_(
                            "isisfish.error.script.delete", codeStorage
                                    .getFile()));
                }
            } catch (Exception eee) {
                returnError(_("isisfish.error.script.delete",
                        codeStorage == null ? null : codeStorage.getFile(), eee
                                .getMessage()), eee);
            }
            setStatusMessage(_("isisfish.message.delete.finished"));
        } else {
            setStatusMessage(_("isisfish.message.delete.canceled"));
        }
    }

    /**
     * Show diff between selected files and files server version.
     */
    public void diffScript() {

        if (log.isDebugEnabled()) {
            log.debug("Method diffScript called on " + codeStorage.getFile());
        }

        try {
            String result = null;
            if (IsisFish.vcs.isOnRemote(codeStorage.getFile())) {
                result = IsisFish.vcs.getDiff(codeStorage.getFile());
            } else {
                result = "File not on remote";
            }
            scriptUI.getActionLogArea().setText(result);
        } catch (VCSException e) {
            if (log.isErrorEnabled()) {
                log.error("Can't get diff", e);
            }
            ErrorHelper.showErrorDialog(_("isisfish.vcs.vcssvn.diff.error"), e);
        }
    }

    /** enum to encapsulate a script module */
    protected enum ScriptMapping {

        Script(
                ScriptStorage.getScriptDirectory(),
                ScriptStorage.SCRIPT_TEMPLATE),
        Simulator(
                SimulatorStorage.getSimulatorDirectory(),
                SimulatorStorage.SIMULATOR_TEMPLATE),
        Export(
                ExportStorage.getExportDirectory(),
                ExportStorage.EXPORT_TEMPLATE),
        Rule(
                RuleStorage.getRuleDirectory(),
                RuleStorage.RULE_TEMPLATE),
        AnalysePlan(
                AnalysePlanStorage.getAnalysePlanDirectory(),
                AnalysePlanStorage.ANALYSE_PLAN_TEMPLATE),
        Sensitivity(
                SensitivityStorage.getSensitivityDirectory(),
                SensitivityStorage.SENSIVITY_TEMPLATE),
        SensitivityExport(
                SensitivityExportStorage.getSensitivityExportDirectory(),
                SensitivityExportStorage.SENSITIVITY_EXPORT_TEMPLATE),
        EquationModel(
                FormuleStorage.getFormuleDirectory(),
                FormuleStorage.FORMULE_TEMPLATE);

        protected File module;
        protected String templatePath;

        private ScriptMapping(File module, String templatePath) {
            this.module = module;
            this.templatePath = templatePath;
        }

        /**
         * Get script type for script path.
         * 
         * @param file file to get type
         * @return ScriptMapping type
         */
        public static ScriptMapping getMappingFor(File file) {

            ScriptMapping result = null;

            // test if path starts with type begin path
            
            // don't forget last / for distinction begin
            // "sensitivity" and "sensitivityexport"

            for (ScriptMapping mapping : ScriptMapping.values()) {
                if (file.getAbsolutePath().startsWith(mapping.getModule().getAbsolutePath() + File.separator)) {
                    result = mapping;
                }
            }

            return result;
        }

        public File getModule() {
            return module;
        }

        public String getTemplatePath() {
            return templatePath;
        }
    }

    /**
     * Generate javadoc and display output in UI.
     */
    public void generateScriptJavadoc() {
        setStatusMessage(_("isisfish.script.menu.javadocgenerating",
                fr.ifremer.isisfish.IsisFish.config.getJavadocDirectory()),
                true);

        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                File rootDatabase = IsisFish.config.getDatabaseDirectory();
                File javadocDirectory = IsisFish.config.getJavadocDirectory();

                StringWriter output = new StringWriter();
                PrintWriter out = new PrintWriter(output);
                int ok = JavadocHelper.generateJavadoc(rootDatabase,
                        javadocDirectory, out);

                if (ok == 0) {
                    scriptUI.getActionLogArea().setText(
                            _("isisfish.script.javadoc.ok", output.toString()));
                    // vert leger
                    scriptUI.getActionLogArea().setBackground(COLOR_SUCCESS);
                } else {
                    scriptUI.getActionLogArea().setText(
                            _("isisfish.script.compilation.failed", output
                                    .toString()));
                    // rouge leger
                    scriptUI.getActionLogArea().setBackground(COLOR_FAILURE);
                }

                setStatusMessage(_("isisfish.script.menu.javadocgenerated",
                        fr.ifremer.isisfish.IsisFish.config.getJavadocDirectory()));
            }
        });
    }

    /**
     * Open a browser displaying javadoc.
     */
    public void showScriptJavadoc() {
        try {
            // in faut ouvrir l'index, sinon, ca ouvre
            // un explorateur de fichier
            File indexFile = new File(IsisFish.config.getJavadocDirectory(),
                    "index.html");

            URI uri = indexFile.toURI();
            Desktop.getDesktop().browse(uri);
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't show script javadocs", e);
            }
        }
    }

    /*
     * @see javax.swing.event.CaretListener#caretUpdate(javax.swing.event.CaretEvent)
     */
    @Override
    public void caretUpdate(CaretEvent e) {
        // selection pas vide si dot = mark
        scriptUI.setTextSelected(e.getDot() != e.getMark());
    }
}
