/* *##%
 * Copyright (C) 2009 Code Lutin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package fr.ifremer.isisfish.ui.script;

import static org.nuiton.i18n.I18n._;

import java.util.ArrayList;
import java.util.List;

import javax.swing.table.AbstractTableModel;

/**
 * File selection table model displayed in import/export script
 * for files selection (first column is checkbox).
 *
 * @author chatellier
 * @version $Revision: 1.0 $
 * 
 * Last update : $Date: 23 juil. 2009 $
 * By : $Author: chatellier $
 */
public class FileSelectionTableModel extends AbstractTableModel {

    /** serialVersionUID. */
    private static final long serialVersionUID = 6327223042377741126L;

    /** Column names. */
    public final static String[] COLUMN_NAMES = { _("isisfish.common.ellipsis"),
            _("isisfish.common.module"), _("isisfish.common.file") };

    /** All files displayed in table. */
    protected List<String> availableFiles;
    
    /** Only selected files. */
    protected List<String> selectedFiles;

    /**
     * Constructor.
     * 
     * @param availableFiles available file list
     */
    public FileSelectionTableModel(List<String> availableFiles) {
        this.availableFiles = availableFiles;
        this.selectedFiles = new ArrayList<String>(availableFiles);
    }

    /*
     * @see javax.swing.table.TableModel#getColumnCount()
     */
    @Override
    public int getColumnCount() {
        return COLUMN_NAMES.length;
    }

    /*
     * @see javax.swing.table.AbstractTableModel#getColumnName(int)
     */
    @Override
    public String getColumnName(int column) {
        return COLUMN_NAMES[column];
    }

    /*
     * @see javax.swing.table.AbstractTableModel#getColumnClass(int)
     */
    @Override
    public Class<?> getColumnClass(int columnIndex) {
        Class<?> result = null;

        switch (columnIndex) {
        case 0:
            result = Boolean.class;
            break;
        default:
            result = String.class;
            break;
        }

        return result;
    }

    /*
     * @see javax.swing.table.AbstractTableModel#isCellEditable(int, int)
     */
    @Override
    public boolean isCellEditable(int rowIndex, int columnIndex) {
        // seulement les cases a cocher sont editables
        return columnIndex == 0;
    }

    /*
     * @see javax.swing.table.TableModel#getRowCount()
     */
    @Override
    public int getRowCount() {
        return availableFiles.size();
    }

    /*
     * @see javax.swing.table.TableModel#getValueAt(int, int)
     */
    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {

        Object result = null;

        String rowFileName = availableFiles.get(rowIndex);
        switch (columnIndex) {
        case 0:
            result = selectedFiles.contains(rowFileName);
            break;
        case 1:
            if (rowFileName.contains(java.io.File.separator)) {
                result = rowFileName.substring(0, rowFileName
                        .lastIndexOf(java.io.File.separator));
            } else {
                result = rowFileName;
            }
            break;
        case 2:
            if (rowFileName.contains(java.io.File.separator)) {
                result = rowFileName.substring(rowFileName
                        .lastIndexOf(java.io.File.separator) + 1);
            } else {
                result = "";
            }
            break;
        }
        return result;
    }

    /*
     * @see javax.swing.table.AbstractTableModel#setValueAt(java.lang.Object, int, int)
     */
    @Override
    public void setValueAt(Object value, int rowIndex, int columnIndex) {

        String rowFileName = availableFiles.get(rowIndex);
        switch (columnIndex) {
        case 0:
            Boolean booleanValue = (Boolean) value;
            if (booleanValue) {
                selectedFiles.add(rowFileName);
            } else {
                selectedFiles.remove(rowFileName);
            }
            break;
        default:
            throw new RuntimeException("Can't edit that column");
        }
    }

    /**
     * Set all files selected.
     * 
     * @param selection {@code true} to select all files, {@code false} to unselect all files
     */
    public void setAllChecked(boolean selection) {
        if (selection) {
            selectedFiles.addAll(availableFiles);
        }
        else {
            selectedFiles.clear();
        }
        fireTableDataChanged();
    }
    
    /**
     * Get current files selection.
     * 
     * @return list of selected file path
     */
    public List<String> getSelectedFiles() {
        return selectedFiles;
    }
}
