/* *##%
 * Copyright (C) 2009 Code Lutin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package fr.ifremer.isisfish.ui.models.analyzeplan;

import static org.nuiton.i18n.I18n._;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.swing.table.AbstractTableModel;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import fr.ifremer.isisfish.IsisFishException;
import fr.ifremer.isisfish.datastore.AnalysePlanStorage;
import fr.ifremer.isisfish.simulator.AnalysePlan;

/**
 * Model de la table de definition de parametre d'un plan d'analyse.
 * 
 * Columns :
 * <li>Parameter name</li>
 * <li>Parameter value</li>
 *
 * @author chatellier
 * @version $Revision: 2658 $
 * 
 * Last update : $Date: 2009-10-13 17:00:50 +0200 (mar., 13 oct. 2009) $
 * By : $Author: chatellier $
 */
public class AnalyzePlanParametersTableModel extends AbstractTableModel {

    /** Log. */
    private static Log log = LogFactory
            .getLog(AnalyzePlanParametersTableModel.class);

    /** serialVersionUID. */
    private static final long serialVersionUID = 3169786638868209920L;

    /** Columns names. */
    public final static String[] COLUMN_NAMES = { _("isisfish.common.name"),
            _("isisfish.common.value") };

    protected AnalysePlan plan;

    protected List<String> planParametersNames;

    /**
     * Constructor with data.
     *  
     * @param plan plan to edit parameters
     */
    public AnalyzePlanParametersTableModel(AnalysePlan plan) {
        super();
        this.plan = plan;

        //TODO map may be sorted
        planParametersNames = new ArrayList<String>();
        Map<String, Class<?>> planParametersNamesAndTypes = AnalysePlanStorage
                .getParameterNames(plan);
        for (String names : planParametersNamesAndTypes.keySet()) {
            planParametersNames.add(names);
        }
    }

    /*
     * @see javax.swing.table.TableModel#getColumnCount()
     */
    @Override
    public int getColumnCount() {
        return COLUMN_NAMES.length;
    }

    /*
     * @see javax.swing.table.TableModel#getRowCount()
     */
    @Override
    public int getRowCount() {
        return planParametersNames.size();
    }

    /*
     * @see javax.swing.table.TableModel#getValueAt(int, int)
     */
    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {

        Object result = null;

        String name = planParametersNames.get(rowIndex);
        switch (columnIndex) {
        case 0:
            result = name;
            break;
        case 1:
            try {
                result = AnalysePlanStorage.getParameterValue(plan, name);
            } catch (IsisFishException e) {
                if (log.isErrorEnabled()) {
                    log.debug("Can't get parameters value", e);
                }
            }
            break;
        default:
            throw new IndexOutOfBoundsException("No such column " + columnIndex);
        }

        return result;
    }

    /*
     * @see javax.swing.table.TableModel#getColumnClass(int)
     */
    @Override
    public Class<?> getColumnClass(int columnIndex) {

        Class<?> result = null;

        switch (columnIndex) {
        case 0:
            result = String.class;
            break;
        case 1:
            result = Object.class;
            break;
        default:
            throw new IndexOutOfBoundsException("No such column " + columnIndex);
        }

        return result;
    }

    /*
     * @see javax.swing.table.TableModel#getColumnName(int)
     */
    @Override
    public String getColumnName(int columnIndex) {
        return COLUMN_NAMES[columnIndex];
    }

    /*
     * @see javax.swing.table.TableModel#isCellEditable(int, int)
     */
    @Override
    public boolean isCellEditable(int rowIndex, int columnIndex) {
        return columnIndex > 0;
    }

    /*
     * @see javax.swing.table.TableModel#setValueAt(java.lang.Object, int, int)
     */
    @Override
    public void setValueAt(Object value, int rowIndex, int columnIndex) {

        if (log.isDebugEnabled()) {
            log.debug("Cell edition (column " + columnIndex + ") = " + value);
        }

        String name = planParametersNames.get(rowIndex);
        switch (columnIndex) {
        case 1:
            try {
                AnalysePlanStorage.setParameterValue(plan, name, value);
            } catch (IsisFishException e) {
                if (log.isErrorEnabled()) {
                    log.error("Can't set parameter value", e);
                }
            }
            break;
        default:
            throw new IndexOutOfBoundsException("Can't edit column "
                    + columnIndex);
        }

    }
}
