/* *##%
 * Copyright (C) 2005 - 2009
 *     Ifremer, Code Lutin, Cedric Pineau, Benjamin Poussin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package fr.ifremer.isisfish.ui.input;

import static org.nuiton.i18n.I18n._;

import java.awt.Component;

import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTree;
import javax.swing.event.TreeSelectionEvent;

import jaxx.runtime.Decorator;
import jaxx.runtime.JAXXContextEntryDef;
import jaxx.runtime.swing.CardLayout2;
import jaxx.runtime.swing.JAXXTree;
import jaxx.runtime.swing.navigation.NavigationTreeModel;
import jaxx.runtime.swing.navigation.NavigationTreeModelBuilder;
import jaxx.runtime.swing.navigation.NavigationTreeSelectionAdapterWithCardLayout;
import jaxx.runtime.swing.navigation.NavigationTreeModel.NavigationTreeNode;
import jaxx.runtime.swing.navigation.NavigationTreeModelBuilder.ChildBuilder;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.persistence.TopiaEntity;

import fr.ifremer.isisfish.entities.Cell;
import fr.ifremer.isisfish.entities.FisheryRegion;
import fr.ifremer.isisfish.entities.Gear;
import fr.ifremer.isisfish.entities.Metier;
import fr.ifremer.isisfish.entities.Population;
import fr.ifremer.isisfish.entities.Port;
import fr.ifremer.isisfish.entities.SetOfVessels;
import fr.ifremer.isisfish.entities.Species;
import fr.ifremer.isisfish.entities.TripType;
import fr.ifremer.isisfish.entities.VesselType;
import fr.ifremer.isisfish.entities.Zone;

/**
 * InputNavigationTreeSelectionAdapter.
 *
 * @author letellier
 * @version $Revision: 1312 $
 *
 * Last update: $Date: 2008-08-28 10:21:07 +0200 (jeu, 28 aoû 2008) $
 * by : $Author: sletellier $
 */
public class InputNavigationTreeSelectionAdapter extends NavigationTreeSelectionAdapterWithCardLayout {

    /** Class logger. */
    private static final Log log = LogFactory.getLog(InputNavigationTreeSelectionAdapter.class);

    protected NavigationTreeNode currentNode = null;
    protected InputContentUI currentUI = null;

    public InputNavigationTreeSelectionAdapter(InputUI context) {
        super(InputUI.class, null, context, Strategy.PER_UI_TYPE);

        if (context.getNavigation() == null) {
            throw new IllegalArgumentException("could not have a null 'navigation' in ui " + context);
        }

        // register in tree this listener
        context.getNavigation().addTreeSelectionListener(this);
    }

    @Override
    protected JPanel getContentContainer() {
        return ((InputUI) context).getInputPane();
    }

    @Override
    protected CardLayout2 getContentLayout() {
        return ((InputUI) context).getCardlayout();
    }

    @Override
    protected NavigationTreeModel getNavigationTreeModel() {
        return (NavigationTreeModel) ((InputUI) context).getNavigation().getModel();
    }

    @Override
    protected void goBackToPreviousNode(TreeSelectionEvent event, Exception e) {
        if (e != null) {
            if (log.isErrorEnabled()) {
                log.error("Error happen, go back to previous node" ,e);
            }
        }
        // go back to previous node
        returnToPreviousNode(getNavigationTree(), event);
        if (log.isInfoEnabled()) {
            log.info("goBackToPreviousNode");
        }
    }


    protected JAXXTree getNavigationTree() {
        return ((InputUI) context).getNavigation();
    }

    @Override
    protected void openUI(Component newUI, NavigationTreeNode node) throws Exception {
        super.openUI(newUI, node);
        // apply refresh method (should a contract for all this ui with a method refresh...)
        try {
            currentUI = (InputContentUI) newUI;
            currentUI.setSensitivity(false);
            InputSaveVerifier verifier = context.getContextValue(InputSaveVerifier.class);
            verifier.removeAllPanels();
            verifier.addCurrentPanel(currentUI);
            currentUI.refresh();
            // tabbed ui hack 
            currentUI.setActionButtons();
            currentNode = node;
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't open ui", e);
            }
        }
    }

    @Override
    protected boolean closeUI(TreeSelectionEvent event, Component component) throws Exception {
        boolean exit = true;
        // by default, we says that component was succesfull closed
        InputSaveVerifier verif = context.getContextValue(InputSaveVerifier.class);
        int responce = verif.checkEdit();
        //boolean exit = checkEdit(event, component);
        if (responce == JOptionPane.OK_OPTION) {
            JTree tree = (JTree) event.getSource();
            NavigationTreeModel model = (NavigationTreeModel) tree.getModel();
            if (currentNode != null) {
                model.nodeChanged(currentNode);
            }
        }
        else if (responce == JOptionPane.CANCEL_OPTION) {
            exit = false;
        }
        if (exit) {
            verif.reset();
        }
        return exit;
    }

    @Override
    protected void addSelectedBeanInContext(NavigationTreeNode node, Object data) {
        if (log.isDebugEnabled()) {
            log.debug("find data for contextPath <" + node.getContextPath() + "> : " + (data == null ? null : data.getClass()));
        }
        InputSaveVerifier verif = context.getContextValue(InputSaveVerifier.class);
        verif.setRootPanel((InputUI)context);
        verif.removeAllEntity();
        if (TopiaEntity.class.isInstance(data)){
            verif.addCurrentEntity((TopiaEntity) data);
        }
        verif.setCurrentNode(node);
    }

    public static NavigationTreeModel getTreeModel(String regionName, FisheryRegion fisheryRegion) {

        NavigationTreeModelBuilder builder = new NavigationTreeModelBuilder("/");

        if (regionName == null) {
            return null;
        }

        ChildBuilder<TopiaEntity> childBuilder = new ChildBuilder<TopiaEntity>(builder) {

            Decorator<? extends TopiaEntity> decorator;

            @Override
            public void init(Class<? extends TopiaEntity> klass) {
                decorator = jaxx.runtime.PropertyDecorator.newDecorator(klass, "name");
            }

            @Override
            public Decorator<? extends TopiaEntity> getDecorator(TopiaEntity child) {
                return decorator;
            }

            @Override
            public String getJXPath(TopiaEntity child) {
                return "..[@topiaId=\"" + child.getTopiaId() + "\"]";
            }

            @Override
            public String getNavigationPath(TopiaEntity child) {
                return child.getTopiaId();
            }
        };

        try {

            NavigationTreeNode root = builder.build(null, regionName, JAXXContextEntryDef.newDef(FisheryRegion.class), "$root", FisheryRegionUI.class, null);

            childBuilder.build(builder.build(root, _("isisfish.input.tree.cells"), "../cell", "$cells", CellUI.class, null),
                    true, Cell.class, fisheryRegion.getCell(), CellUI.class, null);

            childBuilder.build(builder.build(root, _("isisfish.input.tree.zones"), "../zone", "$zones", ZoneUI.class, null),
                    true, Zone.class, fisheryRegion.getZone(), ZoneUI.class, null);

            childBuilder.build(builder.build(root, _("isisfish.input.tree.ports"), "../port", "$ports", PortUI.class, null),
                    true, Port.class, fisheryRegion.getPort(), PortUI.class, null);

            NavigationTreeNode species = builder.build(root, _("isisfish.input.tree.species"), "../species", "$species", SpeciesUI.class, null);

            for (Species specie : fisheryRegion.getSpecies()) {
                NavigationTreeNode speciesChild = builder.build(species,jaxx.runtime.PropertyDecorator.newDecorator(Species.class, "name"),  "..[@topiaId=\"" + specie.getTopiaId() + "\"]", specie.getTopiaId(), SpeciesUI.class, null);
                NavigationTreeNode populations = builder.build(speciesChild, _("isisfish.input.tree.populations"), "../population", "$populations", PopulationUI.class, null);
                childBuilder.build(populations, true, Population.class, specie.getPopulation(), PopulationUI.class, null);
            }

            childBuilder.build(builder.build(root, _("isisfish.input.tree.gears"), "../gear", "$gears", GearUI.class, null),
                    true, Gear.class, fisheryRegion.getGear(), GearUI.class, null);

            childBuilder.build(builder.build(root, _("isisfish.input.tree.metiers"), "../metier", "$metiers", MetierUI.class, null),
                    true, Metier.class, fisheryRegion.getMetier(), MetierUI.class, null);

            childBuilder.build(builder.build(root, _("isisfish.input.tree.triptypes"), "../tripType", "$tripTypes", TripTypeUI.class, null),
                    true, TripType.class, fisheryRegion.getTripType(), TripTypeUI.class, null);

            childBuilder.build(builder.build(root, _("isisfish.input.tree.vesseltypes"), "../vesselType", "$vesselTypes", VesselTypeUI.class, null),
                    true, VesselType.class, fisheryRegion.getVesselType(), VesselTypeUI.class, null);

            childBuilder.build(builder.build(root, _("isisfish.input.tree.setofvessels"), "../setOfVessels", "$setOfVessels", SetOfVesselsUI.class, null),
                    true, SetOfVessels.class, fisheryRegion.getSetOfVessels(), SetOfVesselsUI.class, null);

            childBuilder.build(builder.build(root, _("isisfish.input.tree.strategies"), "../strategy", "$strategies", StrategyUI.class, null),
                    true, fr.ifremer.isisfish.entities.Strategy.class, fisheryRegion.getStrategy(), StrategyUI.class, null);

        } catch (Exception ex) {
            if (log.isErrorEnabled()) {
                log.error(ex.getMessage(), ex);
            }
        }
        return builder.getModel();
    }
}
