/* *##%
 * Copyright (C) 2009 Code Lutin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package fr.ifremer.isisfish.simulator.sensitivity.domain;

import java.io.Serializable;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;

import fr.ifremer.isisfish.simulator.sensitivity.Domain;
import fr.ifremer.isisfish.simulator.sensitivity.visitor.DomainVisitor;

/**
 * Domain discret.
 * 
 * Ensemble de valeurs.
 *
 * @param <E> type des valeurs gérées par le domaine
 * @param <F> type des labels
 * 
 * @author chatellier
 * @version $Revision: 1.0 $
 * 
 * Last update : $Date: 24 févr. 2009 $
 * By : $Author: chatellier $
 */
public class DiscreteDomain<E extends Serializable, F extends Serializable> implements Domain<E,F> {

    /** serialVersionUID. */
    private static final long serialVersionUID = -192647757737396585L;

    /**
     * Value for this domain.
     * 
     * Each "E" values is identified by a Serializable identifier.
     */
    protected SortedMap<F, E> values;
    
    /**
     * Constructor.
     */
    public DiscreteDomain() {
        values = new TreeMap<F, E>();
    }

    /**
     * Get domaine values.
     * 
     * @return the values
     */
    public SortedMap<F, E> getValues() {
        return values;
    }

    /**
     * Set domain values.
     * 
     * @param values the values
     */
    public void setValues(SortedMap<F, E> values) {
        this.values = values;
    }

    /**
     * {@inheritDoc}.
     * 
     * @throws IllegalArgumentException if identifier is not a valid key
     */
    public E getValueForIdentifier(F identifier) throws IllegalArgumentException{
        
        if( values == null || !values.containsKey(identifier)) {
            throw new IllegalArgumentException("Can't get value for identifier " + identifier);
        }
        
        return values.get(identifier);
    }
    
    /**
     * Accept a new visitor.
     * 
     * @param visitor
     */
    public void accept(DomainVisitor visitor) {
        visitor.start(this);
        for(Map.Entry<F, E> value : values.entrySet()) {
            visitor.visit(this, value.getKey(), value.getValue());
        }
        visitor.end(this);
    }
    
    public DiscreteDomain<E,F> clone(){
        DiscreteDomain<E,F> cloned = new DiscreteDomain<E,F>();
        cloned.setValues(this.values);
        return cloned;
        
    }
}
