/* *##%
 * Copyright (C) 2009 Code Lutin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package fr.ifremer.isisfish.simulator.sensitivity;

import java.io.Serializable;

import fr.ifremer.isisfish.simulator.sensitivity.domain.ContinuousDomain;
import fr.ifremer.isisfish.simulator.sensitivity.domain.DiscreteDomain;
import fr.ifremer.isisfish.simulator.sensitivity.visitor.FactorVisitor;

/**
 * Facteur de variation des parametres de simulation.
 * 
 * @param <E> type des valeurs gérées par le facteur
 * @param <F> type des labels
 * 
 * La classe doit être {@link Serializable} avec ses valeurs pour permettre
 * l'export XML.
 * 
 * @author chatellier
 * @version $Revision: 1.0 $
 * 
 * Last update : $Date: 19 févr. 2009 $ By : $Author: chatellier $
 */
public class Factor<E extends Serializable, F extends Serializable> implements Serializable, Cloneable {

    /** serialVersionUID. */
    private static final long serialVersionUID = 1643804268013964453L;

    /**
     * Nom du facteur.
     */
    protected String name;

    /**
     * Commentaire du facteur.
     */
    protected String comment;

    /**
     * Domaine du facteur.
     * 
     * Ensemble des valeurs possibles a prendre en compte. La clé est un label
     * qui permet d'identifier la valeur.
     * 
     * Le domain peut etre : - discret : i.e un ensemble de valeurs - continu :
     * i.e, une borne min, max
     * 
     * @see ContinuousDomain
     * @see DiscreteDomain
     */
    protected Domain<E,F> domain;

    /**
     * Factor value.
     */
    protected E value;

    /**
     * Path permettant d'identifier l'objet et la propriete de l'objet a mettre
     * a jour.
     * 
     * Par exemple: topiaID#gear aura pour effet de recuperer l'objet
     * correspondant au topiaID fournit et d'appeler le propriete
     * <tt>setGear(value)</tt> dessus.
     */
    protected String path;

    /**
     * Constructor.
     * 
     * Protected, name is mandatory.
     */
    protected Factor() {

    }

    /**
     * Constructor with name.
     * 
     * @param name
     *            factor name
     */
    public Factor(String name) {
        this();
        this.name = name;
    }

    /**
     * Get name.
     * 
     * @return the name
     */
    public String getName() {
        return name;
    }

    /**
     * Set name.
     * 
     * @param name
     *            the name to set
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * Get comment.
     * 
     * @return the comment
     */
    public String getComment() {
        return comment;
    }

    /**
     * Set comment.
     * 
     * @param comment comment to set
     */
    public void setComment(String comment) {
        this.comment = comment;
    }

    /**
     * Get domain.
     * 
     * @return the domain
     */
    public Domain<E,F> getDomain() {
        return domain;
    }

    /**
     * Set domain.
     * 
     * @param domain
     *            the domain to set
     */
    public void setDomain(Domain<E,F> domain) {
        this.domain = domain;
    }

    /**
     * Get value.
     * 
     * @return the value
     */
    public E getValue() {
        return value;
    }

    /**
     * Set value.
     * 
     * @param value
     *            new value
     */
    public void setValue(E value) {
        this.value = value;
    }

    /**
     * Set value for label.
     * 
     * @param valueIdentifier
     *            new value identifier to get
     */
    public void setValueForIdentifier(F valueIdentifier) {
        value = domain.getValueForIdentifier(valueIdentifier);
    }

    /**
     * Get path.
     * 
     * @return the path
     */
    public String getPath() {
        return path;
    }

    /**
     * Set path.
     * 
     * Warning, path need to be always a valid entity property reference.
     * 
     * @param path the path to set
     */
    public void setPath(String path) {
        this.path = path;
    }

    /**
     * Accept a new visitor.
     * 
     * @param visitor
     */
    public void accept(FactorVisitor visitor) {
        visitor.start(this);
        visitor.visit(this, domain);
        visitor.end(this);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Object clone() {

        Factor<E,F> f = new Factor<E,F>();
        
        f.domain = domain.clone();
        f.name = name;
        f.path = path;
        // FIXME make real clone() for value
        f.value = value;

        return f;
    }

    @Override
    public String toString() {
        return name;
    }
}
