/* *##%
 * Copyright (C) 2009-2010 Ifremer, Code Lutin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package fr.ifremer.isisfish.simulator.sensitivity;

import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.List;

import fr.ifremer.isisfish.datastore.SimulationStorage;
import fr.ifremer.isisfish.export.SensitivityExport;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.j2r.REngine;
import org.nuiton.j2r.RException;
import org.nuiton.j2r.RProxy;

import fr.ifremer.isisfish.simulator.sensitivity.domain.ContinuousDomain;
import fr.ifremer.isisfish.simulator.sensitivity.domain.DiscreteDomain;
import fr.ifremer.isisfish.simulator.sensitivity.domain.EquationContinuousDomain;
import fr.ifremer.isisfish.simulator.sensitivity.domain.MatrixContinuousDomain;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.util.FileUtil;

import javax.swing.*;

public abstract class AbstractSensitivityCalculator implements
        SensitivityCalculator {

    /** to use log facility, just put in your code: log.info("..."); */
    private static Log log = LogFactory
            .getLog(AbstractSensitivityCalculator.class);

    /** Topia context used for discrete domain nominal value compute. */
    protected TopiaContext context;

    /**
     * Set topia context on current region to nominal value compute.
     * 
     * @param context context
     */
    @Override
    public void setTopiaContext(TopiaContext context) {
        this.context = context;
    }
    
    /**
     * Method to create the isis.factors dataframe in R.
     * 
     * @param plan Design plan of the sensitivity analysis
     * @param outputdirectory the directory on which we will calculate the
     * destination RData file. The RData file will be named directory.RData and
     * saved in the directory parent.
     * 
     * @throws SensitivityException
     */
    public void setIsisFactorsR(DesignPlan plan, File outputdirectory)
            throws SensitivityException {

        String nomFacteur = "nomFacteur<-c(%s)";
        String factorNames = "";
        String nominal = "Nominal<-c(%s)";
        String nominalValues = "";
        String continu = "Continu<-c(%s)";
        String continuValues = "";
        String binf = "Binf<-c(%s)";
        String binfValues = "";
        String bsup = "Bsup<-c(%s)";
        String bsupValues = "";
        String attribute = "attr(isis.factors,\"%s\")<-\"%s\"";
        String isisfactors = "isis.factors<-data.frame(" +
                "\"nomFacteur\"=nomFacteur," +
                "\"Nominal\"=Nominal," +
                "\"Continu\"=Continu," +
                "\"Binf\"=Binf," +
                "\"Bsup\"=Bsup)";

        //Create the vectors
        List<Factor<?, ?>> factors = plan.getFactors();
        for (int i = 0; i < factors.size(); i++) {
            Factor factor = factors.get(i);
            if (i != 0) {
                factorNames += ",";
                nominalValues += ",";
                continuValues += ",";
                binfValues += ",";
                bsupValues += ",";
            }

            //Factor names
            factorNames += "\"" + factor.getName() + "\"";

            if (factor.getDomain() instanceof MatrixContinuousDomain) {
                //Continu values, binf values, bsup values and discrete factors attributes

                nominalValues += getNominalValue(factor);

                //Continu values
                continuValues += "TRUE";

                //Binf values

                binfValues += ((MatrixContinuousDomain) factor.getDomain())
                        .getRealMinBound();

                //Bsup values
                bsupValues += ((MatrixContinuousDomain) factor.getDomain())
                        .getRealMaxBound();

            } else if (factor.getDomain() instanceof EquationContinuousDomain) {
                //Continu values, binf values, bsup values and discrete factors attributes

                nominalValues += ((EquationContinuousDomain) factor.getDomain())
                        .getReferenceValue();

                //Continu values
                continuValues += "TRUE";

                //Binf values
                binfValues += ((EquationContinuousDomain) factor.getDomain())
                        .getValueForIdentifier(0.0);

                //Bsup values
                bsupValues += ((EquationContinuousDomain) factor.getDomain())
                        .getValueForIdentifier(1.0);
            } else if (factor.getDomain() instanceof ContinuousDomain) {
                //Continu values, binf values, bsup values and discrete factors attributes

                nominalValues += getNominalValue(factor);

                //Continu values
                continuValues += "TRUE";

                //Binf values

                binfValues += ((ContinuousDomain) factor.getDomain())
                        .getMinBound();

                //Bsup values
                bsupValues += ((ContinuousDomain) factor.getDomain())
                        .getMaxBound();

            } else {
                nominalValues += getNominalValue(factor);

                //Continu values
                continuValues += "FALSE";

                //Binf values
                binfValues += "0.0";

                //Bsup values
                bsupValues += ((DiscreteDomain) factor.getDomain()).getValues()
                        .size();
            }
        }

        String Rcall;

        
        try {
            REngine engine = new RProxy();

            //clear the engine (to be sure it's empty)
            engine.clearSession();

            Rcall = String.format(nomFacteur, factorNames);
            engine.voidEval(Rcall);

            Rcall = String.format(nominal, nominalValues);
            engine.voidEval(Rcall);

            Rcall = String.format(continu, continuValues);
            engine.voidEval(Rcall);

            Rcall = String.format(binf, binfValues);
            engine.voidEval(Rcall);

            Rcall = String.format(bsup, bsupValues);
            engine.voidEval(Rcall);

            engine.voidEval(isisfactors);

            Rcall = String.format(attribute, "nomModel", "isis-fish-externeR");
            engine.voidEval(Rcall);

            for (Factor factor:factors) {
                if (factor.getDomain() instanceof DiscreteDomain) {
                    String attributeValues = "list(";
                    Collection values = ((DiscreteDomain) factor.getDomain())
                            .getValues().values();                    
                    for (Object j:values) {
                        attributeValues += j + ",";
                    }
                    attributeValues = attributeValues.substring(0,
                            attributeValues.length() - 1);
                    attributeValues += ")";

                    Rcall = String.format(attribute, factor.getName(),
                            attributeValues);
                    engine.voidEval(Rcall);
                }
            }

            //Save the Isis R session
            engine.saveRData(outputdirectory.getParentFile(),
                outputdirectory.getName());

        } catch (RException eee) {
            if (log.isErrorEnabled()) {
                log.error("R evaluation failed", eee);
            }
            throw new SensitivityException("R evaluation failed", eee);
            // Error while retrieving scenario
        }

    }

    protected String getNominalValue(Factor factor){
        String path = factor.getPath();
        String topiaId = path.substring(0,path.lastIndexOf("#"));
        String propertyName=path.substring(path.lastIndexOf("#") + 1);

        Domain domain =factor.getDomain();

        if (domain instanceof MatrixContinuousDomain){
            return "1.0";
        }

        Object value = null;
        try {
            TopiaEntity entity = context.findByTopiaId(topiaId);
            value = BeanUtils.getProperty(entity, propertyName);
        } catch (Exception ee) {
            log.error("An error occurred while trying to get nominal value",ee);
        }
        if (value == null){
            return null;
        }
        log.info("Nominal value :" + value);
        return value.toString();
    }

    protected String editRInstruction(String rCall){
        JLabel label = new JLabel(
                    "Modifier le code R envoyé si vous le souhaitez");
            JTextPane text = new JTextPane();
            text.setText(rCall);
            text.setSize(400, 400);
            text.setPreferredSize(text.getSize());

            Box box = Box.createVerticalBox();
            box.add(label);
            box.add(new JScrollPane(text));

            JOptionPane.showMessageDialog(null, box, "R modif",
                    JOptionPane.QUESTION_MESSAGE);
            return text.getText();        
    }

    protected void testNoDiscrete(List<Factor<?,?>> factors) throws SensitivityException{
        for (Factor factor:factors) {
            if (factor.getDomain() instanceof DiscreteDomain) {
                throw new SensitivityException(factor.getName() +
                        " has a discrete domain, this is not acceptable for " +
                        "this method.");
            }
        }
    }

    protected String createImportInstruction(
            SensitivityExport export,
            List<SimulationStorage> simulationStorages) {

        //the filename to read
        String name = export.getExportFilename();
        //the extension to read
        String extension = export.getExtensionFilename();
        //the rInstruction
        String rInstruction = name + "<-c(";

        //iterate on all the simulations and read the value for each one
        for (int l = 0; l < simulationStorages.size(); l++) {

            SimulationStorage simulationStorage = simulationStorages.get(l);
            File directory = simulationStorage.getDirectory();

            String directoryPath = directory.toString() + File.separator
                    + SimulationStorage.RESULT_EXPORT_DIRECTORY;

            File importFile = new File(directoryPath, name + extension);

            String simulResult="";
            try{
                simulResult = FileUtil.readAsString(importFile);
            } catch (IOException ioe) {
                log.error ("An error occured trying to read a result file : " +
                        ioe.getMessage());
            }


            double simulationResult = Double.valueOf(simulResult);

            if (l < simulationStorages.size() - 1) {
                rInstruction += simulationResult + ",";
            } else {
                rInstruction += simulationResult;
            }
        }
        rInstruction += ")";

        return rInstruction;
    }

    protected void renameObjects(REngine engine, String simulationName)
            throws RException {

        String renameIsisSimule = "%s.isis.simule<-isis.simule";
        String renameIsisFactorDistribution = "%s.isis.factor.distribution" +
                "<-isis.factor.distribution";
        String renameIsisFactor = "%s.isis.factor<-isis.factors";
        String renameIsisMethodExp = "%s.isis.methodExp<-isis.MethodExp";

        engine.voidEval(String.format(renameIsisSimule, simulationName));

        engine.voidEval(String.format(renameIsisFactorDistribution,
                simulationName));

        engine.voidEval(String.format(renameIsisFactor, simulationName));

        engine.voidEval(String.format(renameIsisMethodExp, simulationName));
    }

}
