/* *##%
 *  Copyright (C) 2002-2009 Ifremer, Code Lutin, Benjamin Poussin
 * 
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 *##%*/

package fr.ifremer.isisfish.simulator.launcher;

import static org.nuiton.i18n.I18n._;

import java.rmi.RemoteException;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import fr.ifremer.isisfish.datastore.SimulationStorage;
import fr.ifremer.isisfish.simulator.SimulationControl;
import fr.ifremer.isisfish.simulator.SimulationParameter;

/**
 * Classe responsable de la simulation d'un {@link SimulationItem}. Pour cela
 * il utilise le {@link SimulatorLauncher}. Si la simulation echoue
 * a cause d'une RemoteException alors le job est resoumis dans la queue
 * de simulation par l'appel de la methode
 * {@link SimulationService#reportError}.
 *
 * @author poussin
 * @version $Revision: 3065 $
 * 
 * Last update : $Date: 2010-06-18 12:05:57 +0200 (ven., 18 juin 2010) $
 * By : $Author: chatellier $
 */
public class SimulationJob implements Runnable, Comparable<SimulationJob> {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    protected static Log log = LogFactory.getLog(SimulationJob.class);

    transient protected String id;
    /** l'ensemble des post actions a effectuer pour ce job */
    protected Set<PostAction> postActions = new HashSet<PostAction>();
    /** Le {@link SimulationService} dans lequel a ete cree ce job */
    protected SimulationService simulationService;
    /** si non null contient le {@link SimulationJob} qui a genere ce job, ca
     * veut dire que ce job est du a un plan d'analyse*/
    protected SimulationJob parentJob;
    /** item contenant les infos de la simulation */
    protected SimulationItem item;
    /** la priorite de cet item dans la queue */
    protected int priority;
    /** Le launcher a utiliser pour simuler cet item */
    protected SimulatorLauncher launcher;
    /** Set it to true to restart simulation checking without restarting reel simulation */
    protected boolean onlyCheckControl;
    
    public SimulationJob(SimulationService simulationService,
            SimulationItem item, int priority) {
        this(simulationService, null, item, priority);
    }

    public SimulationJob(SimulationService simulationService,
            SimulationJob parentJob, SimulationItem item, int priority) {
        this.simulationService = simulationService;
        this.parentJob = parentJob;
        this.item = item;
        this.priority = priority;
    }

    public void addPostAction(PostAction postAction) {
        this.postActions.add(postAction);
    }

    public void removePostAction(PostAction postAction) {
        this.postActions.remove(postAction);
    }

    public Set<PostAction> getPostActions() {
        return this.postActions;
    }

    public String getId() {
        if (id == null) {
            id = getItem().getControl().getId();
        }
        return id;
    }

    /** 
     * demande l'annulation/arret de ce job. Si ce job n'etait pas encore actif
     * un {@link SimulationServiceListener#simulationStop} est leve. Sinon
     * il le sera lorsque la simulation se sera convenablement arretee.
     * <p>
     * Dans tous les cas une demande d'arret sur le control de la simulation
     * est fait.
     */
    public void stop() {
        // on essaie d'enlever ce job de la queue, au cas ou il ne serait pas
        // encore lance
        if (simulationService.cancel(this)) {
            // on a pu annuler avant le lancement, on notify un stop puisqu'on
            // ne passera jamais dans le run()
            simulationService.fireStopEvent(this);
        }
        item.getControl().setStopSimulationRequest(true);
    }
    
    /**
     * Resoumet un job.
     */
    public void restart() {
        simulationService.restart(this);
    }

    public SimulationJob getParentJob() {
        return parentJob;
    }

    public SimulationItem getItem() {
        return item;
    }

    public void setLauncher(SimulatorLauncher launcher) {
        this.launcher = launcher;
    }

    public SimulatorLauncher getLauncher() {
        return launcher;
    }

    public int getPriority() {
        return priority;
    }

    /**
     * @param onlyCheckControl the onlyCheckControl to set
     */
    public void setOnlyCheckControl(boolean onlyCheckControl) {
        this.onlyCheckControl = onlyCheckControl;
    }

    /**
     * L'ordre depend :
     *  - de la priorite
     *  - si le nom fini par un chiffre :
     *    - du nom avant le chiffre
     *    - du chiffre
     *  - sinon du nom
     */
    @Override
    public int compareTo(SimulationJob o) {
        int result = this.priority - o.priority;
        if (result == 0) {
            if (this.getId().matches(".*_\\d+") && o.getId().matches(".*_\\d+")) {
                String firstString = this.getId().substring(0, this.getId().lastIndexOf("_"));
                String secondString = o.getId().substring(0, o.getId().lastIndexOf("_"));
                result = firstString.compareTo(secondString);
                if (result == 0) {
                    int firstNumber = Integer.parseInt(this.getId().substring(this.getId().lastIndexOf("_") + 1));
                    int secondNumber = Integer.parseInt(o.getId().substring(o.getId().lastIndexOf("_") + 1));
                    result = firstNumber - secondNumber;
                }
            }
            if (result == 0) {
                result = this.getId().compareTo(o.getId());
            }
        }
        return result;
    }

    /**
     * Fait la simulation. La simulation en elle meme est delegue au 
     * {@link SimulatorLauncher}. Le travail restant ici est le nettoyage,
     * la gestion des erreurs ou l'iteration s'il sagit de plan d'analyse
     * dependant.
     */
    public void run() {
        try {
            SimulationControl control = item.getControl();
            String id = control.getId();
            if (control.isStopSimulationRequest()
                    || (getParentJob() != null && getParentJob().getItem()
                            .getControl().isStopSimulationRequest())) {
                log.info(_("Not start simulation %s because user ask stop",
                                id));
                return;
            }
            SimulationParameter param = item.getParameter();
            if (log.isInfoEnabled()) {
                log.info("Start simulation: " + id);
            }
            simulationService.fireStartEvent(this);

            if (!onlyCheckControl && getParentJob() == null
                    && param.getUseAnalysePlan() && !param.isIndependentPlan()) {
                // on est sur un plan d'analyse dependant, il faut generer les
                // simulation les unes apres les autres
                SimulationService.PrepareSimulationJob i = new SimulationService.PrepareSimulationJob(
                        simulationService, this);
                while (!control.isStopSimulationRequest() && i.hasNext()) {
                    log.info(_("Generate next simulation"));
                    SimulationJob subjob = i.next();
                    subjob.setLauncher(getLauncher());
                    // c'est bloquant seulement le temps du lancement
                    // (envoie sur caparmor par exemple)
                    // l'appel peut ensuite passer alors que la simulation
                    // n'a pas effectivement demarré
                    subjob.run();
                    
                    // FIXME temp fix les thread des launchers
                    // ne sont plus bloquants
                    // on bloque le thread par un sleep
                    // tant que le sous thread n'est pas fini
                    SimulationItem subItem = subjob.getItem();
                    SimulationControl subControl = subItem.getControl();
                    do {
                        Thread.sleep(2000);
                        
                        // on boucle tant que:
                        // - la simulation n'a pas été arreté
                        // - la simulation n'est pas a 0
                        // - la simulation n'est aps à la fin
                        // - la simulation n'existe pas localement
                    } while (!subControl.isStopSimulationRequest() &&
                            (subControl.getProgress() == 0 || subControl.getProgress() < subControl.getProgressMax()
                            || !SimulationStorage.exists(subControl.getId())));

                    // FIXME on fait manuellement le post simulation pour
                    // plans dépendant
                    if (!subControl.isStopSimulationRequest()) {
                        SimulationStorage simulation = subjob.getLauncher().getSimulationStorage(simulationService, subControl);
                        i.finished(subjob, simulation);
                    }
                }

            } else {
                // on est sur une simple simulation, ou le resultat d'un plan

                // Dans le cas d'un simulation simple seulement
                // on mémorise la simulation comme "lancée" et
                // devant être re monitorée au lancement d'isis
                // FIXME try to remove static call
                SimulationMonitor.getInstance().simulationStart(this);

                /* code moved to
                fr.ifremer.isisfish.simulator.launcher.InProcessSimulatorLauncher.simulate(SimulationService, SimulationControl, File)
                // set date to 0 at beginning of simulation
                control.setDate(new Date());
                control.setProgress(0);
                control.setStarted(true);

                int lastYear = param.getNumberOfYear();
                int lastDate = lastYear * Month.NUMBER_OF_MONTH;
                control.setProgressMax(lastDate);*/

                //SimulationStorage simulation = null;
                if (!onlyCheckControl) {
                    //File zip = item.getSimulationZip();

                    // get prescript content, only if UsePreScript is set to true
                    // FIXME EC-20100410 use prescriot already present in params
                    /*String simulationPrescript = null;
                    if(param.getUsePreScript()) {
                        simulationPrescript = param.getPreScript();
                        item.setSimulationPrescriptContent(simulationPrescript);
                    }*/

                    launcher.simulate(simulationService, item);
                }

                // simulation ended
                //simulation = launcher.getSimulationStorage(simulationService,
                //        control);

                /*
                 
                 Code moved to {@link fr.ifremer.isisfish.simulator.launcher.SimulationMonitor#doPostSimulationOperation(SimulationJob, SimulatorLauncher)}
                 
                try {
                    // copie les exports de simulation dans le repertoire
                    // souhaiter par l'utilisateur
                    if (param.getExportNames() != null
                            && param.getExportNames().size() > 0) {
                        File exportDir = SimulationStorage
                                .getResultExportDirectory(simulation
                                        .getDirectory());
                        // FIXME verifier que pour les plans on a bien tous les
                        // export, c-a-d que les export son fait dans un sous rep
                        // portant le nom de la simu
                        FileUtil.copyAndRenameRecursively(exportDir, new File(
                                param.getExportDirectory()), exportDir
                                .getName(), id);
                    }
                } catch (Exception eee) {
                    log.error(_("Can't export simulation %s", id), eee);
                }

                try {
                    // Si l'utilisateur souhaite seulement les exports
                    // on supprimer la/les simulations.
                    if (param.getOnlyExport()) {
                        // pour les plan dependant il faut le faire apres toutes
                        // les simulations donc pas ici
                        if (getParentJob() == null || param.getUseAnalysePlan()
                                && param.isIndependentPlan()) {
                            simulation.delete(false);
                        }
                    }
                } catch (Exception eee) {
                    log.error(_("Can't delete simulation %s ", id), eee);
                }

                for (PostAction action : postActions) {
                    try {
                        action.finished(this, simulation);
                    } catch (Exception eee) {
                        log.error(_("Can't do post action %s", action), eee);
                    }
                }*/
            }

        } catch (Throwable eee) {
            log.warn(_("Can't simulate %s", item.getControl().getId()), eee);
            if (eee instanceof RemoteException) {
                simulationService.reportError(getLauncher(), this);
            } else {
                // on ne le fait pas pour RemoteException, car la simulation
                // va etre resoumise
                for (PostAction action : postActions) {
                    action.exception(this, eee);
                }
            }
        } /*finally {
                   simulationService.fireStopEvent(this);
               } */
    }

    /**
     * Interface permettant d'implanter des actions a faire apres la simulation.
     * Ces actions ne se declenchent pas pour les job de plan d'analyse pere.
     */
    public static interface PostAction {
        
        /**
         * Appelé lorsque la simulation s'arrete normalement.
         * 
         * @param job le job qui a fait la simulation
         * @param sim la simulation qui vient d'etre faite
         */
        public void finished(SimulationJob job, SimulationStorage sim);

        /**
         * Appeler lorsque la simulation a echoué.
         * 
         * @param job le job qui a fait la simulation
         * @param eee l'exception qui a ete levee
         */
        public void exception(SimulationJob job, Throwable eee);
    }

}
