/* *##%
 * Copyright (C) 2006 - 2009
 *     Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package fr.ifremer.isisfish.simulator;

import java.io.File;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;

import fr.ifremer.isisfish.datastore.SimulationStorage;

/**
 * Keep all information on one simulation.
 * 
 * <li> Launch parameter
 * <li> Database (TopiaContext)
 * <li> SimulationControl
 * <li> Effectif by pop (N)
 * <li> Result
 * 
 * Created: 3 juil. 2006 17:05:27
 *
 * @author poussin
 * @version $Revision: 2938 $
 *
 * Last update: $Date: 2010-01-22 16:42:09 +0100 (ven., 22 janv. 2010) $
 * by : $Author: chatellier $
 */
public class SimulationContext {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static Log log = LogFactory.getLog(SimulationContext.class);

    protected Map<String, Object> values = new HashMap<String, Object>();
    protected SimulationStorage simulation = null;
    protected SimulationControl simulationControl = null;
    protected PopulationMonitor populationMonitor = null;
    protected MetierMonitor metierMonitor = null;
    protected RuleMonitor ruleMonitor = null;
    protected ResultManager resultManager = null;
    protected Set<SimulationListener> simulationListeners = new LinkedHashSet<SimulationListener>();
    protected ClassLoader classLoader = null;
    protected File scriptDirectory;

    /** TopiaContext must be used by rule action to modify data */
    protected TopiaContext db = null;

    /** TopiaContext must be used to save result */
    protected TopiaContext dbResult = null;

    /** Context value used in equation. */
    protected Map<String, Double> contextEquationValue = new HashMap<String, Double>();
    /** Operator used in equation values */
    protected Map<String, String> contextEquationOperator = new HashMap<String, String>();
    
    private static ThreadLocal<SimulationContext> simulationContext = new ThreadLocal<SimulationContext>() {
        protected synchronized SimulationContext initialValue() {
            return new SimulationContext();
        }
    };

    protected SimulationContext() {

    }

    /**
     * @return simulation context for the current simulation (current thread)
     */
    public static SimulationContext get() {
        return simulationContext.get();
    }

    /**
     * remove simulation associate with current thread
     */
    public static void remove() {
        SimulationContext current = get();
        current.values.clear();
        current.classLoader = null;
        simulationContext.remove();
    }

    /**
     * Add simulation listener, if listener is {@link SimulationResultListener}, it's 
     * automatically added as listener on {@link ResultManager}
     * @param l
     */
    public void addSimulationListener(SimulationListener l) {
        simulationListeners.add(l);
        if (l instanceof SimulationResultListener) {
            getResultManager().addSimulationResultListener(
                    (SimulationResultListener) l);
        }
    }

    public void closeDB() {
        if (db != null) {
            try {
                db.closeContext();
            } catch (TopiaException eee) {
                if (log.isDebugEnabled()) {
                    log.debug("Can't close simulation topia context", eee);
                }
            }
        }
    }

    public void closeDBResult() {
        if (dbResult != null) {
            try {
                dbResult.commitTransaction();
                dbResult.closeContext();
            } catch (TopiaException eee) {
                if (log.isDebugEnabled()) {
                    log.debug("Can't close simulation result topia context",
                            eee);
                }
            }
        }
    }

    /**
     * Remove simulation listener, if listener is {@link SimulationResultListener}, it's 
     * Automatically removed as listener on {@link ResultManager}
     * @param l
     */
    public void removeSimulationListener(SimulationListener l) {
        simulationListeners.remove(l);
        if (l instanceof SimulationResultListener) {
            getResultManager().addSimulationResultListener(
                    (SimulationResultListener) l);
        }
    }

    public void fireBeforeSimulation() {
        for (SimulationListener l : simulationListeners) {
            l.beforeSimulation(this);
        }
    }

    public void fireAfterSimulation() {
        for (SimulationListener l : simulationListeners) {
            l.afterSimulation(this);
        }
    }

    /**
     * @return Returns the classLoader.
     */
    public ClassLoader getClassLoader() {
        return this.classLoader;
    }

    /**
     * @param classLoader The classLoader to set.
     */
    public void setClassLoader(ClassLoader classLoader) {
        this.classLoader = classLoader;
    }

    /**
     * Permet de recuperer une valeur prealablement stocker avec un setValue
     * @param name le nom de la valeur souhaitée
     * @return la valeur ou null si aucune valeur ne porte se nom
     */
    public Object getValue(String name) {
        Object result = values.get(name);
        return result;
    }

    /**
     * Permet de stocker une valeur en fonction d'une cle, cela peut-etre util
     * pour partager des informations entre Rule ou d'autre objet.
     * @param name le nom de la valeur
     * @param value la valeur
     */
    public void setValue(String name, Object value) {
        values.put(name, value);
    }

    /**
     * @return Returns the topiaContext.
     */
    public SimulationStorage getSimulationStorage() {
        return this.simulation;
    }

    /**
     * @param simulation The simulation storage to set.
     */
    public void setSimulationStorage(SimulationStorage simulation) {
        this.simulation = simulation;
    }

    /**
     * @return Returns the simulationControl.
     */
    public SimulationControl getSimulationControl() {
        return this.simulationControl;
    }

    /**
     * @param simulationControl The simulationControl to set.
     */
    public void setSimulationControl(SimulationControl simulationControl) {
        this.simulationControl = simulationControl;
    }

    /**
     * @return Returns the populationMonitor.
     */
    public PopulationMonitor getPopulationMonitor() {
        if (this.populationMonitor == null) {
            this.populationMonitor = new PopulationMonitor();
        }
        return this.populationMonitor;
    }

    /**
     * @return Returns the metierMonitor.
     */
    public MetierMonitor getMetierMonitor() {
        if (this.metierMonitor == null) {
            this.metierMonitor = new MetierMonitor();
        }
        return this.metierMonitor;
    }

    /**
     * @return Returns the ruleMonitor.
     */
    public RuleMonitor getRuleMonitor() {
        if (this.ruleMonitor == null) {
            this.ruleMonitor = new RuleMonitor();
        }
        return this.ruleMonitor;
    }

    /**
     * @return Returns the resultManager.
     */
    public ResultManager getResultManager() {
        if (this.resultManager == null) {
            this.resultManager = new ResultManager(this);
        }
        return this.resultManager;
    }

    /**
     * This method return TopiaContext that Rule action must used to modify
     * data. This TopiaContext is rollbacked after each step.
     * @return TopiaContext that Rule action must used
     * @throws TopiaException 
     */
    public TopiaContext getDB() throws TopiaException {
        if (db == null && getSimulationStorage() != null) {
            db = getSimulationStorage().getMemStorage().beginTransaction();
        }
        return db;
    }

    /**
     * this method return TopiaContext that must be used to save result
     * @return Returns the dbResult.
     * @throws TopiaException 
     */
    public TopiaContext getDbResult() throws TopiaException {
        if (dbResult == null && getSimulationStorage() != null) {
            dbResult = getSimulationStorage().getStorage().beginTransaction();
        }
        return this.dbResult;
    }

    /**
     * @return Returns the scriptDirectory.
     */
    public File getScriptDirectory() {
        return this.scriptDirectory;
    }

    /**
     * Script directory to use.
     * 
     * Used to chnage isis-database-3 directory in running simulation context.
     * 
     * @param scriptDirectory
     */
    public void setScriptDirectory(File scriptDirectory) {
        this.scriptDirectory = scriptDirectory;
    }

    /**
     * @param message message
     */
    public void message(String message) {
        log.info(message);
        if (getSimulationControl() != null) {
            getSimulationControl().setText(message);
        }
    }

    /**
     * Add value/key pair into context.
     * 
     * @param key key
     * @param value value
     * @param operator operator
     */
    public void setValue(String key, Double value, String operator) {
        contextEquationValue.put(key, value);
        contextEquationOperator.put(key, operator);
    }
    
    /**
     * Return value from context.
     * 
     * @param key key
     * @param defaultValue default value or value to compute
     * @return computed value or <tt>defaultValue</tt> if not found
     */
    public double getValueAndCompute(String key, double defaultValue) {
        
        double result = defaultValue;

        if (contextEquationValue.containsKey(key) && contextEquationOperator.containsKey(key)) {

            Double value = contextEquationValue.get(key);
            String operator = contextEquationOperator.get(key);
            
            if (log.isTraceEnabled()) {
                log.trace("Found key '" + key + "' current value = " + value);
            }
            
            if ("+".equals(operator)) {
                result = value + defaultValue;
            }
            else if ("-".equals(operator)) {
                result = value - defaultValue;
            }
            else if ("*".equals(operator)) {
                result = value * defaultValue;
            }
            else if ("/".equals(operator)) {
                result = value / defaultValue;
            }
            else {
                throw new IllegalArgumentException("Unknown operator '" + operator + "'");
            }
        }
        else {
            if (log.isTraceEnabled()) {
                log.trace("No key defined for key '" + key + "' in context");
            }
        }
        
        return result;
    }
}
