/* *##%
 * Copyright (C) 2007
 *     Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

/* *
 * ResultManager.java
 *
 * Created: 13 nov. 07 11:54:47
 *
 * @author poussin
 * @version $Revision: 2658 $
 *
 * Last update: $Date: 2009-10-13 17:00:50 +0200 (mar., 13 oct. 2009) $
 * by : $Author: chatellier $
 */

package fr.ifremer.isisfish.simulator;

import static org.nuiton.i18n.I18n._;

import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.math.matrix.MatrixND;

import fr.ifremer.isisfish.IsisFishException;
import fr.ifremer.isisfish.datastore.ExportStorage;
import fr.ifremer.isisfish.entities.Population;
import fr.ifremer.isisfish.export.Export;
import fr.ifremer.isisfish.rule.Rule;
import fr.ifremer.isisfish.types.Date;


/**
 * Used to store and retrieve result during simulation. This class manage
 * result listener
 * 
 * @author poussin
 */
public class ResultManager {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(ResultManager.class);

    protected SimulationContext context = null;
    protected Set<SimulationResultListener> listeners = new LinkedHashSet<SimulationResultListener>();
    protected SimulationResultGetter getter = null;
    
    /** result enabled */
    transient protected Set<String> enabledResult = null;

    /**
     * 
     */
    public ResultManager(SimulationContext context) {
        this.context = context;
    }
    
    /**
     * Don't use this method to add listener, use
     * {@link SimulationContext#addSimulationListener(SimulationListener)}
     * @param l
     */
    protected void addSimulationResultListener(SimulationResultListener l) {
        listeners.add(l);
        if (l instanceof SimulationResultGetter) {
            getter = (SimulationResultGetter) l;
        }
    }
    
    /**
     * Don't use this method to remove listener, use
     * {@link SimulationContext#removeSimulationListener(SimulationListener)}
     * @param l
     */
    protected void removeSimulationResultListener(SimulationResultListener l) {
        listeners.remove(l);
        if (l == getter) {
            getter = null;
            // search new getter
            for (SimulationResultListener i : listeners) {
                if (i instanceof SimulationResultGetter) {
                    getter = (SimulationResultGetter) i;
                    break;
                }
            }
        }
    }
    
    /**
     * Permet de savoir si lorsque l'on ajoutera ce resultat, il sera
     * sauvé ou non.
     */
     public boolean isEnabled(String name){
         name = name.trim();
         if (enabledResult == null) {
             SimulationParameter param = context.getSimulationStorage().getParameter();
             enabledResult = new HashSet<String>();
             
             Collection<String> resultEnabled = param.getResultEnabled();
             enabledResult.addAll(resultEnabled);
             List<String> exportNames = param.getExportNames();
             if (exportNames != null) {
                 for (String exportName : exportNames) {
                     ExportStorage storage = ExportStorage.getExport(exportName);
                     try {
                         Export export = storage.getNewExportInstance();
                         for (String resultName : export.getNecessaryResult()) {
                             enabledResult.add(resultName);
                         }
                     } catch (IsisFishException eee) {
                         if (log.isWarnEnabled()) {
                             log.warn(_("Can't instanciate export %1$s", exportName), eee);
                         }
                     }
                 }
             }
             List<Rule> rules = param.getRules();
             if (rules != null) {
                 for (Rule rule : rules) {
                     for (String resultName : rule.getNecessaryResult()) {
                         enabledResult.add(resultName);
                     }
                 }
             }
             List<AnalysePlan> plans = param.getAnalysePlans();
             if (plans != null) {
                 for (AnalysePlan plan : plans) {
                     for (String resultName : plan.getNecessaryResult()) {
                         enabledResult.add(resultName);
                     }
                 }
             }
             log.info("Enabled result: " + enabledResult);
         }
         boolean result = enabledResult.contains(name);
         return result;
     }
     public void addActiveRule(Date date, Rule rule) throws IsisFishException {
         log.warn("addActiveRule not implemented");
         // FIXME: addActiveRule not implemented
     }
     
     public void addResult(Date date, MatrixND mat) throws IsisFishException{
         addResult(false, date, mat.getName(), mat);
     }

     public void addResult(Date date, Population pop, MatrixND mat) throws IsisFishException{
         addResult(false, date, mat.getName(), pop, mat);
     }

     public void addResult(boolean force, Date date, MatrixND mat) throws IsisFishException{
         addResult(force, date, mat.getName(), mat);
     }

     public void addResult(boolean force, Date date, Population pop, MatrixND mat) throws IsisFishException{
         addResult(force, date, mat.getName(), pop, mat);
     }

     public void addResult(Date date, String name, Population pop, MatrixND mat) throws IsisFishException{
         addResult(false, date, name, pop, mat);
     }

     public void addResult(Date date, String name, MatrixND mat) throws IsisFishException{
         addResult(false, date, name, mat);
     }
     
     public void addResult(boolean force, Date date, String name, Population pop, MatrixND mat) throws IsisFishException{
         // don't call generic addResult, because pop name is added
         if (force || isEnabled(name)) {
             for (SimulationResultListener l : listeners) {
                 l.addResult(context, date, name + " " + pop, mat);
             }
         }
     }

     public void addResult(boolean force, Date date, String name, MatrixND mat) throws IsisFishException{
         if (force || isEnabled(name)) {
             for (SimulationResultListener l : listeners) {
                 l.addResult(context, date, name, mat);
             }
         }
     }

     /**
      * Retourne la matrice stocke pour un pas de temps
      * @param date le pas de temps que l'on souhaite
      * @param pop la population pour lequelle on souhaite le resultat
      * @param name le nom des resultats dont on veut la matrice
      * @return La matrice demandée ou null si aucune matrice ne correspond a
      * la demande.
      */
      public MatrixND getMatrix(Date date, Population pop, String name){
          name += " " + pop;
          return getMatrix(date, name);
      }

      public MatrixND getMatrix(Date date, String name){
          MatrixND result = null;
         if (getter != null) {
             result = getter.getMatrix(context, date, name);
          }
         return result;
      }
      
      /**
       * Retourne une matrice contenant tous les pas de temps.
       * @param pop la population pour lequel on souhaite la matrice
       * @param name le nom des resultats dont on veut une matrice globale.
       */
      public MatrixND getMatrix(Population pop, String name){
          name += " " + pop;
          return getMatrix(name);
      }

      /**
       * Retourne une matrice contenant tous les pas de temps.
       * @param name le nom des resultats dont on veut une matrice globale.
       */
      public MatrixND getMatrix(String name){
          MatrixND result = null;
          if (getter != null) {
              result = getter.getMatrix(context, name);
          }
          return result;
      }
   
}


