/* *##%
 * Copyright (C) 2006 - 2009
 *     Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package fr.ifremer.isisfish.rule;

import fr.ifremer.isisfish.entities.Metier;
import fr.ifremer.isisfish.simulator.SimulationContext;
import fr.ifremer.isisfish.types.Date;

/**
 * Interface d'une regle. setValue et getValue permet de stocker des
 * informations liées a l'instance de la règle.
 * 
 * <p> Lorsqu'on implante une rule, les paramètres que l'on souhaite qu'elle 
 * ait, doivent etre des attributs public commencant par param.
 * 
 * <p> Les types acceptable pour un parametre sont:
 * <li> un nombre (int, long, float, double, ...)
 * <li> un boolean
 * <li> une chaine String
 * <li> un objet Date ({@link fr.ifremer.isisfish.types.Date})
 * <li> un objet Month ({@link fr.ifremer.isisfish.types.Month})
 * <li> Le type d'une entité (@see fr.ifremer.isisfish.entities)
 * 
 * <p> Pour supporter d'autre type il faut créer des converters commons-beanutils
 * //{@link fr.ifremer.isisfish.simulator.SimulationParameter#ruleParametertoProperties(int, Rule)}
 * 
 * Created: 12 janv. 2006 17:02:43
 *
 * @author poussin
 * @version $Revision: 2940 $
 *
 * Last update: $Date: 2010-01-22 16:46:41 +0100 (ven., 22 janv. 2010) $
 * by : $Author: chatellier $
 */
public interface Rule {

    /**
     * Return necessary results names for rule execution.
     * 
     * @return the necessaryResult name
     */
    public String[] getNecessaryResult();

    /**
     * Permet d'afficher a l'utilisateur une aide sur la regle.
     * 
     * @return L'aide ou la description de la regle
     * @throws Exception
     */
    public String getDescription() throws Exception;

    /**
     * Appelé au démarrage de la simulation, cette méthode permet d'initialiser
     * la règle.
     * 
     * @param context La simulation pour lequel on utilise cette regle
     * @throws Exception
     */
    public void init(SimulationContext context) throws Exception;

    /**
     * La condition qui doit etre vrai pour faire les actions.
     * 
     * Si la confition return vrai alors les methodes
     * {@link #preAction(SimulationContext, Date, Metier)} et
     * {@link #postAction(SimulationContext, Date, Metier)} seront appelée.
     * 
     * @param context La simulation pour lequel on utilise cette regle
     * @param date la date courante
     * @param metier le metier
     * @return vrai si on souhaite que les actions soit faites
     * @throws Exception s'il y a une erreur, met fin a la simulation
     */
    public boolean condition(SimulationContext context, Date date, Metier metier)
            throws Exception;

    /**
     * Si la condition est vrai alors cette action est executée avant le pas
     * de temps de la simulation.
     * 
     * @param context La simulation pour lequel on utilise cette regle
     * @param date la date courant
     * @param metier le metier
     * @throws Exception s'il y a une erreur, met fin a la simulation
     */
    public void preAction(SimulationContext context, Date date, Metier metier)
            throws Exception;

    /**
     * Si la condition est vrai alors cette action est executée apres le pas
     * de temps de la simulation.
     * 
     * @param context La simulation pour lequel on utilise cette regle
     * @param date la date courante
     * @param metier le metier
     * @throws Exception s'il y a une erreur, met fin a la simulation
     */
    public void postAction(SimulationContext context, Date date, Metier metier)
            throws Exception;

    /**
     * Permet de recuperer une valeur prealablement stockée avec un
     * {@link #setValue(String, Object)}.
     * 
     * @param name le nom de la valeur souhaitée
     * @return la valeur ou null si aucune valeur ne porte se nom
     */
    public Object getValue(String name);

    /**
     * Permet de stocker une valeur en fonction d'une cle.
     * 
     * @param name le nom de la valeur
     * @param value la valeur
     */
    public void setValue(String name, Object value);
}
