/*
* ##% Copyright (C) 2002, 2003, 2004, 2005, 2006, 2007 Code Lutin,
* Benjamin Poussin, Tony Chemit
*
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
* ##% */
package fr.ifremer.isisfish.logging.io;

import fr.ifremer.isisfish.logging.LogLevel;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

/**
 * This class offers some OffsetReader and LineReader implementations.
 *
 * @author chemit
 */

public class LineReaderUtil {

    /**
     * a LineReader to keep only lines that match a regex pattern.
     *
     * @author chemit
     */
    public static class PatternLineReader extends LineReader {

        protected Pattern pattern;

        public PatternLineReader(File file, OffsetReader offsetReader, String level, int flags) {
            super(file, offsetReader);
            pattern = Pattern.compile(level, flags);
        }

        public PatternLineReader(LineReader parent, OffsetReader offsetReader, String level, int flags) {
            super(parent, offsetReader);
            pattern = Pattern.compile(level, flags);
        }

        @Override
        public boolean match(String line) {
            return pattern.matcher(line).matches();
        }

        @Override
        public String toString() {
            return "(pattern:" + pattern + ')' + super.toString();
        }
    }

    /**
     * a LineReader to keep only lines from a certain {@link LogLevel}.
     *
     * @author chemit
     */
    public static class LevelLineReader extends LineReader {

        protected String level;

        public LevelLineReader(File file, OffsetReader offsetReader, LogLevel level) {
            super(file, offsetReader);
            this.level = level.toString();
        }

        public LevelLineReader(LineReader parent, OffsetReader offsetReader, LogLevel level) {
            super(parent, offsetReader);
            this.level = level.toString();
        }

        @Override
        public boolean match(String line) {
            return line.startsWith(level);
        }

        @Override
        public String toString() {
            return "(level:" + level + ')' + super.toString();
        }
    }

    /**
     * a LineReader to keep only lines that match one of the given {@link LogLevel}.
     *
     * @author chemit
     */
    public static class LevelsLineReader extends LineReader {

        protected List<String> levels;

        public LevelsLineReader(File file, OffsetReader offsetReader, LogLevel... level) {
            super(file, offsetReader);
            levels = new ArrayList<String>();
            for (LogLevel logLevel : level) {
                levels.add(logLevel.toString());
            }
        }

        public LevelsLineReader(LineReader parent, OffsetReader offsetReader, LogLevel... levels) {
            super(parent, offsetReader);
            this.levels = new ArrayList<String>();
            for (LogLevel level : levels) {
                if (!this.levels.contains(level.toString())) {
                    this.levels.add(level.toString());
                }
            }
        }

        @Override
        public boolean match(String line) {
            if (levels.isEmpty()) {
                return true;
            }
            for (String s : levels) {
                if (line.startsWith(s)) {
                    return true;
                }
            }
            return false;
        }
        @Override
        public String toString() {
            return "(levels:" + levels + ')' + super.toString();
        }
    }


}
