/*
 * ##% Copyright (C) 2002 - 2009
 *     Ifremer, Code Lutin, Benjamin Poussin, Tony Chemit
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * ##% */
package fr.ifremer.isisfish.logging.console;

import fr.ifremer.isisfish.logging.LogLevel;
import fr.ifremer.isisfish.logging.LogLevelUtil;
import static fr.ifremer.isisfish.logging.console.LogConsole.*;
import fr.ifremer.isisfish.logging.io.FileOffsetReader;
import fr.ifremer.isisfish.logging.io.LineReader;
import fr.ifremer.isisfish.logging.io.LineReaderUtil;
import fr.ifremer.isisfish.logging.io.LineReaderUtil.LevelsLineReader;
import fr.ifremer.isisfish.logging.io.LineReaderUtil.PatternLineReader;
import fr.ifremer.isisfish.logging.io.MemoryOffsetReader;
import org.apache.commons.logging.Log;
import static org.apache.commons.logging.LogFactory.getLog;
import static org.nuiton.i18n.I18n._;
import org.nuiton.widget.StatusBar;

import javax.swing.*;
import java.awt.event.AdjustmentEvent;
import java.awt.event.AdjustmentListener;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * This class is responsible of provinding LogRecord, it deals with LogReecord
 * stored in a file with a serializedForm for performance issue.
 * <p/>
 * <b>Use one LogConsoleHandler for one log file</b>
 *
 * @author chemit
 */
public class LogConsoleHandler implements PropertyChangeListener, MouseWheelListener, AdjustmentListener {

    static private final Log log = getLog(LogConsoleHandler.class);

    /** the current line reader */
    protected LineReader reader;

    /** list of LevelsReader */
    protected List<LevelsLineReader> levelsReaders;

    /** ui model to be used in dialog */
    protected LogConsoleModel model;

    /** the directory where store readers */
    protected File readerDirectory;

    /** a flag to block scroll bar adjustement events */
    protected boolean dontAdjust = true;

    /** the console status bar where to notify user events */
    protected StatusBar statusBar;

    /** the dialog to send mail */
    protected LogMail logMail;

    protected PropertyChangeSupport propertyListeners = new PropertyChangeSupport(this);

    public LogConsoleHandler(LogConsoleModel model) throws IOException {

        this.model = model;

        this.levelsReaders = new ArrayList<LevelsLineReader>();

        // create the reader directory
        File logFile = model.getLogFile();        
        File tmpDir = new File(System.getProperty("java.io.tmpdir"),"isis-log-cache");
        if (!tmpDir.exists()) {
            tmpDir.mkdirs();
        }
        this.readerDirectory = new File(tmpDir,logFile.getParentFile().getName()+ logFile.getName() + "_offsets");
        this.readerDirectory.deleteOnExit();        
        log.info(this);
    }

    /**
     * start the handler (will start all internal services required)
     *
     * @throws IOException if any
     */
    public void start() throws IOException {

        long t0 = System.currentTimeMillis();

        if (!readerDirectory.exists()) {
            readerDirectory.mkdir();
        }

        // the first reader is a level reader (with all levels)
        LineReader reader = getLevelReader(0);

        // open it
        openReader(reader);

        log.info("with reader " + this.reader + " in " + (System.currentTimeMillis() - t0));
    }

    /**
     * stop the handler and all internal services
     *
     * @throws Exception if any problem while closing operation
     */
    public void close() throws Exception {

        for (LevelsLineReader levelsReader : levelsReaders) {
            try {
                levelsReader.close();
            } catch (IOException e) {
                log.warn(_("could not close reader %1$s", levelsReader));
            }
        }

        if (reader != null && reader.isOpen()) {
            try {
                reader.close();
            } catch (IOException e) {
                log.warn(_("could not close reader %1$s", reader));
            }
        }        
    }

    @Override
    protected void finalize() throws Throwable {
        super.finalize();
        close();
    }

    /**
     * read the required frame from current reader
     *
     * @param offset the offset to use
     * @throws IOException if any problem while reading
     */
    public void read(long offset) throws IOException {
        if (offset < 0) {
            offset = 0;
        }
        //log.debug(" offset:" + offset + " width:" + model.nbLinesInEditor);

        String[] result;

        // obtain lines from reader
        result = reader.readLine(offset, model.nbLinesInEditor);
        // mark the new first position
        model.setFirstLinePosition(offset);

        // add lines in model
        model.allItems.clear();
        model.allItems.addAll(Arrays.asList(result));

        // notify ui that model changed
        model.fireStateChanged();
    }


    public void mouseWheelMoved(MouseWheelEvent e) {
        int unitsToScroll = e.getUnitsToScroll();
        long newOffset;

        newOffset = model.getFirstLinePosition() + unitsToScroll;

        if (unitsToScroll > 0) {
            if (model.isEOF()) {
                // nothing to do we are already at the tail of the stream
                return;
            }
        } else {
            if (model.isBOF()) {
                // nothing to do we are already at the head of the stream
                return;
            }
        }

        try {
            dontAdjust = true;
            read(newOffset);
        } catch (IOException e1) {
            log.warn(_("could not read at offset %1$s for reason %2$s", newOffset, e1.getMessage()));
        }
    }

    public void adjustmentValueChanged(AdjustmentEvent e) {
        if (!e.getValueIsAdjusting() && !dontAdjust) {
            try {
                //TODO Fix bug :sometimes when going at tail, it goes head
                //TODO (the offset must not be good...)
                //log.info("value:"+e.getValue());
                dontAdjust = true;
                read(e.getValue());
            } catch (IOException e1) {
                log.warn(_("could not read at offset %1$s for reason %2$s", e.getValue(), e1.getMessage()));
            }

        }
        dontAdjust = false;
    }

    public void propertyChange(PropertyChangeEvent evt) {
        String properyName = evt.getPropertyName();

        if (properyName.equals(LogConsole.DISPOSE_CHANGED_PROPERTY)) {
            // ui required a dispose
            try {
                close();
            } catch (Exception e) {
                log.warn(_("isisfish.error.log.console.dispose" ,this,e.getMessage()));
            }
            return;
        }

        //log.info(properyName + " : " + evt.getNewValue());

        if (properyName.equals(RESET_CHANGED_PROPERTY)) {
            // reset filter model
            model.setLevels(0);
            model.setSearchText("");
            changeFilter();

            return;
        }
        if (properyName.equals(EDITOR_SIZE_CHANGED_PROPERTY)) {
            JEditorPane comp = (JEditorPane) evt.getNewValue();

            // compute new model#nbLinesInEditor value
            int height = comp.getHeight();
            //TODO Find exactly the number of lines...
            int nbLines = (height / 14) - 2;
            int oldValue = model.getNbLinesInEditor();
            if (oldValue != nbLines) {
                log.info("change newLinesInEditor : " + oldValue + " --> " + nbLines);
                model.setNbLinesInEditor(nbLines);
                if (!model.allItems.isEmpty()) {
                    if (model.allItems.size() > nbLines) {
                        // remove on tail
                        while (model.allItems.size() > nbLines) {
                            model.allItems.remove(model.allItems.size() - 1);
                        }
                    } else {
                        String newLine;
                        long lastPos = model.getLastLinePosition() + 1;
                        while (model.allItems.size() < nbLines) {
                            // add to tail
                            try {
                                newLine = reader.readLine(lastPos++);
                                if (newLine == null) {
                                    break;
                                }
                                model.allItems.add(newLine);
                            } catch (IOException e) {
                                if (log.isErrorEnabled()) {
                                    log.error("Can't read log file", e);
                                }
                            }
                        }
                    }
                }
                model.fireStateChanged();
            }
            return;
        }

        if (properyName.equals(TEXT_CHANGED_PROPERTY)) {
            String newValue = (String) evt.getNewValue();
            model.setSearchText(newValue);
            changeFilter();
            return;
        }

        LogLevel level = LogLevel.valueOf(properyName);

        if (level != null) {
            Boolean newValue = (Boolean) evt.getNewValue();
            int oldLevels = model.getLevels();
            int newLevels;
            if (newValue) {
                newLevels = LogLevelUtil.addToSet(level.ordinal(), oldLevels);
            } else {
                newLevels = LogLevelUtil.removeFromSet(level.ordinal(), oldLevels);
            }
            model.setLevels(newLevels);

            changeFilter();
        }
    }

    protected void changeFilter() {
        String searchText = model.getSearchText();
        int levels = model.getLevels();

        log.info("levels:" + model.levels + ", searchText:" + searchText);
        long t0;
        t0 = System.currentTimeMillis();
        LineReader parent;
        // we have a levels filter
        try {
            parent = getLevelReader(levels);
        } catch (IOException e) {
            log.warn("could not get level reader [" + levels + "] for reason " + e.getMessage());
            return;
        }

        LineReader reader = parent;
        if (!searchText.trim().equals("")) {
            // this is a pattern filter, based on parent reader
            reader = getPatternReader(searchText, parent);
        }

        if (this.reader instanceof PatternLineReader) {
            // we close the pattern reader for the moment
            // but we could keep it in memory ?
            try {
                this.reader.close();
                this.reader = null;
            } catch (IOException e) {
                log.warn("could not close pattern reader [" + this.reader + "] for reason " + e.getMessage());
            }
        }

        try {
            // open it
            openReader(reader);
            // obtain the head of the stream
            read(0);
            dontAdjust = true;
            model.fireStateChanged();
            getStatusBar().setStatus(_("filter loaded in %1$s ms : found %2$s lines.", (System.currentTimeMillis() - t0), reader.getNbLines()));
        } catch (IOException e) {
            log.warn("could not open reader [" + this.reader + "] for reason " + e.getMessage());
        }
    }

    protected LineReader getPatternReader(String searchText, LineReader parent) {
        LineReader reader;
        reader = new PatternLineReader(parent, new MemoryOffsetReader(5000), searchText, 0);
        reader.setId(parent.getId() + ":" + searchText);
        log.info(reader);
        return reader;
    }

    //TODO Improve algorithm : always try to find the closest reader to use
    protected LineReader getLevelReader(int levels) throws IOException {
        String levelsStr = levels + "";
        LevelsLineReader reader = null;
        for (LineReaderUtil.LevelsLineReader levelsReader : levelsReaders) {
            if (levelsReader.getId().equals(levelsStr)) {
                reader = levelsReader;
            }
        }
        if (reader == null) {
            File offsetFile = new File(readerDirectory, "offsets_" + levels);
            FileOffsetReader offsetReader = new FileOffsetReader(offsetFile);
            if (levels == 0) {
                // this is the all levels reader to create, depends on no others
                reader = new LevelsLineReader(model.getLogFile(), offsetReader);
            } else {
                // other level readers depens on all levels reader
                LogLevel[] logLevels = LogLevelUtil.getLogLevels(levels);
                // create the levels reader
                reader = new LevelsLineReader(getLevelReader(0), offsetReader, logLevels);
            }
            reader.setId(levelsStr);
            levelsReaders.add(reader);
        }
        if (!reader.isOpen()) {
            // open reader
            reader.open();
        }
        log.info(reader);
        return reader;
    }

    protected void openReader(LineReader reader) throws IOException {
        this.reader = reader;
        if (!this.reader.isOpen()) {
            // open reader
            this.reader.open();
        }
        // save the number of lines of reader in model
        model.nbLines = this.reader.getNbLines();
    }

    public void addPropertyChangeListener(PropertyChangeListener listener) {
        propertyListeners.addPropertyChangeListener(listener);
    }

    public void addPropertyChangeListener(String propertyName, PropertyChangeListener listener) {
        propertyListeners.addPropertyChangeListener(propertyName, listener);
    }

    public void removePropertyChangeListener(PropertyChangeListener listener) {
        propertyListeners.removePropertyChangeListener(listener);
    }

    public void removePropertyChangeListener(String propertyName, PropertyChangeListener listener) {
        propertyListeners.removePropertyChangeListener(propertyName, listener);
    }

    public StatusBar getStatusBar() {
        return statusBar;
    }

    public void setStatusBar(StatusBar statusBar) {
        this.statusBar = statusBar;
    }

    public void openLogMail() {
        if (logMail==null) {
            logMail = new LogMail(
                    statusBar,                    
                    model.getFrom(),
                    model.getLogFile(),
                    model.getLogFile().getParentFile(),
                    model.getSmtpServer());
        }
        logMail.setVisible(true);
    }
}