/*
 * ##% Copyright (C) 2002 - 2009
 *     Ifremer, Code Lutin, Benjamin Poussin, Tony Chemit
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * ##% */
package fr.ifremer.isisfish.logging;

import fr.ifremer.isisfish.IsisConfig;
import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.IsisFishException;
import fr.ifremer.isisfish.datastore.SimulationStorage;
import fr.ifremer.isisfish.logging.console.LogConsole;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.log4j.Appender;
import org.apache.log4j.FileAppender;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.log4j.PatternLayout;
import static org.nuiton.i18n.I18n._;

import java.io.File;
import java.io.IOException;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;

/**
 * FIXME poussin, a priori cette classe ne supporte pas plusieurs simulation
 * en meme temps, ce qui est le cas si on a plusieurs processeurs de dispo :(
 * 
 * 
 * Usefull class for dealing with hot configuration of log4J. this is temporary
 * we must find a way to abstract this layer.
 * <p/>
 * the class offers three public static methods :
 * <p/>
 * {@link #addSimulationAppender(String, String, String, String, String, String)}
 * to add a logger for a given simulation in a given thread, with simulLogLevel,
 * scriptLogLvel and libLogLevel given.
 * <p/>
 * {@link #removeAppender(String, String)}  to remove a appender of a simulation
 * <p/>
 * {@link #showSimulationLogConsole(String)} to display the log console of
 * a simulation, given his name.
 *
 * @author chemit
 */
public class SimulationLoggerUtil {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(SimulationLoggerUtil.class);

    static private Map<String, Level> simulLevelKeeper = null;
    static private Map<String, Level> scriptLevelKeeper = null;
    static private Map<String, Level> libLevelKeeper = null;

    /** pattern to use for simulation appender */
    public static final String LOG_PATTERN = "%p|%d{ABSOLUTE}|%F|%L|%M|%m%n";

    /** les catégories rattachées au logger de simulation. */
    private static final String[] simulLoggerToChange = {
            "fr.ifremer.isisfish",
    };

    /** les catégories rattachées au logger de script. */
    private static final String[] scriptLoggerToChange = {
            "exports",
            "formules",
            "rules",
            "scripts",
            "simulators",
            "sensitivityexports"
    };

    /** les catégories rattachées au logger de librairies. */
    private static final String[] libLoggerToChange = {
            "org.nuiton"
    };

    /**
     * Add a simple {@link FileAppender} for a given simulation id.
     *
     * @param filename       the filename where appender store logs
     * @param appenderId     the id of appender to add
     * @param simulLogLevel  simulator logger level
     * @param libLogLevel    libraries logger level
     * @param scriptLogLevel scripts logger level
     * @param threadName     the thread to exclude (if null don't use it)
     * @throws IsisFishException if we could not create appender
     */
    public static void addSimulationAppender(String filename,
                                             String appenderId,
                                             String simulLogLevel,
                                             String scriptLogLevel,
                                             String libLogLevel,
                                             String threadName)
            throws IsisFishException {
        PatternLayout layout = new PatternLayout();
        layout.setConversionPattern(LOG_PATTERN);

        ThreadFilter filter = new ThreadFilter(threadName);

        simulLevelKeeper = prepareLogger(simulLogLevel, simulLoggerToChange);
        scriptLevelKeeper = prepareLogger(scriptLogLevel, scriptLoggerToChange);
        libLevelKeeper = prepareLogger(libLogLevel, libLoggerToChange);

        //TODO See if we use should use the buffered version : define buffer size

        try {
            FileAppender appender = new FileAppender(layout, filename, false);
            appender.setName(appenderId);

            // thread filter come first
            appender.addFilter(filter);

            // add the appender to the root appender
            Logger.getRootLogger().addAppender(appender);

            if (log.isDebugEnabled()) {
                log.debug(_("isisfish.log.addAppender", appenderId));
            }
        } catch (IOException e) {
            log.error(_("isisfish.error.log.createAppender", appenderId, e.getMessage()));
            throw new IsisFishException(e);
        }

    }

    /**
     * Remove the appender used for simulation, add restore level to logger.
     *
     * @param appenderId the appender id to remove
     */
    public static void removeSimulationAppender(String appenderId) {
        if (log.isDebugEnabled()) {
            log.debug(_("isisfish.log.removeAppender", appenderId));
        }
        removeAppender(null, appenderId);
        // push back to original levels
        retablishLogger(simulLevelKeeper);
        simulLevelKeeper = null;
        retablishLogger(scriptLevelKeeper);
        scriptLevelKeeper = null;
        retablishLogger(libLevelKeeper);
        libLevelKeeper = null;
    }

    /**
     * Open a new log console for the given simulation
     *
     * @param simulationName name of the simulation to use
     * @throws Exception todo
     */
    public static void showSimulationLogConsole(String simulationName) throws Exception {
        if (simulationName != null) {

            try {
                SimulationStorage storage;

                storage = SimulationStorage.getSimulation(simulationName);

                File logFile = new File(storage.getSimulationLogFile());

                String smtpServer = IsisFish.config.getSmtpServer();
                String defaultFrom = IsisFish.config.getUserMail();

                String defaultTo = IsisConfig.REPORT_EMAIL;
                String title = _("isisfish.simulation.log.console.title", simulationName);

                LogConsole.newConsole(logFile, smtpServer, defaultFrom, defaultTo, title);

                log.info(_("isisfish.simulation.log.showConsole", simulationName));

            } catch (Exception eee) {
                log.warn(_("isisfish.error.simulation.log.openAppender", simulationName, eee.getMessage()), eee);
                throw eee;
            }
        }
    }

    /**
     * Find categories instanciated keep their level and swap to new
     * required level.
     * <p/>
     * Return the dico produced to be resotre later {@link #retablishLogger(java.util.Map)}
     *
     * @param logLevel           the required level
     * @param categoriesToChange list of categories
     * @return the dico of matching categories with their orginal level
     */
    static Map<String, Level> prepareLogger(String logLevel, String[] categoriesToChange) {
        Map<String, Level> result = new HashMap<String, Level>();
        Enumeration enumeration;

        Level level = Level.toLevel(logLevel);

        enumeration = Logger.getRootLogger().getLoggerRepository().getCurrentLoggers();
        while (enumeration.hasMoreElements()) {
            Object o = enumeration.nextElement();
            if (o instanceof Logger) {
                Logger logger = (Logger) o;
                String loggerName = logger.getName();
                for (String category : categoriesToChange) {
                    if (loggerName.startsWith(category)) {
                        // we found a logger to keep at his level
                        result.put(logger.getName(), logger.getLevel());
                        if (log.isDebugEnabled()) {
                            log.debug(_("isisfish.log.swapLogLevel", loggerName, logger.getLevel(), logLevel));
                        }
                        // change to new level
                        logger.setLevel(level);
                        break;
                    }
                }
            }
        }
        return result;
    }

    /**
     * Restore for the given categories, the associated level store in dico.
     *
     * @param levelKeeper the dico of categories to swap back to original levels
     */
    static void retablishLogger(Map<String, Level> levelKeeper) {
        Logger rootLogger = Logger.getRootLogger();
        for (Map.Entry<String, Level> entry : levelKeeper.entrySet()) {
            Logger logger = rootLogger.getLoggerRepository().getLogger(entry.getKey());
            if (logger != null) {
                Level oldLevel = entry.getValue();
                if (log.isDebugEnabled()) {
                    log.debug(_("isisfish.log.restoreLogLevel", logger.getName(), logger.getLevel(), oldLevel));
                }
                logger.setLevel(oldLevel);
            }
        }
        levelKeeper.clear();
    }

    /**
     * Remove a Log4J appender given his name for a given category.
     * <p/>
     * It category is null, we use the rootLogger.
     *
     * @param category category of appender to remove, it null use rootLogger
     * @param name     name of appender to remove
     */
    static void removeAppender(String category, String name) {
        // get logger for the category
        Logger logger = category == null ? Logger.getRootLogger() : Logger.getLogger(category);
        if (logger == null) {
            Logger.getRootLogger().warn(_("isisfish.error.log.closeAppender", name, category));
            return;
        }
        // get the required appender
        Appender app = logger.getAppender(name);
        if (app == null) {
            logger.warn(_("isisfish.error.log.foundAppender", name, category));
            return;
        }
        logger.info(_("isisfish.log.closeAppender", name, category));
        // close appender
        app.close();
        // and remove it from the logger
        logger.removeAppender(app);
    }
}