/* *##%
 * Copyright (C) 2006 - 2009
 *     Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package fr.ifremer.isisfish.export;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.Writer;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.beanutils.ConvertUtilsBean;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.persistence.TopiaEntity;

import fr.ifremer.isisfish.datastore.ExportStorage;
import fr.ifremer.isisfish.datastore.SensitivityExportStorage;
import fr.ifremer.isisfish.datastore.SimulationStorage;
import fr.ifremer.isisfish.util.ConverterUtil;

/**
 * Helper for exports manipulation.
 * 
 * Created: 20 janv. 2006 01:52:04
 *
 * @author poussin
 * @version $Revision: 2658 $
 *
 * Last update: $Date: 2009-10-13 17:00:50 +0200 (mar., 13 oct. 2009) $
 * by : $Author: chatellier $
 */
public class ExportHelper {

    /** Class logger. */
    private static Log log = LogFactory.getLog(ExportHelper.class);

    /**
     * Permet de faire les exports pour une simulation.
     * 
     * @param simulation La simulation pour lequel il faut faire les exports
     * @param destdir le repertoire dans lequel il faut ecrire les exports
     * @param exports les instances des exports à faire
     * @param compileDir le nom du répertoire ou les classes d'export sont compilées
     */
    public static void doExport(SimulationStorage simulation, File destdir,
            List<Export> exports, File compileDir) {

        // on ne compte plus ici les temps d'instanciations
        // deplacer dans SimulationExportResultWrapper#afterSimulation(SimulationContext)
        
        long timeStart = System.currentTimeMillis();
        for (Export export : exports) {
            String exportName = ExportStorage.getName(export);
            long time = System.currentTimeMillis();
            try {
                exportToFile(simulation, destdir, export);
            } catch (Exception eee) {
                if (log.isWarnEnabled()) {
                    log.warn("Can't export object: " + exportName, eee);
                }
            }
            simulation.getInformation().addExportTime(exportName,
                    System.currentTimeMillis() - time);
        }
        simulation.getInformation().addExportTime("All export",
                System.currentTimeMillis() - timeStart);
    }

    /**
     * Do single export.
     * 
     * @param simulation La simulation pour lequel il faut faire les exports
     * @param destdir le repertoire dans lequel il faut ecrire les exports
     * @param export le nom des exports a faire
     * @throws Exception si une erreur survient
     */
    protected static void exportToFile(SimulationStorage simulation,
            File destdir, Export export) throws Exception {
        String filename = export.getExportFilename();
        String extension = export.getExtensionFilename();

        File file = new File(destdir, filename + extension);
        // prevent two export with same name
        // name MyExport.csv become MyExport_1.csv
        int val = 0;
        while (file.exists()) {
            val++;
            file = new File(destdir, filename + extension + "_" + val);
        }

        Writer out = new BufferedWriter(new FileWriter(file));
        export.export(simulation, out);
        out.close();
    }

    /**
     * Recupere dans prop les valeurs des champs specifique au export et met a
     * jour les champs de l'export.
     * 
     * @param exportIndex l'index de l'export
     * @param export l'export a mettre à jour
     * @param context le topia context dont on a besoin
     * @param props les proprietes contenant les parametre de l'export
     */
    public static void populateSensitivityExport(int exportIndex, TopiaContext context,
            SensitivityExport export, Properties props) {
        //ConvertUtilsBean beanUtils = ConverterUtil.getConverter(region
        //        .getStorage());
        String exportName = SensitivityExportStorage.getName(export);
        String paramTag = "sensitivityexport." + exportIndex + ".parameter.";

        for (Map.Entry<String, Class<?>> entry : SensitivityExportStorage
                .getParameterNames(export).entrySet()) {
            String propName = entry.getKey();
            Class<?> type = entry.getValue();
            if (TopiaEntity.class.isAssignableFrom(type)) {
                type = TopiaEntity.class;
            }
            String valueString = "not initialized";
            try {
                valueString = props.getProperty(paramTag + propName);
                
                // ATTENTION il semblerait que notre intance de beanUtils
                // soit ecrasé par celle du MatrixType ...
                // a ne pas deplacer avant la boucle
                ConvertUtilsBean beanUtils = ConverterUtil.getConverter(context);
                
                Object value = beanUtils.convert(valueString, type);
                log.info("DEBUG: set sensitivityexport param: " + paramTag + propName
                        + " = " + value + "(" + valueString + ")");
                SensitivityExportStorage.setParameterValue(export, propName, value);
            } catch (Exception eee) {
                if (log.isWarnEnabled()) {
                    log.info("Properties: " + props);
                    log.warn("Can't reload field " + propName + " for export "
                            + exportName + " with value " + valueString, eee);
                }
            }
        }
    }
}
