/* *##%
 * Copyright (C) 2006 - 2010
 *     Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

/* *
 * ImportFromV2.java
 *
 * Created: 9 oct. 06 18:13:40
 *
 * @author poussin
 * @version $Revision: 2935 $
 *
 * Last update: $Date: 2010-01-22 16:37:21 +0100 (ven., 22 janv. 2010) $
 * by : $Author: chatellier $
 */

package fr.ifremer.isisfish.datastore.update;

import static org.nuiton.i18n.I18n._;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.StringReader;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.zip.GZIPInputStream;

import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.apache.commons.lang.time.DurationFormatUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.math.matrix.MatrixFactory;
import org.nuiton.math.matrix.MatrixHelper;
import org.nuiton.math.matrix.MatrixND;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.util.ArrayUtil;
import org.nuiton.util.FileUtil;
import org.nuiton.util.Resource;
import org.nuiton.util.VersionNumberUtil;
import org.dom4j.Document;
import org.dom4j.Element;
import org.dom4j.Node;
import org.dom4j.io.SAXReader;

import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.IsisFishDAOHelper;
import fr.ifremer.isisfish.datastore.RegionStorage;
import fr.ifremer.isisfish.entities.Cell;
import fr.ifremer.isisfish.entities.CellDAO;
import fr.ifremer.isisfish.entities.EffortDescription;
import fr.ifremer.isisfish.entities.EffortDescriptionDAO;
import fr.ifremer.isisfish.entities.Equation;
import fr.ifremer.isisfish.entities.FisheryRegion;
import fr.ifremer.isisfish.entities.Gear;
import fr.ifremer.isisfish.entities.GearDAO;
import fr.ifremer.isisfish.entities.Metier;
import fr.ifremer.isisfish.entities.MetierDAO;
import fr.ifremer.isisfish.entities.MetierSeasonInfoDAO;
import fr.ifremer.isisfish.entities.MetierSeasonInfoImpl;
import fr.ifremer.isisfish.entities.Population;
import fr.ifremer.isisfish.entities.PopulationDAO;
import fr.ifremer.isisfish.entities.PopulationGroup;
import fr.ifremer.isisfish.entities.PopulationGroupDAO;
import fr.ifremer.isisfish.entities.PopulationImpl;
import fr.ifremer.isisfish.entities.PopulationSeasonInfoDAO;
import fr.ifremer.isisfish.entities.PopulationSeasonInfoImpl;
import fr.ifremer.isisfish.entities.Port;
import fr.ifremer.isisfish.entities.PortDAO;
import fr.ifremer.isisfish.entities.Selectivity;
import fr.ifremer.isisfish.entities.SelectivityDAO;
import fr.ifremer.isisfish.entities.SetOfVessels;
import fr.ifremer.isisfish.entities.SetOfVesselsDAO;
import fr.ifremer.isisfish.entities.Species;
import fr.ifremer.isisfish.entities.SpeciesDAO;
import fr.ifremer.isisfish.entities.Strategy;
import fr.ifremer.isisfish.entities.StrategyDAO;
import fr.ifremer.isisfish.entities.StrategyMonthInfo;
import fr.ifremer.isisfish.entities.StrategyMonthInfoDAO;
import fr.ifremer.isisfish.entities.TargetSpecies;
import fr.ifremer.isisfish.entities.TargetSpeciesDAO;
import fr.ifremer.isisfish.entities.TripType;
import fr.ifremer.isisfish.entities.TripTypeDAO;
import fr.ifremer.isisfish.entities.VesselType;
import fr.ifremer.isisfish.entities.VesselTypeDAO;
import fr.ifremer.isisfish.entities.Zone;
import fr.ifremer.isisfish.entities.ZoneDAO;
import fr.ifremer.isisfish.equation.PopulationGrowthReverse;
import fr.ifremer.isisfish.equation.PopulationNaturalDeathRate;
import fr.ifremer.isisfish.equation.PopulationReproductionEquation;
import fr.ifremer.isisfish.equation.SelectivityEquation;
import fr.ifremer.isisfish.types.Month;
import fr.ifremer.isisfish.types.RangeOfValues;
import fr.ifremer.isisfish.types.TimeUnit;
import fr.ifremer.isisfish.ui.input.EquationEditorPaneUI;


/**
 * @author poussin
 *
 */

public class ImportFromV2 {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(ImportFromV2.class);
    
    static private final String XSL_DIRECTORY = "/v2xsl";
    static private final String LAST_2_VERSION = "2.3";
    
    Map<String, TopiaEntity> cache = new HashMap<String, TopiaEntity>();
    Element root;
    /** topiaId des equations non convertibles automatiquement */
    List<String> equationId = new ArrayList<String>();
    boolean manual = true;
    Node current = null;
    
    
    public ImportFromV2(boolean manual) {
        this.manual = manual;
    }
    
    /**
     * Commence par nettoyer le fichier en supprimant les simulations, script
     * et regle de gestion
     * 
     * Converti ensuite le fichier dans la version 2.3 des données 
     * 
     * @param file
     * @return le nouveau fichier converti et propre
     * @throws Exception 
     */
    private File convertXML(File file) throws Exception {

        log.info("Nettoyage du fichier");
        URL cleaner = Resource.getURL(XSL_DIRECTORY + "/isis-fish-cleaner.xsl");
        File dest = File.createTempFile("isis-tmp-dbconvert", "xml");
        convert(file, dest, cleaner);
        file = dest;

        log.info("recherche de la version");
        String version = "0";
        BufferedReader reader = new BufferedReader(new FileReader(file));
        String line = reader.readLine();
        for (int i=0; i<5 && !(line.contains("version=") && line.contains("<dbobjects")); i++) {
            // au max on lit 5 lignes, car normalement version est sur la 2eme
            line = reader.readLine();
        }
        if (line.contains("version=") && line.contains("<dbobjects")) {
            version = line.replaceAll(".*?version=['\"]([0-9.]+)['\"].*", "$1");
        }   

        log.info("Conversion du fichier depuis la version " + version + " vers " + LAST_2_VERSION);
        
        file = convertXML(file, version, LAST_2_VERSION);
        return file;
    }
    
    /**
     * Verifie la version du fichier, et le modifie si besoin avant de retourner
     * le flux. Si le fichier est plus récent que la version d'isis, une
     * exception est leve.
     * @param file
     * @param versionStart
     * @param versionEnd
     * @return
     * @throws Exception
     */
    private File convertXML(File file, String versionStart, String versionEnd) throws Exception {
         if(VersionNumberUtil.greaterThan(versionStart, versionEnd)){
             throw new Exception(_("isisfish.error.import.recent.files"));
         }else if(VersionNumberUtil.smallerThan(versionStart, versionEnd)){
             // upgrate du fichier
             try{
                 // recherche des fichiers XSL de conversion
                 URL [] xslurl = getXSLFile(versionStart, versionEnd);
                 if(xslurl == null){
                     throw new Exception(_("isisfish.error.import.convertible"));
                 }
                 File dest = file;
                 for (URL aXslurl : xslurl) {
                     File source = dest;
                     dest = File.createTempFile("isis-tmp-dbconvert", "xml");
                     convert(source, dest, aXslurl);
                 }
                 file = dest;
             }catch(Exception eee){
                 throw new Exception(_("isisfish.error.conversion.data"), eee);
             }
         }
         return file;
     }

     /**
     * Retourne les fichiers xsl qu'il faut appliquer dans l'ordre qu'il faut.
     * @param versionStart la version de départ du fichier
     * @param versionEnd la version auquel doit arriver le fichier
     * @return la liste des URLs des fichiers XSL a appliquer, ou null s'il est impossible de trouver les fichiers necessaire.
     */
     protected URL [] getXSLFile(String versionStart, String versionEnd) {
         List<URL> result = new ArrayList<URL>();
         List xslurl = Resource.getURLs(".*?" + XSL_DIRECTORY + "/isis-fish-"+versionStart+"_.*.xsl$");
         while(xslurl.size() != 0){
             URL xsl = (URL)xslurl.get(0);
             result.add(xsl);
             String end = getVersionEnd(xsl);
             if(VersionNumberUtil.equals(versionEnd, end)){
                 return result.toArray(new URL[result.size()]);
             }
             xslurl = Resource.getURLs(".*?" + XSL_DIRECTORY + "/isis-fish-"+end+"_.*.xsl$");
         }
         return null;
     }

     protected String getVersionEnd(URL xslurl){
         String version = xslurl.toExternalForm();
         version = version.substring(version.lastIndexOf("_")+1, version.length()-".xsl".length());
         return version;
     }
    
    /**
     * Converti un fichier XML en lui applicant un fichier XSL
     * @param sourceFile le fichier source XML
     * @param destFile le fichier résultat de l'opération
     * @param stylesheetURL l'url de la feuille de style a appliquer
     * @throws Exception
     */
    protected void convert(File sourceFile, File destFile, URL stylesheetURL) throws Exception {
        log.info("transforme data with " + stylesheetURL);
        StreamSource stylesheet = new StreamSource(stylesheetURL.openStream());
        Transformer transformer = TransformerFactory.newInstance().newTransformer(stylesheet);

        InputStream inf = new FileInputStream(sourceFile);
        if ("gz".equalsIgnoreCase(FileUtil.extension(sourceFile))) {
            inf = new GZIPInputStream(inf);
        }
        InputStreamReader in = new InputStreamReader(inf, "ISO-8859-1");
        StreamSource source = new StreamSource(in);

        FileOutputStream outf = new FileOutputStream(destFile);
        OutputStreamWriter out = new OutputStreamWriter(outf, "ISO-8859-1");
        StreamResult dest = new StreamResult(out);

        transformer.transform(source, dest);

        in.close();
        out.close();
    }

     @SuppressWarnings({"unchecked"})
     protected List<Node> selectNodes(Node node,String path) {
         List<Node> list;
         list = node.selectNodes(path);
         return list;
     }

      @SuppressWarnings({"unchecked"})
     protected List<Element> selectElements(Node node,String path) {
         List<Element> list;
         list = node.selectNodes(path);
         return list;
     }
    /**
     * Converti un fichier XML version 2 en un zip importable en v3
     * 
     * @param xml
     * @throws Exception 
     */
    public void importXML(File xml) throws Exception {
        log.info("importXML from file : " + xml);
        
        xml = convertXML(xml);
        
        // si le fichier fini par .gz alors le lire en le decompressant
        InputStream in = new BufferedInputStream(new FileInputStream(xml));
        if ("gz".equalsIgnoreCase(FileUtil.extension(xml))) {
            in = new GZIPInputStream(in);
        }
        
        SAXReader reader = new SAXReader();
        Document doc = reader.read(in);
        root = doc.getRootElement();

        List<Node> regionNames = selectNodes(root,"/dbobjects/dbobject[@type='Region']/attribute[@name='nom']");
        if (regionNames.size() == 0) {
            log.error("Can't find region in XML");
        }
        for (Node regionName : regionNames) {
            String name = regionName.getText();
            if (name != null && !"".equals(name)) {
                log.info("Region found: " + name);

                if (RegionStorage.exists(name)) {
                    log.error("region already exist, can't import " + name);
                } else {
                    long statTime = System.nanoTime();
                    log.info("import region ...");
                    RegionStorage storage = RegionStorage.create(name);
                    TopiaContext context = storage.getStorage().beginTransaction();
                    Element region = regionName.getParent();

                    try {
                        importRegion(context, region);
                        importCells(context, region);
                        importZones(context, region);
                        importPorts(context, region);
                        importSpecies(context, region);
                        importGears(context, region);
                        importMetiers(context, region);
                        importTripTypes(context, region);
                        importVesselTypes(context, region);
                        importSetOfVessels(context, region);
                        importStrategies(context, region);
                    } catch (Exception eee) {
                        if (log.isWarnEnabled()) {
                            log.warn("Error during import current XML was:\n" + current.asXML());
                        }
                        throw eee;
                    }
                    long endTime = System.nanoTime();
                    long time = (endTime - statTime) / 1000000;
                    log.info("Import time: " + DurationFormatUtils.formatDuration(time, "s'.'S"));
                            
                    context.commitTransaction();
                    log.info("region imported");
                    if (manual) {
                        log.info("Ask user for manual equation conversion");
//                        EquationEditorFrame frame = new EquationEditorFrame(true, true);
                        EquationEditorPaneUI frame = new EquationEditorPaneUI();
                        frame.setTitle(_("isisfish.message.import.equation.convert"));
                        for (String id : equationId) {
                            context = storage.getStorage().beginTransaction();
                            Equation eq = (Equation)context.findByTopiaId(id);
                            
//                            frame.getEquationEditorPane().setEquation(eq);
                            frame.setEquation(eq.getCategory(), eq.getName(), eq.getJavaInterface(), eq.getContent());
                            frame.setVisible(true);
                            if (frame.isOk()) {
                                eq.setContent(frame.getEditor().getText());
                            }
                            context.commitTransaction();
                        }
                        frame.dispose();
                    }
                    context.closeContext();
                }
            } else {
                log.error("Region found but name is empty, can't import it");
            }
        }
    }
    
    /**
     * @param context
     * @param region
     * @throws Exception 
     */
    private void importRegion(TopiaContext context, Element region) throws Exception {
        log.debug("import de la region");
        FisheryRegion region3 = RegionStorage.getFisheryRegion(context);
        cache.put(getId(region), region3);
        current = region; 
        region3.setComment(getStringValue(region, "commentaire"));
        region3.setCellLengthLatitude(getFloatValue(region, "pasMailleLatitude"));
        region3.setCellLengthLongitude(getFloatValue(region, "pasMailleLongitude"));
        region3.setMaxLatitude(getFloatValue(region, "maxLatitude"));
        region3.setMaxLongitude(getFloatValue(region, "maxLongitude"));
        region3.setMinLatitude(getFloatValue(region, "minLatitude"));
        region3.setMinLongitude(getFloatValue(region, "minLongitude"));
    }

    /**
     * @param context
     * @param region
     * @throws Exception 
     */
    private void importCells(TopiaContext context, Element region) throws Exception {
        List<Node> mailles = getListValue(region, "mailles");
        log.info("import de " + mailles.size() + " mailles");
        for (Node node : mailles) {
            Node maille = getElementById(node.getText());
            CellDAO dao = IsisFishDAOHelper.getCellDAO(context);
            Cell cell = dao.create();
            cache.put(getId(maille), cell);
            current = maille;
            cell.setName(getStringValue(maille, "nom"));                    
            cell.setLatitude(getFloatValue(maille, "latitude"));
            cell.setLongitude(getFloatValue(maille, "longitude"));
            cell.setLand(getBoolValue(maille, "terre"));
            cell.setComment(getStringValue(maille, "commentaire"));
        }
    }

    /**
     * @param context
     * @param region
     * @throws Exception 
     */
    private void importZones(TopiaContext context, Element region) throws Exception {
//        log.error("zones size ***** " + root.selectNodes("/dbobjects/dbobject[@type='SecteurSimple']/attribute[@name='region' and text()='" + getId(region) + "']/..").size());
        List<Node> secteurs = getListValue("SecteurSimple", "region", getId(region));
        log.info("import de " + secteurs.size() + " zones");
        for (Node secteur : secteurs) {
            ZoneDAO dao = IsisFishDAOHelper.getZoneDAO(context);
            Zone zone = dao.create();
            cache.put(getId(secteur), zone);
            current = secteur; 
            zone.setName(getStringValue(secteur, "nom"));             
            zone.setComment(getStringValue(secteur, "commentaire"));

            List<Node> ms = getListValue(secteur, "maille");
            for (Node maille : ms ) {
                Cell cell = (Cell)cache.get(maille.getText());
                zone.addCell(cell);
            }
        }

    }

    /**
     * @param context
     * @param region
     * @throws Exception 
     */
    private void importPorts(TopiaContext context, Element region) throws Exception {
        List<Node> ports = getListValue(region, "ports");
        log.info("import de " + ports.size() + " ports");
        for (Node node : ports) {
            Node port = getElementById(node.getText());
            PortDAO dao = IsisFishDAOHelper.getPortDAO(context);
            Port port3 = dao.create();
            cache.put(getId(port), port3);
            current = port;
            port3.setName(getStringValue(port, "name"));
            port3.setCell((Cell)cache.get(getStringValue(port, "maille")));
            port3.setComment(getStringValue(port, "commentaire"));
        }

    }

    /**
     * @param context
     * @param region
     * @throws Exception 
     */
    private void importSpecies(TopiaContext context, Element region) throws Exception {
        List<Node> metapopIds = getListValue(region, "metaPopulations");
        log.info("import de " + metapopIds.size() + " Species");
        for (Node metapopid : metapopIds) {
            Node metapop = getElementById(metapopid.getText());
            SpeciesDAO dao = IsisFishDAOHelper.getSpeciesDAO(context);
            Species species = dao.create();
            cache.put(getId(metapop), species);
            current = metapop;
            species.setName(getStringValue(metapop, "nomEspece"));
            species.setScientificName(getStringValue(metapop, "nomScientifique"));
            species.setCodeCEE(getIntValue(metapop, "codeCEE"));
            species.setCodeRubbin(getStringValue(metapop, "codeRubbin"));
            species.setAgeGroupType(getBoolValue(metapop, "enAge"));
            species.setComment(getStringValue(metapop, "commentaire"));

            List<Node> popIds = getListValue(metapop, "population");
            log.info("import de " + popIds.size() + " Population");
            for (Node popid : popIds) {
                Node pop = getElementById(popid.getText());
                PopulationDAO popdao = IsisFishDAOHelper.getPopulationDAO(context);
                PopulationImpl pop3 = (PopulationImpl)popdao.create();
                cache.put(getId(pop), pop3);
                current = pop;
                species.addPopulation(pop3);
                pop3.setSpecies(species);
                pop3.setName(getStringValue(pop, "nom"));
                pop3.setGeographicId(getStringValue(pop, "idGeographique"));
                pop3.setPlusGroup(getBoolValue(pop, "classePlus"));
                pop3.setGrowthContent(getEquationStringValue(pop, "croissance"));
                convertEquation(pop3.getGrowth(), getStringValue(pop, "croissance"));
                pop3.setGrowthReverseContent(getEquationStringValue(pop, "inverseCroissance"));
                convertEquation(pop3.getGrowthReverse(), getStringValue(pop, "inverseCroissance"));
                pop3.setNaturalDeathRateContent(getEquationStringValue(pop, "mortaliteNaturelleEquation"));
                convertEquation(pop3.getNaturalDeathRate(), getStringValue(pop, "mortaliteNaturelleEquation"));
                pop3.setReproductionEquationContent(getEquationStringValue(pop, "equationReproduction"));
                convertEquation(pop3.getReproductionEquation(), getStringValue(pop, "equationReproduction"));
                pop3.setMonthGapBetweenReproRecrutement(getIntValue(pop, "nbMoisEntreReproRecrutement"));
                pop3.setRecruitmentDistribution(getMatrixValue(pop, "etalementRecrutement")); //TODO la semantique devrait etre: sem.add(_("isisfish.common.month", i));
                pop3.setComment(getStringValue(pop, "commentaire"));

                List<Zone> zones = getZoneValue(pop, "zonePopulation");
                for (Zone zone : zones) {
                    pop3.addPopulationZone(zone);
                }
                    
                zones = getZoneValue(pop, "zoneRecrutement");
                for (Zone zone : zones) {
                    pop3.addRecruitmentZone(zone);
                }

                zones = getZoneValue(pop, "zoneReproduction");
                for (Zone zone : zones) {
                    pop3.addReproductionZone(zone);
                }

                MatrixND mat = pop3.getMappingZoneReproZoneRecru();
                if (mat != null) {
                    mat = mat.copy();
                    List<Node> mappings = getListValue(pop, "mappingZoneReproZoneRecru");
                    for (Node mappingId : mappings) {
                        Node mapping = getElementById(mappingId.getText());
                        Zone zoneRepro = (Zone) cache.get(getStringValue(mapping, "zoneReproduction"));
                        Zone zoneRecru = (Zone) cache.get(getStringValue(mapping, "zoneRecrutement"));
                        double coeff = getDoubleValue(mapping, "coeff");
                        mat.setValue(zoneRepro, zoneRecru, coeff);
                    }
                    pop3.setMappingZoneReproZoneRecru(mat);
                }

                String eqMeanWeight = 
                    "if (group == null) return 0;\n" +
                    "  switch (group.getId()) {\n";
                String eqPrice = 
                    "if (group == null) return 0;\n" +
                    "  switch (group.getId()) {\n";

                int classMat = getIntValue(pop, "classeMature");
                List<Node> classes = getListValue(pop, "classes");
                for (Node classeId : classes) {
                    Node classe = getElementById(classeId.getText());
                    PopulationGroupDAO daogroup = IsisFishDAOHelper.getPopulationGroupDAO(context);
                    PopulationGroup group = daogroup.create();
                    cache.put(getId(classe), group);
                    current = classe;
                    pop3.addPopulationGroup(group);
                    group.setPopulation(pop3);

                    group.setId(getIntValue(classe, "id"));
                    group.setAge(getDoubleValue(classe, "age"));
                    group.setReproductionRate(getDoubleValue(classe, "coefficientFecondite"));
                    group.setMinLength(getDoubleValue(classe, "longueurMin"));
                    group.setMaxLength(getDoubleValue(classe, "longueurMax"));
                    group.setComment(getStringValue(classe, "commentaire"));

                    
                    eqMeanWeight += "    case " + group.getId() + ": return " + getStringValue(classe, "poidsMoyen") + ";\n";
                    eqPrice += "    case " + group.getId() + ": return " + getStringValue(classe, "price") + ";\n";

                    if (group.getId() == classMat) {
                        pop3.setMaturityGroup(group);
                    }
                }

                eqMeanWeight += 
                    "    default: return 0;\n" +
                    "  }\n";                        
                pop3.setMeanWeightContent(eqMeanWeight);

                eqPrice += 
                    "    default: return 0;\n" +
                    "  }\n";                        
                pop3.setPriceContent(eqPrice);

                String seasonComment = "";
                List<Node> infoSaisons = getListValue(pop, "infoSaison");
                for (Node infoSaisonId : infoSaisons) {
                    Node infoSaison = getElementById(infoSaisonId.getText());
                    PopulationSeasonInfoDAO daoSeason = IsisFishDAOHelper.getPopulationSeasonInfoDAO(context);
                    PopulationSeasonInfoImpl season = (PopulationSeasonInfoImpl)daoSeason.create();
                    cache.put(getId(infoSaison), season);
                    current = infoSaison; 
                        
                    pop3.addPopulationSeasonInfo(season);
                    season.setPopulation(pop3);
                    
                    season.setGroupChange(getBoolValue(infoSaison, "changementClass"));
                    season.setSimpleLengthChangeMatrix(getBoolValue(infoSaison, "matriceChangementLongueurSimple"));
                    season.setReproduction(getBoolValue(infoSaison, "reproduction"));
                    season.setUseEquationMigration(getBoolValue(infoSaison, "useMigrationEquation"));
                    Month first = getFirstMonthSeason(infoSaison, "saison");
                    Month last = getLastMonthSeason(infoSaison, "saison");
                    season.setFirstMonth(first);
                    season.setLastMonth(last);
                    
                    mat = season.createNoSpacializedChangeGroupMatrix();
                    if (!season.getSimpleLengthChangeMatrix()) {
                        mat = season.spacializeLengthChangeMatrix(mat);
                    }
                    mat.paste(getMatrixValue(infoSaison, "matriceChangementLongueur"));                                
                    season.setLengthChangeMatrix(mat);

                    mat = season.getReproductionDistribution().copy();
                    mat.paste(getMatrixValue(infoSaison, "distributionRepro"));
                    season.setReproductionDistribution(mat);

                    if (season.getUseEquationMigration()) {
                        season.setMigrationEquationContent(getEquationStringValue(infoSaison, "migrationEquation"));
                        convertEquation(season.getMigrationEquation(), getStringValue(infoSaison, "migrationEquation"));
                        season.setEmigrationEquationContent(getEquationStringValue(infoSaison, "emigrationEquation"));
                        convertEquation(season.getEmigrationEquation(), getStringValue(infoSaison, "emigrationEquation"));
                        season.setImmigrationEquationContent(getEquationStringValue(infoSaison, "immigrationEquation"));
                        convertEquation(season.getImmigrationEquation(), getStringValue(infoSaison, "immigrationEquation"));                        
                    } else {
                        mat = season.getMigrationMatrix().copy();
                        List<Node> migs = getListValue(infoSaison, "migration");
                        for (Node migId : migs) {
                            Node migration = getElementById(migId.getText());
                            current = migration;
                            PopulationGroup group = (PopulationGroup)cache.get(getStringValue(migration, "classe"));
                            double coeff = getDoubleValue(migration, "coefficient");
                            Zone zoneDep = (Zone)cache.get(getStringValue(migration, "secteurDepart"));
                            Zone zoneArr = (Zone)cache.get(getStringValue(migration, "secteurArrivee"));
                            try {
                                mat.setValue(group, zoneDep, zoneArr, coeff);
                            } catch (Exception eee) {
                                log.warn("Error during migration import for " + season, eee);
                            }
                        }
                        season.setMigrationMatrix(mat);

                        mat = season.getEmigrationMatrix().copy();                            
                        List<Node> emigs = getListValue(infoSaison, "emigration");
                        for (Node emigId : emigs) {
                            Node emigration = getElementById(emigId.getText());
                            current = emigration;
                            PopulationGroup group = (PopulationGroup)cache.get(getStringValue(emigration, "classe"));
                            double coeff = getDoubleValue(emigration, "coefficient");
                            Zone zoneDep = (Zone)cache.get(getStringValue(emigration, "secteurDepart"));
                            try {
                                mat.setValue(group, zoneDep, coeff);
                            } catch (Exception eee) {
                                log.warn("Error during emigration import for " + season, eee);
                            }
                        }
                        season.setEmigrationMatrix(mat);

                        mat = season.getImmigrationMatrix().copy();
                        List<Node> immigs = getListValue(infoSaison, "immigration");
                        for (Node immigId : immigs) {
                            Node immigration = getElementById(immigId.getText());
                            current = immigration;
                            PopulationGroup group = (PopulationGroup)cache.get(getStringValue(immigration, "classe"));
                            double coeff = getDoubleValue(immigration, "coefficient");
                            Zone zoneArr = (Zone)cache.get(getStringValue(immigration, "secteurArrivee"));
                            try {
                                mat.setValue(group, zoneArr, coeff);
                            } catch (Exception eee) {
                                log.warn("Error during immigration import for " + season, eee);
                            }
                        }
                        season.setImmigrationMatrix(mat);
                    }

                    seasonComment += getStringValue(infoSaison, "commentaire");

                    mat = pop3.getCapturability().copy();
                    List<Node> captus = getListValue(infoSaison, "capturabilite");
                    for (Node captuId : captus) {
                        Node captu = getElementById(captuId.getText());
                        PopulationGroup group = (PopulationGroup)cache.get(getStringValue(captu, "classe"));
                        double coeff = getDoubleValue(captu, "coefficient");
                        mat.setValue(group, season, coeff);
                    }
                    pop3.setCapturability(mat);
                }
                pop3.setSeasonsComment(seasonComment);                                            
            }
        }
    }

    /**
     * @param context
     * @param region
     * @throws Exception 
     */
    private void importGears(TopiaContext context, Element region) throws Exception {
//        List<Node> engins = root.selectNodes("/dbobjects/dbobject[@type='Engin']/attribute[@name='region' and text()='" + getId(region) + "']/..");
        List<Node> engins = getListValue("Engin", "region", getId(region));
        log.info("import de " + engins.size() + " Engins");
        for (Node engin : engins) {
            GearDAO dao = IsisFishDAOHelper.getGearDAO(context);
            Gear gear = dao.create();
            cache.put(getId(engin), gear);
            current = engin;
            gear.setName(getStringValue(engin, "nom"));
            gear.setCost(getDoubleValue(engin, "cost"));
            gear.setStandardisationFactor(getDoubleValue(engin, "facteurStandard"));
            gear.setParameterName(getStringValue(engin, "parametreNom"));
            gear.setEffortUnit(getStringValue(engin, "uniteMesureEffort"));
            gear.setPossibleValue(getRangeOfValues(engin, "parametreGammePossible"));
            gear.setComment(getStringValue(engin, "commentaire"));

            List<Node> selectivites = getListValue(engin, "selectivite");
            for (Node selectiviteId : selectivites) {
                Node selectivite = getElementById(selectiviteId.getText());
                SelectivityDAO daoSelectivity = IsisFishDAOHelper.getSelectivityDAO(context);
                Selectivity selectivity = daoSelectivity.create();
                cache.put(getId(selectivite), selectivity);
                current = selectivite;
                gear.addPopulationSelectivity(selectivity);
                selectivity.setGear(gear);
                selectivity.setPopulation((Population)cache.get(getStringValue(selectivite, "population")));
                Equation eq = selectivity.getEquation();
                eq.setContent(getEquationStringValue(selectivite, "equation"));
                convertEquation(eq, getStringValue(selectivite, "equation"));
                selectivity.setEquation(eq);
                
            }
        }

    }

    /**
     * @param context
     * @param region
     * @throws Exception 
     */
    private void importMetiers(TopiaContext context, Element region) throws Exception {
        List<Node> metiers = getListValue(region, "metiers");
        log.info("import de " + metiers.size() + " Metiers");
        for (Node metierId : metiers) {
            Node metier = getElementById(metierId.getText());
            MetierDAO dao = IsisFishDAOHelper.getMetierDAO(context);
            Metier metier3 = dao.create();
            cache.put(getId(metier), metier3);
            current = metier;
            metier3.setName(getStringValue(metier, "nom"));
            metier3.setGear((Gear)cache.get(getStringValue(metier, "engin")));
            metier3.setGearParameterValue(getStringValue(metier, "valeurParamControlable"));
            metier3.setComment(getStringValue(metier, "commentaire"));

            List<Node> infoSaisons = getListValue(metier, "infoSaison");
            for (Node infoSaisonId : infoSaisons) {
                Node infoSaison = getElementById(infoSaisonId.getText());
                MetierSeasonInfoDAO daoSeason = IsisFishDAOHelper.getMetierSeasonInfoDAO(context);
                MetierSeasonInfoImpl season = (MetierSeasonInfoImpl)daoSeason.create();
                cache.put(getId(infoSaison), season);
                current = infoSaison;
                metier3.addMetierSeasonInfo(season);
                season.setMetier(metier3);

                Month first = getFirstMonthSeason(infoSaison, "saison");
                Month last = getLastMonthSeason(infoSaison, "saison");
                season.setFirstMonth(first);
                season.setLastMonth(last);

                String commentTarget = "";
                List<Node> especesCaptus = getListValue(infoSaison, "especesCaptu");
                for (Node especesCaptuId : especesCaptus) {
                    Node especesCaptu = getElementById(especesCaptuId.getText());
                    TargetSpeciesDAO daoTarget = IsisFishDAOHelper.getTargetSpeciesDAO(context);
                    TargetSpecies target = daoTarget.create();
                    cache.put(getId(especesCaptu), target);
                    current = especesCaptu;
                    season.addSpeciesTargetSpecies(target);
                    target.setMetierSeasonInfo(season);
                    target.setPrimaryCatch(getBoolValue(especesCaptu, "capturePrimaire"));
                    target.setSpecies((Species)cache.get(getStringValue(especesCaptu, "metaPopulation")));
                    Equation eq = target.getTargetFactorEquation();
                    eq.setContent(getEquationStringValue(especesCaptu, "equation"));
                    convertEquation(eq, getStringValue(especesCaptu, "equation"));
                    target.setTargetFactorEquation(eq);
                    commentTarget += getStringValue(especesCaptu, "commentaire");
                }
                metier3.setCapturableSpeciesComment(commentTarget);

                List<Zone> zones = getZoneValue(infoSaison, "secteur");
                for (Zone zone : zones) {
                    season.addZone(zone);
                }
            }
        }   
    }

    /**
     * @param context
     * @param region
     * @throws Exception 
     */
    private void importTripTypes(TopiaContext context, Element region) throws Exception {
        String regionId = getId(region);
//        log.error(regionId + "******" + root.selectNodes("/dbobjects/dbobject[@type='TripType']/attribute[@name='region' and text()='" + regionId + "']").size());
        List<Node> tripTypes = getListValue("TripType", "region", getId(region));
        log.info("import de " + tripTypes.size() + " TripTypes");
        for (Node tripType : tripTypes) {
            TripTypeDAO dao = IsisFishDAOHelper.getTripTypeDAO(context);
            TripType tripType3 = dao.create();
            cache.put(getId(tripType), tripType3);
            current = tripType;
            tripType3.setName(getStringValue(tripType, "name"));
            tripType3.setTripDuration(new TimeUnit(0).setDay(getDoubleValue(tripType, "tripDuration")));
            tripType3.setMinTimeBetweenTrip(new TimeUnit(0).setDay(getDoubleValue(tripType, "minTimeBetweenTrip")));
            tripType3.setComment(getStringValue(tripType, "commentaire"));
        }
    }

    /**
     * @param context
     * @param region
     * @throws Exception 
     */
    private void importSetOfVessels(TopiaContext context, Element region) throws Exception {
//        List<Element> setOfVessels = root.selectNodes("/dbobjects/dbobject[@type='SetOfVessels']/attribute[@name='region' and text()='" + getId(region) + "']/..");
        List<Node> setOfVessels = getListValue("SetOfVessels", "region", getId(region));
        log.info("import de " + setOfVessels.size() + " SetOfVessels");
        for (Node sov : setOfVessels) {
            SetOfVesselsDAO dao = IsisFishDAOHelper.getSetOfVesselsDAO(context);
            SetOfVessels sov3 = dao.create();
            cache.put(getId(sov), sov3);
            current = sov;
            sov3.setName(getStringValue(sov, "name"));
            sov3.setCapitalDeprecation(getDoubleValue(sov, "capitalDeprecation"));
            sov3.setFixedCosts(getDoubleValue(sov, "fixedCosts"));
            sov3.setInterestCost(getDoubleValue(sov, "interestCosts"));
            sov3.setNumberOfVessels(getIntValue(sov, "numberOfVessels"));
            sov3.setPort((Port)cache.get(getStringValue(sov, "port")));
            sov3.setVesselCosts(getDoubleValue(sov, "vesselCosts"));
            sov3.setVesselType((VesselType)cache.get(getStringValue(sov, "vesselType")));
            sov3.setComment(getStringValue(sov, "commentaire"));

            List<Node> possibleMetiers = getListValue(sov, "possibleMetiers");
            log.trace("import de " + possibleMetiers.size() + " EffortDescription");
            for (Node node : possibleMetiers) {
                Node effort = getElementById(node.getText());
                EffortDescriptionDAO daoEffort = IsisFishDAOHelper.getEffortDescriptionDAO(context);
                EffortDescription effort3 = daoEffort.create();
                cache.put(getId(effort), effort3);
                current = effort;
                sov3.addPossibleMetiers(effort3);
                effort3.setSetOfVessels(sov3);
                effort3.setCrewFoodCost(getDoubleValue(effort, "crewFoodCost"));
                effort3.setCrewShareRate(getDoubleValue(effort, "crewShareRate"));
                effort3.setCrewSize(getIntValue(effort, "crewSize"));
                effort3.setFishingOperation(getIntValue(effort, "fishingOperation"));
                effort3.setFishingOperationDuration(new TimeUnit(0).setDay(getDoubleValue(effort, "fishingOperationDuration")));
                effort3.setFixedCrewSalary(getDoubleValue(effort, "fixedCrewSalary"));
                effort3.setGearsNumberPerOperation(getIntValue(effort, "gearsNumberPerOperation"));
                effort3.setLandingCosts(getDoubleValue(effort, "landingCosts"));
                effort3.setPossibleMetiers((Metier)cache.get(getStringValue(effort, "metier")));
                effort3.setOtherRunningCost(getDoubleValue(effort, "otherRunningCosts"));
                effort3.setRepairAndMaintenanceGearCost(getDoubleValue(effort, "repairAndMaintenanceGearCost"));
                effort3.setSetOfVessels((SetOfVessels)cache.get(getStringValue(effort, "setOfVessels")));
                effort3.setUnitCostOfFishing(getDoubleValue(effort, "unitCostOfFishing"));                
            }
        }
    }

    /**
     * @param context
     * @param region
     * @throws Exception 
     */
    private void importVesselTypes(TopiaContext context, Element region) throws Exception {
//        List<Element> vesselTypes = root.selectNodes("/dbobjects/dbobject[@type='VesselType']/attribute[@name='region' and text()='" + getId(region) + "']/..");

        List<Node> vesselTypes = getListValue("VesselType", "region", getId(region));
        log.info("import de " + vesselTypes.size() + " VesselTypes");
        for (Node vesselType : vesselTypes) {
            VesselTypeDAO dao = IsisFishDAOHelper.getVesselTypeDAO(context);
            VesselType vesselType3 = dao.create();
            cache.put(getId(vesselType), vesselType3);
            current = vesselType;
            vesselType3.setName(getStringValue(vesselType, "name"));
            vesselType3.setActivityRange(getDoubleValue(vesselType, "activityRange"));
            vesselType3.setLength(getIntValue(vesselType, "length"));
            vesselType3.setMaxTripDuration(new TimeUnit(0).setDay(getDoubleValue(vesselType, "maxTripDuration")));
            vesselType3.setMinCrewSize(getIntValue(vesselType, "minCrewSize"));
            vesselType3.setSpeed(getDoubleValue(vesselType, "speed"));
            vesselType3.setUnitFuelCostOfTravel(getDoubleValue(vesselType, "unitFuelCostOfTravel"));
            vesselType3.setComment(getStringValue(vesselType, "commentaire"));

            List<Node> possibleTripTypes = getListValue(vesselType, "possibleTripTypes");
            for (Node node : possibleTripTypes) {
                vesselType3.addTripType((TripType)cache.get(node.getText()));
            }
        }
    }

    /**
     * @param context
     * @param region
     * @throws Exception 
     */
    private void importStrategies(TopiaContext context, Element region) throws Exception {
        List<Node> strategies = getListValue(region, "strategies");
        log.info("import de " + strategies.size() + " Strategies");
        for (Node node : strategies) {
            Node strategie = getElementById(node.getText());
            StrategyDAO dao = IsisFishDAOHelper.getStrategyDAO(context);
            Strategy strategy = dao.create();
            cache.put(getId(strategie), strategy);
            current = strategie;
            strategy.setName(getStringValue(strategie, "name"));
            strategy.setProportionSetOfVessels(getDoubleValue(strategie, "proportionSetOfVessels"));
            strategy.setSetOfVessels((SetOfVessels)cache.get(getStringValue(strategie, "setOfVessels")));
            strategy.setComment(getStringValue(strategie, "commentaire"));

            List<StrategyMonthInfo> allsmi = new ArrayList<StrategyMonthInfo>(12);
            List<Node> strategyMonthInfos = getListValue(strategie, "strategyMonthInfos");
            log.debug("import de " + strategyMonthInfos.size() + " StrategyMonthInfo for " + strategy.getName());
            for (Node strategyMonthInfoId : strategyMonthInfos) {
                Node smi = getElementById(strategyMonthInfoId.getText());
                StrategyMonthInfoDAO daoSMI = IsisFishDAOHelper.getStrategyMonthInfoDAO(context);
                StrategyMonthInfo smi3 = daoSMI.create();
                cache.put(getId(smi), smi3);
                current = smi;
                allsmi.add(smi3);
                smi3.setStrategy(strategy);
                smi3.setMonth(Month.MONTH[getIntValue(smi, "month")]);
                smi3.setMinInactivityDays(getIntValue(smi, "minInactivityDays"));
                smi3.setNumberOfTrips(getIntValue(smi, "numberOfTrips"));
                smi3.setTripType((TripType)cache.get(getStringValue(smi, "tripType")));                        

                List<Node> propStrMets = getListValue("PropStrMet", "strategyMonthInfo", getId(smi));
                //root.selectNodes("/dbobjects/dbobject[@type='PropStrMet']/attribute[@name='strategyMonthInfo' and text()='" + getId(smi) + "']/..");
                log.trace("Import de " + propStrMets.size() + " propStrMets");
                for (Node propStrMet : propStrMets) {
                    current = propStrMet;
                    Metier metier = (Metier)cache.get(getStringValue(propStrMet, "metier"));
                    double prop = getDoubleValue(propStrMet, "proportion");
                    try {
                        smi3.setProportionMetier(metier, prop);
                    } catch (NoSuchElementException eee) {
                        // Il arrive parfois que le metier ne soit pas un metier
                        // possible, du coup on ne fait rien. (sans doute
                        // une erreur d'effacement en v2
                        if (log.isDebugEnabled()) {
                            log.debug("Can't set prop for: " + current.asXML());
                        }
                    }
                }
            }
            strategy.clearStrategyMonthInfo();
            strategy.addAllStrategyMonthInfo(allsmi);

        }
    }

    /**
     * Retourne tous les objets d'un certain type qui on comme valeur, pour le 
     * champs attribute, id
     * @param type
     * @param attribute
     * @param id
     * @return
     */
    private List<Node> getListValue(String type, String attribute, String id) {
        List<Node> tmp = selectNodes(root,"/dbobjects/dbobject[@type='" + type + "']");
        List<Node> result = new ArrayList<Node>();
        for (Node e : tmp) {
            if (id.equals(getStringValue(e, attribute))) {
                result.add(e);
            }
        }
        return result;
    }
    
    /**
     * @param node
     * @param attribute
     * @return
     */
    private List<Zone> getZoneValue(Node node, String attribute) {
        List<Zone> result = new ArrayList<Zone>();
        
        String id = getStringValue(node, attribute);
        if (id.contains("fr.ifremer.db.SecteurSimpleFactory")) {
            result.add((Zone)cache.get(id));
        } else {        
            Node metasecteur = getElementById(id);
            if (metasecteur != null) {
                List<Node> secteurs = getListValue(metasecteur, "secteur");
                for (Node secteurSimple : secteurs) {
                    result.add((Zone)cache.get(secteurSimple.getText()));
                }
            }
        }

        return result;
    }

    /**
     * @param node
     * @param attribute
     * @return
     * @throws Exception 
     */
    private RangeOfValues getRangeOfValues(Node node, String attribute) throws Exception {
        String paramString = getStringValue(node, attribute);
        
        SAXReader reader = new SAXReader();
        Document doc = reader.read(new StringReader(paramString));
        Element param = doc.getRootElement();

        boolean cont = "true".equalsIgnoreCase(param.attributeValue("continue"));
        String type = RangeOfValues.TYPES[0];
        String result = "";
        String sep = "";
        List<Element> values = selectElements(param,"value");
        for (Element value : values) {
            String oldType = value.attributeValue("type");
            result += sep + value.getText();
                        
            if (cont) {
                sep = "-";
            } else {
                sep = ";";
            }

            if ("String".equalsIgnoreCase(oldType)) {
                type = RangeOfValues.TYPES[0];
            } else if ("Double".equalsIgnoreCase(oldType) || "Float".equalsIgnoreCase(oldType)) {
                type = RangeOfValues.TYPES[2];
            } else {
                type = RangeOfValues.TYPES[1];
            } 
        }
        result = type + "[" + result + "]";
        
        return new RangeOfValues(result);
    }

    /**
     * @param node
     * @param attribute
     * @return
     * @throws Exception 
     */
    private String getEquationStringValue(Node node, String attribute) throws Exception {
        String equation = getStringValue(node, attribute);
        
        SAXReader reader = new SAXReader();
        Document doc = reader.read(new StringReader(equation));
        Element eq = doc.getRootElement();

        String result;
        result = eq.getText();
        return result;
    }

    /**
     * 
     * @param e l'equation a passer en Java
     * @param xml la representation XML de l'equation originale
     */
    private void convertEquation(Equation e, String xml) {
        String content = e.getContent();
        if (content.trim().length() == 0) {
            content = "return 0;";
        } else {
            try {
                Double.parseDouble(content);
                // juste un nombre on le retourne
                content = "return " + content + ";";
            } catch (Exception eee) {
                content = content.replaceAll("^#", "//");
                content = content.replaceAll("Ln", "Math.log");
                content = content.replaceAll("Exp", "Math.exp");
                content = content.replaceAll("getNom", "getName");
                content = content.replaceAll("getNumMois()", "getMonthNumber()");
                if (e.getJavaInterface().equals(SelectivityEquation.class)) {
                    content = content.replaceAll("longueur", "group.getLength()");
                    content = content.replaceAll("getValeurParamControlable", "getGearParameterValueAsDouble");
                    content = content.replaceAll("getValeurParamControlableAsNumber", "getGearParameterValueAsDouble");
                }
                if (e.getJavaInterface().equals(PopulationGrowthReverse.class)) {
                    content = content.replaceAll("longueur", "length");
                }
                if (e.getJavaInterface().equals(PopulationNaturalDeathRate.class) || 
                        e.getJavaInterface().equals(PopulationReproductionEquation.class)) {
                    content = content.replaceAll("classe", "group");
                    content = content.replaceAll("getCoefficientFecondite", "getReproductionRate");
                    content = content.replaceAll("getPoidsMoyen", "getMeanWeight");
                }
                if (!manual) {
                    log.warn("Equation must be converted by you: " + xml);
                    content = "/*\n" + content + "\n*/\n";
                    content += "throw new Exception(\"Equation imported from isis v2. You must rewrite equation in Java:\");\n";
                } else {
                    log.debug("Equation must be converted by you: " + xml);
                }
                equationId.add(e.getTopiaId());
            }
        }
        e.setContent(content);
    }
    
    /**
     * @param node
     * @param attribute
     * @return
     * @throws Exception 
     */
    private Month getLastMonthSeason(Node node, String attribute) throws Exception {
        String seasonXML = getStringValue(node, attribute);

        SAXReader reader = new SAXReader();
        Document doc = reader.read(new StringReader(seasonXML));
        Element season = doc.getRootElement();
        
        String monthString = season.attributeValue("last");
        Month result;
        result = Month.MONTH[Integer.parseInt(monthString)];
        return result;
    }

    /**
     * @param node
     * @param attribute
     * @return
     * @throws Exception 
     */
    private Month getFirstMonthSeason(Node node, String attribute) throws Exception {
        String seasonXML = getStringValue(node, attribute);

        SAXReader reader = new SAXReader();
        Document doc = reader.read(new StringReader(seasonXML));
        Element season = doc.getRootElement();
        
        String monthString = season.attributeValue("first");
        Month result;
        result = Month.MONTH[Integer.parseInt(monthString)];
        return result;
    }

    /**
     * @param node
     * @param attribute
     * @return
     * @throws Exception 
     */
    private MatrixND getMatrixValue(Node node, String attribute) throws Exception {
        String matrixString = getStringValue(node, attribute);
        
//        MatrixND result = (MatrixND)decoder.decode(matrixString);
        SAXReader reader = new SAXReader();
        Document doc = reader.read(new StringReader(matrixString));
        Element mat = (Element)doc.getRootElement().selectSingleNode("/object/matrix");
        if (mat == null) {
            mat = (Element)doc.getRootElement().selectSingleNode("/object/MatriceND");
        }
        
        String defaultValueString = mat.attributeValue("defaultValue");
        if (defaultValueString == null) {
            defaultValueString = mat.attributeValue("defaultElem");
        }
        double defaultValue = Double.parseDouble(defaultValueString);
        int [] dim = ArrayUtil.asIntArray(mat.attributeValue("dimensions").split("[,;]"));
        
        MatrixND result = MatrixFactory.getInstance().create(dim);
        MatrixHelper.fill(result, defaultValue);
            
        List<Element> elems = selectElements(mat,"element");
        for (Element e : elems) {
            int [] coord = ArrayUtil.asIntArray(e.attributeValue("path").split(","));
            double value = Double.parseDouble(e.attributeValue("value"));
            result.setValue(coord, value);
        }
                
        return result;
    }

    private String getId(Node node) {
        Element e = (Element)node;
        String result;
        result = e.attributeValue("__id__");
        return result;
    }
    
    /**
     * 
     * @param id
     * @return
     */
    private Node getElementById(String id) {
        Node result;
        result = root.selectSingleNode("/dbobjects/dbobject[@__id__='" + id + "']");
        return result;
    }
    
    private boolean getBoolValue(Node node, String attribute) {
        String val = getStringValue(node, attribute);
        boolean result;
        result = "true".equalsIgnoreCase(val);
        return result;
    }
    
    private int getIntValue(Node node, String attribute) {
        String val = getStringValue(node, attribute);
        int result;
        result = Integer.parseInt(val);
        return result;
    }
    
    private double getDoubleValue(Node node, String attribute) {
        String val = getStringValue(node, attribute);
        double result;
        result = Double.parseDouble(val);
        return result;
    }
    
    private float getFloatValue(Node node, String attribute) {
        String val = getStringValue(node, attribute);
        float result;
        result = Float.parseFloat(val);
        return result;
    }
    
    @SuppressWarnings({"unchecked"})
    private List<Node> getListValue(Node node, String attribute) {
        List<Node> result;
        result = node.selectNodes("attribute[@name='" + attribute + "']");
        return result;
    }
    
    private String getStringValue(Node node, String attribute) {
        String result = "";
        Node att = node.selectSingleNode("attribute[@name='" + attribute + "']");
        if (att != null) {
            result = att.getText();
        } else {
            log.warn("Can't fing attribute : '" + attribute + "' on node: " + node);
        }
        return result;
    }
    
    public static void main(String[] args) throws Exception {
        IsisFish.init();
        boolean interactif = true;
        if (args.length == 0) {
            System.out.println("Usage ImportFromV2 <xml file> [<xml file> [<xml file> ...]]");
            new ImportFromV2(interactif).importXML(new File(
                    "/usr/local/src/PROJET/lutin/isis-fish/base-a-migrer/Iroise250706.xml.gz"));
            //AppliTECTAC-2.3.2.xml.gz"));
        } else {
            for (String filename : args) {
                if ("--nomanual".equals(filename)) {
                    interactif = false;
                } else {
                    new ImportFromV2(interactif).importXML(new File(filename));
                }
            }
        }
    }
}


