/* *##%
 * Copyright (C) 2007 - 2009 Ifremer, Code Lutin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package fr.ifremer.isisfish.datastore.migration;

import static org.nuiton.i18n.I18n._;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;

import javax.swing.JOptionPane;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.migration.DatabaseManager;
import org.nuiton.topia.migration.callback.MigrationCallbackHandler;
import org.nuiton.util.VersionNumberUtil;

/**
 * MigrationCallbackHandler.
 *
 * @author chatellier
 * @version $Revision: 3014 $
 *
 * Last update : $Date: 2010-04-07 10:25:35 +0200 (mer., 07 avril 2010) $
 * By : $Author: chatellier $
 */
public class SwingMigrationCallbackHandler implements MigrationCallbackHandler {

    /** to use log facility, just put in your code: log.info("..."); */
    static private Log log = LogFactory
            .getLog(SwingMigrationCallbackHandler.class);

    /*
     * @see org.nuiton.topia.migration.callback.MigrationCallbackHandler#doMigration(java.lang.String, java.lang.String)
     */
    @Override
    public MigrationChoice doMigration(DatabaseManager dbManager,
            String dbVersion, String applicationVersion) {

        int answer = JOptionPane.showConfirmDialog(
                null, _("isisfish.misc.databasemigration.question", dbVersion, applicationVersion),
                _("isisfish.misc.databasemigration.title"),
                JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE);

        MigrationChoice result = (answer == JOptionPane.YES_OPTION) ? MigrationChoice.MIGRATION
                : MigrationChoice.NO_MIGRATION;
        
        if (result == MigrationChoice.MIGRATION) {
            result = migrate(dbManager, dbVersion, applicationVersion);
        }
        return result;
    }

    /**
     * Perform real migration.
     * 
     * @param dbManager database manager
     * @param dbVersion database version
     * @param applicationVersion application version
     * @return {@code MigrationChoice.CUSTOM_MIGRATION} if success, {@code MigrationChoice.NO_MIGRATION} otherwise.
     */
    protected MigrationChoice migrate(DatabaseManager dbManager,
            String dbVersion, String applicationVersion) {
        // migration manuel sans l'aide de topia-service
        // tant que c pas fait on dit non
        MigrationChoice result = MigrationChoice.NO_MIGRATION;
        // ouverture d'une connexion direct JDBC sur la base
        try {
            Connection conn = dbManager.getConnection();
            try {
                conn.setAutoCommit(false);
                if (VersionNumberUtil.smallerThan(dbVersion, "3.2")) {
                    // on fait une migration vers 3.2
                    migrateTo3_2(conn);
                }

                // do same thing for other version
                if (VersionNumberUtil.smallerThan(dbVersion, "3.3")) {
                    // on fait une migration vers 3.2
                    migrateTo3_3(conn);
                }

                // commit des modifs
                conn.commit();
                // la migration a reussi, on dit que c fait
                result = MigrationChoice.CUSTOM_MIGRATION;
            } catch (Exception eee) {
                // en cas d'erreur
                log.error("Migration impossible de la base", eee);
                // rollback du travail en cours
                conn.rollback();
            } finally {
                // close database connexion
                conn.close();
            }
        } catch (Exception eee) {
            log.error("Error lors de la tentative de migration", eee);
        }
        
        return result;
    }
    
    protected void migrateTo3_2(Connection conn) throws SQLException {
        if (log.isInfoEnabled()) {
            log.info("Migrate to version 3.2");
        }

        //alteration du schema
        // nouvelles equations dans les tables

        String[] sqls = new String[] {
                "alter table SETOFVESSELS add column TECHNICALEFFICIENCYEQUATION VARCHAR(255);",
                "alter table STRATEGY add column INACTIVITYEQUATIONUSED BIT default false;",
                "alter table STRATEGY add column INACTIVITYEQUATION VARCHAR(255);",
                "alter table STRATEGYMONTHINFO alter NUMBEROFTRIPS double;",
                "alter table STRATEGYMONTHINFO alter MININACTIVITYDAYS double;" };
        for (String sql : sqls) {
            log.info("try " + sql);
            PreparedStatement sta = conn.prepareStatement(sql);
            sta.executeUpdate();
            log.info("done " + sql);
        }
    }

    protected void migrateTo3_3(Connection conn) throws SQLException {
        if (log.isInfoEnabled()) {
            log.info("Migrate to version 3.3");
        }

        String[] sqls = new String[] {
                // replace in equation
                "update EQUATION set content = replace(content, 'org.codelutin.', 'org.nuiton.');",
                // replace in matrix semantics
                "update POPULATION set recruitmentDistribution_semantics = replace(recruitmentDistribution_semantics, 'org.codelutin.', 'org.nuiton.') where recruitmentDistribution_semantics IS NOT null;",
                "update POPULATION set mappingZoneReproZoneRecru_semantics = replace(mappingZoneReproZoneRecru_semantics, 'org.codelutin.', 'org.nuiton.') where mappingZoneReproZoneRecru_semantics IS NOT null;",
                "update POPULATION set capturability_semantics = replace(capturability_semantics, 'org.codelutin.', 'org.nuiton.') where capturability_semantics IS NOT null;",
                "update POPULATIONSEASONINFO set reproductionDistribution_semantics = replace(reproductionDistribution_semantics, 'org.codelutin.', 'org.nuiton.') where reproductionDistribution_semantics IS NOT null;",
                "update POPULATIONSEASONINFO set lengthChangeMatrix_semantics = replace(lengthChangeMatrix_semantics, 'org.codelutin.', 'org.nuiton.') where lengthChangeMatrix_semantics IS NOT null;",
                "update POPULATIONSEASONINFO set migrationMatrix_semantics = replace(migrationMatrix_semantics, 'org.codelutin.', 'org.nuiton.') where migrationMatrix_semantics IS NOT null;",
                "update POPULATIONSEASONINFO set emigrationMatrix_semantics = replace(emigrationMatrix_semantics, 'org.codelutin.', 'org.nuiton.') where emigrationMatrix_semantics IS NOT null;",
                "update POPULATIONSEASONINFO set immigrationMatrix_semantics = replace(immigrationMatrix_semantics, 'org.codelutin.', 'org.nuiton.') where immigrationMatrix_semantics IS NOT null;",
                "update RESULT set matrix_semantics = replace(matrix_semantics, 'org.codelutin.', 'org.nuiton.') where matrix_semantics IS NOT null;",
                "update STRATEGYMONTHINFO set proportionMetier_semantics = replace(proportionMetier_semantics, 'org.codelutin.', 'org.nuiton.') where proportionMetier_semantics IS NOT null;",
                // add .shp default extension
                "update FISHERYREGION set MAPFILES = regexp_replace(MAPFILES, '(,|$)', '.shp$1') where MAPFILES IS NOT null AND LENGTH(TRIM(MAPFILES)) > 0;"
        };
        for (String sql : sqls) {
            log.info("try " + sql);
            PreparedStatement sta = conn.prepareStatement(sql);
            sta.executeUpdate();
            log.info("done " + sql);
        }
    }

}
