/* *##%
 * Copyright (C) 2005 - 2010
 *     Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package fr.ifremer.isisfish.datastore;

import java.io.File;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.map.ReferenceMap;

import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.IsisFishException;
import fr.ifremer.isisfish.simulator.Simulator;
import fr.ifremer.isisfish.vcs.VCSException;

/**
 * Gestion des fichers CVS de type {@link Simulator} (appartenant au module Simulator).
 * 
 * Created: 18 août 2005 15:07:36 CEST
 *
 * @author Grégoire DESSARD <dessard@codelutin.com>
 * @version $Revision: 2938 $
 *
 * Last update: $Date: 2010-01-22 16:42:09 +0100 (ven., 22 janv. 2010) $
 * by : $Author: chatellier $
 */
public class SimulatorStorage extends JavaSourceStorage {
	
    public static final String SIMULATOR_PATH = "simulators";
    
    /** Template freemarker pour les scripts. */
    public static final String SIMULATOR_TEMPLATE = "templates/script/simulator.ftl";

    /** Simulators cache. */
    static private Map<String, SimulatorStorage> simulatorsCache = (Map<String, SimulatorStorage>) new ReferenceMap();

    /**
     * Constructeur.
     *
     * @param rootSrc
     */
    protected SimulatorStorage(File rootSrc, File directory, String name) {
        super(rootSrc, directory, name);
    }
    
    public static File getSimulatorDirectory() {
        File result = new File(getContextDatabaseDirectory(), SIMULATOR_PATH);
        result.mkdirs();
        return result;
    }
    

    /**
     * Retourne une nouvelle instance du simulateur. Compile le fichier si besoin.
     * 
     * @return new simulator instance
     * @throws IsisFishException 
     */
    public Simulator getNewSimulatorInstance() throws IsisFishException {
        Simulator result = (Simulator)getNewInstance();
        return result;
    }
    
    /**
     * Retourne le storage pour le simulateur demandée.
     * 
     * {@link SimulatorStorage} is cached by name.
     * 
     * @param name le nom de la regle souhaitée
     * @return Le storage pour la regle
     */
    static public SimulatorStorage getSimulator(String name) {
        String cacheName = getContextDatabaseCacheKey(name);
        SimulatorStorage result = simulatorsCache.get(cacheName);
        if (result == null) {
            result = new SimulatorStorage(getContextDatabaseDirectory(),
                    getSimulatorDirectory(), name);
            simulatorsCache.put(cacheName, result);
        }
        return result;
    }

    static public void checkout() throws VCSException {
        checkout(IsisFish.config.getDatabaseDirectory(), SIMULATOR_PATH);
    } 
    
    /**
     * Retourne la liste des noms de toutes les régions disponible en local
     * 
     * @return la liste des noms de toutes les régions disponible en local
     */
    static public List<String> getSimulatorNames() {
        File dir = getSimulatorDirectory();
        List<String> result = getStorageNames(dir);
        return result;
    }

    /**
     * Retourne la liste des noms de toutes les régions disponible en local qui
     * ne sont pas encore sur le serveur CVS
     * 
     * @return liste de noms de regions
     */
    static public List<String> getNewSimulatorNames() {
        List<String> result = getSimulatorNames();
        result.removeAll(getRemoteSimulatorNames());
        return result;
    }

    /**
     * Retourne la liste des noms de toutes les régions disponible sur le
     * serveur CVS
     * 
     * @return la liste des noms de toutes les régions disponible sur le serveur
     *         CVS. Si le serveur n'est pas disponible la liste retournée est
     *         vide.
     */
    static public List<String> getRemoteSimulatorNames() {
        File dir = getSimulatorDirectory();
        List<String> result = getRemoteStorageNames(dir);
        return result;

    }

    /**
     * Retourne la liste des noms de toutes les régions disponible sur le
     * serveur CVS qui ne sont pas encore en local
     * 
     * @return liste de noms de regions
     */
    static public List<String> getNewRemoteSimulatorNames() {
        List<String> result = getRemoteSimulatorNames();
        result.removeAll(getSimulatorNames());
        return result;
    }
}
