/* *##% Copyright (C) 2009-2010 Ifremer, Code Lutin
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA. ##%
 */

package fr.ifremer.isisfish.datastore;

import fr.ifremer.isisfish.IsisFishException;
import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.simulator.sensitivity.SensitivityCalculator;
import fr.ifremer.isisfish.util.Doc;
import fr.ifremer.isisfish.util.DocHelper;
import fr.ifremer.isisfish.util.Docable;
import fr.ifremer.isisfish.vcs.VCSException;

import static org.nuiton.i18n.I18n._;

import org.apache.commons.collections.map.ReferenceMap;
import org.apache.commons.lang.ObjectUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.File;
import java.lang.reflect.Field;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * Cette class permet de stocker les fichiers de calculateur de sensibilité.
 * <p/>
 * Gere les fichiers VCS de type {@link SensitivityCalculator} (package sensitivity).
 * 
 * Created: 17 août 2005 11:11:51 CEST
 *
 * @author chatellier <chatellier@codelutin.com>
 * @version $Revision: 2938 $
 * Last update: $Date: 2010-01-22 16:42:09 +0100 (ven., 22 janv. 2010) $ by : $Author: chatellier $
 */
public class SensitivityStorage extends JavaSourceStorage implements Docable { // SensitivityStorage

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static Log log = LogFactory.getLog(SensitivityStorage.class);

    /** Emplacement de stockage des fichiers de sensibilité */
    public final static String SENSITIVITY_PATH = "sensitivity";

    /** Template freemarker pour les scripts de sensibilité. */
    public static final String SENSIVITY_TEMPLATE = "templates/script/sensitivity.ftl";
    
    /**
     * Prefix des paramètres de la classe.
     * 
     * ex : public int param_count;
     */
    protected final static String PARAM_PREFIX = "param_";

    /** Cache. */
    protected static Map<String, SensitivityStorage> sensitivityCache = new ReferenceMap();

    /**
     * Build new {@link SensitivityStorage}.
     *
     * @param rootSrc   repertoire root de stockage des calculateurs de sensibilité.
     * @param directory le repertoire ou devrait se trouver le calculateur de sensibilité
     * @param name      le nom du calculateur de sensibilité
     */
    protected SensitivityStorage(File rootSrc, File directory, String name) {
        super(rootSrc, directory, name);
    }

    /**
     * Get sensitivity storage directory.
     * 
     * Create directory if not exists.
     * 
     * @return sensitivity storage directory
     */
    public static File getSensitivityDirectory() {
        File result = new File(getContextDatabaseDirectory(), SENSITIVITY_PATH);
        result.mkdirs();
        return result;
    }

    /**
     * Retourne le nom de tous les calculateurs de sensibilité existant.
     *
     * @return le nom de tous les calculateurs de sensibilité existans en local
     */
    public static List<String> getSensitivityNames() {
        File dir = getSensitivityDirectory();
        return getStorageNames(dir);
    }

    /**
     * Retourne le storage pour le calculateur demandé.
     *
     * @param name le nom du calculateur souhaité
     * @return Le {@link SensitivityStorage} pour le calculateur
     */
    public static SensitivityStorage getSensitivity(String name) {
        String cacheName = getContextDatabaseCacheKey(name);
        SensitivityStorage result = sensitivityCache.get(cacheName);
        if (result == null) {
            result = new SensitivityStorage(getContextDatabaseDirectory(), getSensitivityDirectory(), name);
            sensitivityCache.put(cacheName, result);
        }
        return result;
    }

    /**
     * Retourne une nouvelle instance du calculateur. Compile le fichier si besoin
     *
     * @return retourne une nouvelle instance du calculateur
     * @throws IsisFishException s'il y a un problème d'instanciation
     */
    public SensitivityCalculator getNewSensitivityInstance() throws IsisFishException {
        Object result = getNewInstance();
        return (SensitivityCalculator) result;
    }

    /**
     * Permet de recuperer les parametres du calculateur.
     *
     * @return Une map contenant le nom du parametre et son type
     * @throws IsisFishException s'il est impossible de retourner une map
     *                           convenable
     */
    public Map<String, Class<?>> getParameterNames() throws IsisFishException {
        Map<String, Class<?>> result = null;
        // On essai de recuperer les paramètres depuis la classe compilé
        // si c possible        
        try {
            if (0 == compile(false, null)) {
                SensitivityCalculator calculator = getNewSensitivityInstance();
                result = SensitivityStorage.getParameterNames(calculator);
            }
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error(_("isisfish.error.source.parameter"), eee);
            }
        }
        // si on ne reussi pas avec la classe compilé on essai en parsant le
        // source
        if (result == null) {
            try {
                //String code = getContent();
                log.fatal("FIXME a faire recherche des parametres dans le source");
                throw new IsisFishException(
                        _("isisfish.error.source.parameter"));
                // TODO implanter la recherche des parametres.
                // Se sont les attributs commencant par 'param_' et ils doivent 
                // etre public non static, non transient.
                // ex: public Date param_Date.
                // Lorsqu'on ajoute un champs dans la map il ne faut pas mettre
                // le prefix param.
            } catch (Exception eee) {
                throw new IsisFishException(
                        _("isisfish.error.source.parameter"), eee);
            }
        }
        return result;
    }

    /**
     * Recherche par introspection tous les parametres de la classe
     * commencant par {@link #PARAM_PREFIX}.
     *
     * @param calculator le calculateur dont on souhaite les infos de parametre
     * @return retourne le nom et le type des parametres du calculateur
     */
    public static Map<String, Class<?>> getParameterNames(
            SensitivityCalculator calculator) {
        Map<String, Class<?>> result = new LinkedHashMap<String, Class<?>>();
        for (Field field : calculator.getClass().getFields()) {
            if (field.getName().startsWith(PARAM_PREFIX)) {
                result.put(field.getName().substring(PARAM_PREFIX.length()),
                        field.getType());
            }
        }
        return result;
    }

    /**
     * Donne la valeur d'un parametre par introspection.
     *
     * @param name le nom du parametre
     * @param calculator le calculateur dont on souhaite la valeur du parametre
     * @return la valeur courante du parametre
     * @throws IsisFishException s'il y a un probleme pour recuperer la valeur
     */
    public static Object getParameterValue(SensitivityCalculator calculator,
            String name) throws IsisFishException {
        if (calculator == null || name == null || "".equals(name)) {
            return null;
        }
        try {
            String fieldName = PARAM_PREFIX + name;
            Field field = calculator.getClass().getDeclaredField(fieldName);
            return field.get(calculator);
        } catch (IllegalAccessException eee) {
            throw new IsisFishException("Can't get plan parameter: " + name,
                    eee);
        } catch (NoSuchFieldException eee) {
            throw new IsisFishException("Can't get plan parameter: " + name,
                    eee);
        }
    }

    /**
     * Modifie la valeur d'un attribut pas introspection.
     *
     * @param name  le nom de l'attribut
     * @param value la valeur de l'attribut
     * @param calculator le calculator dont on souhaite modifier la valeur de parametre
     * @throws IsisFishException s'il y a un probleme
     */
    public static void setParameterValue(SensitivityCalculator calculator,
            String name, Object value) throws IsisFishException {
        try {
            String fieldName = PARAM_PREFIX + name;
            Field field = calculator.getClass().getDeclaredField(fieldName);
            field.set(calculator, value);
        } catch (IllegalAccessException eee) {
            throw new IsisFishException("Can't modify plan parameter: " + name
                    + " with '" + value + "'("
                    + ObjectUtils.identityToString(value) + ")", eee);
        } catch (NoSuchFieldException eee) {
            throw new IsisFishException("Can't modify plan parameter: " + name
                    + " with '" + value + "'("
                    + ObjectUtils.identityToString(value) + ")", eee);
        } catch (IllegalArgumentException eee) {
            throw new IsisFishException("Can't modify plan parameter: " + name
                    + " with '" + value + "'("
                    + ObjectUtils.identityToString(value) + ")", eee);
        }
    }

    /**
     * Recupere les paramètres et leur valeur pour les retourner sous forme
     * de chaine. Pour pouvoir par exemple les afficher à l'utilisateur.
     *
     * @param calculator le calculateur dont on souhaite la valeur du parametre en string
     * @return la valeur sous forme de string
     * @throws IsisFishException
     */
    public static String getParamAsString(SensitivityCalculator calculator)
            throws IsisFishException {
        StringBuffer result = new StringBuffer();

        for (String name : getParameterNames(calculator).keySet()) {
            Object value = getParameterValue(calculator, name);

            result.append(name).append(" : ").append(value);
            result.append("\n");
        }
        return result.toString();
    }

    /**
     * Effectue un chekout VCS sur le répertoire des calculateurs.
     * 
     * @see VersionStorage#checkout(File, String)
     * @see #SENSITIVITY_PATH
     * 
     * @throws VCSException if an error occurs during checkout
     */
    public static void checkout() throws VCSException {
        checkout(IsisFish.config.getDatabaseDirectory(), SENSITIVITY_PATH);
    }

    /**
     * Retourne la liste des noms de tous les calculateurs disponibles en local qui
     * ne sont pas encore sur le serveur VCS.
     *
     * @return liste de noms de calculateurs
     */
    static public List<String> getNewSensitivityNames() {
        List<String> result = getSensitivityNames();
        result.removeAll(getRemoteSensitivityNames());
        return result;
    }

    /**
     * Retourne la liste des noms de tous les calculateurs disponibles sur le
     * serveur VCS
     *
     * @return la liste des noms de tous les calculateurs disponibles sur le serveur
     *         VCS. Si le serveur n'est pas disponible la liste retournée est
     *         vide.
     */
    static public List<String> getRemoteSensitivityNames() {
        File dir = getSensitivityDirectory();
        return getRemoteStorageNames(dir);

    }

    /**
     * Retourne la liste des noms de tous les calculateurs disponibles sur le
     * serveur VCS qui ne sont pas encore en local.
     *
     * @return liste de noms de regions
     * @throws VCSException
     */
    static public List<String> getNewRemoteSensitivityNames()
            throws VCSException {
        List<String> result = getRemoteSensitivityNames();
        result.removeAll(getSensitivityNames());
        return result;
    }

    /**
     * @return the @Doc of the underlied {@link SensitivityCalculator} class
     * @see DocHelper
     * @see Doc
     * @see Docable
     */
    public Doc getClassDoc() {
        Doc result = null;
        try {
            Class<?> klazz = getCodeClass();
            result = DocHelper.getClassDoc(klazz);
        } catch (IsisFishException e) {
            if (log.isWarnEnabled()) {
                log.warn(_("isisfish.error.not.found.code", this));
            }
        }
        return result;
    }

    /**
     * @param fieldName name of the field to inspect
     * @return the @Doc of the given field of the underlied {@link SensitivityCalculator} class
     * @see DocHelper
     * @see Doc
     * @see Docable
     */
    public Doc getFieldDoc(String fieldName) {
        Doc result = null;
        try {
            Class<?> klazz = getCodeClass();
            result = DocHelper.getFieldDoc(klazz, fieldName);
        } catch (Exception e) {
            log.warn(_("isisfish.error.not.found.field", fieldName, this));
        }
        return result;
    }

    /**
     * @param paramName the name of the param to inspect
     * @return the doc associated with the param,
     *         (says the field {@link #PARAM_PREFIX} + fieldName)
     */
    public Doc getParamDoc(String paramName) {
        Doc doc;
        doc = getFieldDoc(PARAM_PREFIX + paramName);
        return doc;
    }

    /**
     * <b>Be ware this method require to instanciate a AnalysePlan, so
     * it would be better to call as often as possible.</b>
     *
     * @return the descript of the instanciate AnalysePlan
     * @see Doc
     * @see Docable
     */
    public String getDescription() {
        String result = null;
        try {
            SensitivityCalculator calculator = getNewSensitivityInstance();
            result = calculator == null ? null : calculator.getDescription();
        } catch (Exception e) {
            log.warn(_("isisfish.error.not.found.description", this));
        }
        return result;
    }

} // SensitivityStorage
