/* *##%
 * Copyright (C) 2002-2010 
 *    Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package fr.ifremer.isisfish.datastore;

import static org.nuiton.i18n.I18n._;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.collections.map.ReferenceMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.math.matrix.MatrixFactory;
import org.nuiton.math.matrix.MatrixIterator;
import org.nuiton.math.matrix.MatrixND;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.util.ArrayUtil;
import org.nuiton.util.HashList;

import fr.ifremer.isisfish.IsisFishDAOHelper;
import fr.ifremer.isisfish.IsisFishException;
import fr.ifremer.isisfish.entities.ActiveRule;
import fr.ifremer.isisfish.entities.ActiveRuleDAO;
import fr.ifremer.isisfish.entities.Population;
import fr.ifremer.isisfish.entities.Result;
import fr.ifremer.isisfish.entities.ResultDAO;
import fr.ifremer.isisfish.export.Export;
import fr.ifremer.isisfish.export.SensitivityExport;
import fr.ifremer.isisfish.rule.Rule;
import fr.ifremer.isisfish.simulator.AnalysePlan;
import fr.ifremer.isisfish.simulator.SimulationContext;
import fr.ifremer.isisfish.simulator.SimulationException;
import fr.ifremer.isisfish.simulator.SimulationResultGetter;
import fr.ifremer.isisfish.simulator.SimulationResultListener;
import fr.ifremer.isisfish.types.Date;
import fr.ifremer.isisfish.types.Month;

/**
 * Cette classe permet de conserver des résultats de simulation. Elle permet
 * ensuite de les récupérer.
 * 
 * Created: 29 sept. 2004
 *
 * @author Benjamin Poussin <poussin@codelutin.com>
 * @version $Revision: 2919 $
 *
 * Mise a jour: $Date: 2010-01-15 16:45:05 +0100 (ven., 15 janv. 2010) $
 * par : $Author: chatellier $
 */
public class ResultStorage implements SimulationResultListener, SimulationResultGetter{ // ResultStorage

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(ResultStorage.class);

    protected SimulationStorage simulation = null;
//    transient protected HashMap<String, MatrixND> globalMatrix = new HashMap<String, MatrixND>();
    transient protected ReferenceMap cacheContext = new ReferenceMap(ReferenceMap.HARD, ReferenceMap.WEAK);

    /** cache to maintains some result. key: String(date + ':' + name), value: matrix
     * TODO: cache will be more efficient if it keep at min the number of result by year */
    transient protected ReferenceMap cache = new ReferenceMap(ReferenceMap.HARD, ReferenceMap.SOFT);
    /** contains all available result as string: String(date + ':' + name) */
    transient protected Set<String> availableResult = null;
    /** result enabled */
    transient protected Set<String> enabledResult = null;

    /**
    * Les ResultStorage ne doivent pas etre instancier directement, mais
    * recuperer a partir d'un
    * {@link fr.ifremer.isisfish.datastore.SimulationStorage#getResultStorage()}
    */
    public ResultStorage(SimulationStorage simulation) {
        this.simulation = simulation;
    }

    protected void putInCache(Date date, String name, MatrixND mat, TopiaContext context) {
        String key = date + ":" + name;
        putInCache(key, mat, context);
    }
    protected void putInCache(String name, MatrixND mat, TopiaContext context) {
        if (mat != null) {
            cache.put(name, mat);
            cacheContext.put(name, context);
        }
    }

    protected MatrixND getInCache(Date date, String name) {
        String key = date + ":" + name;
        MatrixND result = getInCache(key);
        return result;
    }
    protected MatrixND getInCache(String name) {
        MatrixND result = null;
        TopiaContext context = (TopiaContext)cacheContext.get(name);
        if (context != null && !context.isClosed()) {
            // on verifie que le context existe encore car on peut vouloir
            // naviguer dans les semantics
            result = (MatrixND)cache.get(name);
        }
        return result;
    }

    /**
     * Retourne le nom de tous les resultats disponibles le nom est constitué
     * de la date et du nom du resultat.
     * 
     * @return available results
     */
    protected Set<String> getAvailableResult() {
        if (availableResult == null) {
            availableResult = new HashSet<String>();
            try {
                TopiaContext tx = null;
                boolean mustClose = false;

                if (simulation == SimulationContext.get().getSimulationStorage()) {
                    tx = SimulationContext.get().getDbResult();
                }

                if (tx == null) {
                    // not in simulation, create transaction
                    tx = simulation.getStorage().beginTransaction();
                    mustClose = true;
                }
                @SuppressWarnings("unchecked")
                List<String> result = (List<String>)tx.find(
                        "Select resultDate||':'||name from fr.ifremer.isisfish.entities.Result");
                if (mustClose) {
                    tx.closeContext();
                }
                availableResult.addAll(result);
            } catch (Exception eee) {
                if (log.isWarnEnabled()) {
                    log.warn("Can't get result available", eee);
                }
            }

        }
        return availableResult;
    }

    /**
     * Verifie si un resultat est disponible pour une date donnée
     * @param date
     * @param name
     * @return {@code true} if result is available
     */
    protected boolean isAvailableResult(Date date, String name) {
        String key = date.getDate() + ":" + name;
        boolean result = getAvailableResult().contains(key);
        return result;
    }

    /**
     * Ajoute un resultat comme etant disponible pour une date donnée.
     * 
     * @param date
     * @param name
     */
    protected void addAvailableResult(Date date, String name) {
        String key = date.getDate() + ":" + name;
        getAvailableResult().add(key);
    }

    /**
     * Permet de savoir si lorsque l'on ajoutera ce resultat, il sera
     * sauvé ou non.
     * 
     * Check for result name returned by :
     * <ul>
     *  <li>{@link Export#getNecessaryResult()}</li>
     *  <li>{@link SensitivityExport#getNecessaryResult()}</li>
     *  <li>{@link Rule#getNecessaryResult()}</li>
     *  <li>{@link AnalysePlan#getNecessaryResult()}</li>
     * </ul>
     * 
     * @param name result name
     * @return {@code true} if result is enabled
     */
    public boolean isEnabled(String name){
        name = name.trim();
        if (enabledResult == null) {
            enabledResult = new HashSet<String>();

            Collection<String> resultEnabled = simulation.getParameter().getResultEnabled();
            enabledResult.addAll(resultEnabled);

            // test on export
            List<String> exportNames = simulation.getParameter().getExportNames();
            if (exportNames != null) {
                for (String exportName : exportNames) {
                    ExportStorage storage = ExportStorage.getExport(exportName);
                    try {
                        Export export = storage.getNewExportInstance();
                        for (String resultName : export.getNecessaryResult()) {
                            enabledResult.add(resultName);
                        }
                    } catch (IsisFishException eee) {
                        if (log.isWarnEnabled()) {
                            log.warn(_("isisfish.error.instanciate.export", exportName), eee);
                        }
                    }
                }
            }

            // test on sensitivity export
            List<SensitivityExport> sensitivityExports = simulation.getParameter().getSensitivityExport();
            if (sensitivityExports != null) {
                for (SensitivityExport sensitivityExport : sensitivityExports) {
                    for (String resultName : sensitivityExport.getNecessaryResult()) {
                        enabledResult.add(resultName);
                    }
                }
            }

            // test on rules
            List<Rule> rules = simulation.getParameter().getRules();
            if (rules != null) {
                for (Rule rule : rules) {
                    for (String resultName : rule.getNecessaryResult()) {
                        enabledResult.add(resultName);
                    }
                }
            }

            // test on plans
            List<AnalysePlan> plans = simulation.getParameter().getAnalysePlans();
            if (plans != null) {
                for (AnalysePlan plan : plans) {
                    for (String resultName : plan.getNecessaryResult()) {
                        enabledResult.add(resultName);
                    }
                }
            }
            log.info("Enabled result: " + enabledResult);
        }
        boolean result = enabledResult.contains(name);
        return result;
    }

    public void addResult(Date date, MatrixND mat) throws IsisFishException{
        addResult(false, date, mat.getName(), mat);
    }

    public void addResult(Date date, Population pop, MatrixND mat) throws IsisFishException{
        addResult(false, date, mat.getName(), pop, mat);
    }

    public void addResult(boolean force, Date date, MatrixND mat) throws IsisFishException{
        addResult(force, date, mat.getName(), mat);
    }

    public void addResult(boolean force, Date date, Population pop, MatrixND mat) throws IsisFishException{
        addResult(force, date, mat.getName(), pop, mat);
    }

    public void addResult(Date date, String name, Population pop, MatrixND mat) throws IsisFishException{
        addResult(false, date, name, pop, mat);
    }

    public void addResult(Date date, String name, MatrixND mat) throws IsisFishException{
        addResult(false, date, name, mat);
    }

    public void addResult(boolean force, Date date, String name, Population pop, MatrixND mat) throws IsisFishException{
        if (force || isEnabled(name)) {
            doAddResult(date, name + " " + pop, mat);
        }
    }

    public void addResult(boolean force, Date date, String name, MatrixND mat) throws IsisFishException{
        if (force || isEnabled(name)) {
            doAddResult(date, name, mat);
        }
    }

    protected void doAddResult(Date date, String name, MatrixND mat) throws IsisFishException{
        try {
            TopiaContext tx = null;
            boolean mustClose = false;

            if (simulation == SimulationContext.get().getSimulationStorage()) {
                tx = SimulationContext.get().getDbResult();
            }
            if (tx == null) {
                // not in simulation, create transaction
                tx = simulation.getStorage().beginTransaction();
                mustClose = true;
            }
            doAddResult(date, name, mat, tx);
            if (mustClose) {
                tx.commitTransaction();
                tx.closeContext();
            }
        } catch (TopiaException eee) {
            log.warn("Can't add result '" + name + "' at date " + date , eee);
        }
    }

    protected void doAddResult(Date date, String name, MatrixND mat, TopiaContext tx) throws IsisFishException{
        // si la matrice n'a pas de semantique on refuse
        for (int i=0; i<mat.getNbDim(); i++) {
            // la semantique n'est pas bonne des qu'il y a un null dedans
            if (mat.getSemantics(i).contains(null)) {
                throw new SimulationException("Erreur le résultat que vous souhaitez enregistrer n'a pas d'information convenable pour la dimension: " + i + " " + mat.getDimensionName(i) );
            }
        }

        // on fait une copie pour avoir reellement des resultats independant
        MatrixND newMat = mat.copy();
        try {
            ResultDAO resultPS = IsisFishDAOHelper.getResultDAO(tx);
            Result result = resultPS.create();
            result.setResultDate(date);
            result.setName(name);
            result.setMatrix(newMat);
            resultPS.update(result);

            addAvailableResult(date, name);
            putInCache(date, name, newMat, tx);
        } catch (TopiaException eee) {
            log.warn("Can't add result '" + name + "' at date " + date , eee);
        }
    }

    public void addActiveRule(Date date, Rule rule) throws IsisFishException {
        try {
            TopiaContext tx = null;
            boolean mustClose = false;

            if (simulation == SimulationContext.get().getSimulationStorage()) {
                tx = SimulationContext.get().getDbResult();
            }
            if (tx == null) {
                // not in simulation, create transaction
                tx = simulation.getStorage().beginTransaction();
                mustClose = true;
            }
            ActiveRuleDAO ps = IsisFishDAOHelper.getActiveRuleDAO(tx);
            ActiveRule result = ps.create();
            result.setActiveRuleDate(date);
            result.setName(RuleStorage.getName(rule));
            result.setParam(RuleStorage.getParamAsString(rule));
            ps.update(result);
            if (mustClose) {
                tx.commitTransaction();
                tx.closeContext();
            }
        } catch (TopiaException eee) {
            throw new IsisFishException("Can't add result", eee);
        }
    }

    /**
     * Retourne la liste de tous les résultats. Si le résultat est categorisé
     * par une population alors le nom de la population est automatiquement
     * ajouté au nom du résultat.
     */
    public List<String> getResultName() {

        List<String> result = null;
        try {
            TopiaContext tx = null;
            boolean mustClose = false;

            if (simulation == SimulationContext.get().getSimulationStorage()) {
                tx = SimulationContext.get().getDbResult();
            }
            if (tx == null) {
                // not in simulation, create transaction
                tx = simulation.getStorage().beginTransaction();
                mustClose = true;
            }
            ResultDAO resultPS = IsisFishDAOHelper.getResultDAO(tx);


            result = (List<String>)resultPS.getContext().find(
                    "Select distinct name from fr.ifremer.isisfish.entities.Result order by name");
            if (mustClose) {
                tx.closeContext();
            }
        } catch (TopiaException eee) {
            if (log.isWarnEnabled()) {
                log.warn("Can't get result name", eee);
            }
        }
        if (result == null) {
            result = new ArrayList<String>();
        }
        return result;
    }

    /**
     * Retourne la matrice stocke pour un pas de temps
     * @param date le pas de temps que l'on souhaite
     * @param pop la population pour lequelle on souhaite le resultat
     * @param name le nom des resultats dont on veut la matrice
     * @return La matrice demandée ou null si aucune matrice ne correspond a
     * la demande.
     */
    public MatrixND getMatrix(Date date, Population pop, String name) {
        String newName = name + " " + pop;
        return getMatrix(date, newName);
    }

    public MatrixND getMatrix(Date date, String name) {
        MatrixND mat = getInCache(date, name);
        if (mat == null && isAvailableResult(date, name)) {
            try {
                TopiaContext tx = null;
                boolean mustClose = false;

                if (simulation == SimulationContext.get().getSimulationStorage()) {
                    tx = SimulationContext.get().getDbResult();
                }
                if (tx == null) {
                    // not in simulation, create transaction
                    tx = simulation.getStorage().beginTransaction();
                    mustClose = true;
                }
                mat = getMatrix(date, name, tx);
                if (mustClose) {
                    // FIXME transaction never closed
                    // quand peut on fermer la transaction ?
                    // lorsque plus aucune matrice ne l'utilise.
                    // donc mettre la matrice et la connexion dans une map
                    // la matrice dans une weak reference. Des que la matrice
                    // est liberer faire un close sur la transaction
                }
            } catch (Exception eee) {
                if (log.isWarnEnabled()) {
                    log.warn("Can't return matrix '" + name + "' for date " + date, eee);
                }
            }
        }
        return mat;
    }

    /**
     * Retourne la matrice stocke pour un pas de temps.
     * 
     * @param date le pas de temps que l'on souhaite
     * @param name le nom des resultats dont on veut la matrice
     * @param tx TopiaContext a utiliser pour recuperer les resultats et donc les semantiques
     * @return La matrice demandée ou {@code null} si aucune matrice ne correspond a
     * la demande.
     */
     public MatrixND getMatrix(Date date, String name, TopiaContext tx) {
        MatrixND mat = getInCache(date, name);
        if (mat == null && isAvailableResult(date, name)) {
            try {
                ResultDAO resultPS = IsisFishDAOHelper.getResultDAO(tx);
                Result result = resultPS.findByProperties("resultDate", date, "name", name);
                if (result != null) {
                    mat = result.getMatrix();
                    putInCache(date, name, mat, tx);
                }
            } catch (Exception eee) {
                if (log.isWarnEnabled()) {
                    log.warn("Can't return matrix '" + name + "' for date " + date, eee);
                }
            }
        }
        return mat;
    }

    /**
     * Retourne une matrice contenant tous les pas de temps.
     * @param pop la population pour lequel on souhaite la matrice
     * @param name le nom des resultats dont on veut une matrice globale.
     */
    public MatrixND getMatrix(Population pop, String name) {
        String newName = name + " " + pop;
        return getMatrix(newName);
    }

    /**
     * Retourne une matrice contenant tous les pas de temps.
     * @param name le nom des resultats dont on veut une matrice globale.
     */
    public MatrixND getMatrix(String name) {
        MatrixND resultMat = null;
        try {
            TopiaContext tx = null;
            boolean mustClose = false;

            if (simulation == SimulationContext.get().getSimulationStorage()) {
                tx = SimulationContext.get().getDbResult();
            }
            if (tx == null) {
                // not in simulation, create transaction
                tx = simulation.getStorage().beginTransaction();
                mustClose = true;
            }
            resultMat = getMatrix(name, tx);
            if (mustClose) {
                // FIXME transaction never closed
                // quand peut on fermer la transaction ?
                // lorsque plus aucune matrice ne l'utilise.
                // donc mettre la matrice et la connexion dans une map
                // la matrice dans une weak reference. Des que la matrice
                // est liberer faire un close sur la transaction
            }
        } catch (TopiaException eee) {
            if (log.isWarnEnabled()) {
                log.warn("Can't get result: " + name, eee);
            }
        }
        return resultMat;
    }

    /**
     * Retourne une matrice contenant tous les pas de temps.
     * @param name le nom des resultats dont on veut une matrice globale.
     * @param tx TopiaContext a utiliser pour recuperer les resultats et donc les semantiques
     */
    public MatrixND getMatrix(String name, TopiaContext tx) {
        log.debug("Get result: " + name);

        MatrixND resultMat = getInCache(name);
        if(resultMat != null){
            return resultMat;
        }

        // recuperation des resultats qui nous interesse
        List<Result> results = null;
        try {
            ResultDAO resultPS = IsisFishDAOHelper.getResultDAO(tx);
            results = resultPS.findAllByName(name);
        } catch (TopiaException eee) {
            if (log.isWarnEnabled()) {
                log.warn("Can't get result: " + name, eee);
            }
        }

        // s'il n'y pas de resultat, on retourne null
        if (results == null || results.size() == 0) {
            return null;
        }

        // Creation des listes pour chaque dimension

        // creation de la liste de date
        Date lastDate = getLastDate();
        List<Date> dates = new ArrayList<Date>();
        Date date = new Date(0);
        dates.add(date);
        while (date.before(lastDate)) {
            date = date.next();
            dates.add(date);
        }

        if (log.isTraceEnabled()) {
            log.trace("List des dates: "+ dates);
        }

        // recuperation des dimensions des matrices
        MatrixND mat = (MatrixND)results.get(0).getMatrix();

        // recuperation des noms des dimensions
        String [] dimNames = new String[1 + mat.getNbDim()];
        dimNames[0] = _("isisfish.common.date");
        for (int i=1; i<dimNames.length; i++) {
            dimNames[i] = mat.getDimensionName(i-1);
        }


        // creation de la semantique pour la matrice resultat. +1 pour les dates
        List [] sem = new List[1 + mat.getNbDim()];
        sem[0] = dates;

        for (int i=1; i<sem.length; i++) {
            sem[i] = new HashList();
        }

        for (Result result: results) {
            MatrixND mattmp = result.getMatrix();
            if (log.isTraceEnabled()) {
                log.trace("Ajout de la semantics: "+ Arrays.asList(mattmp.getSemantics()));
            }

            for (int s=0; s<mattmp.getNbDim(); s++) {
                sem[s+1].addAll(mattmp.getSemantics(s));
            }
        }

        if (log.isTraceEnabled()) {
            log.trace("La semantique final est: "+ Arrays.asList(sem));
        }

        // creation de la matrice resultat
        resultMat = MatrixFactory.getInstance().create(name, sem, dimNames);

        // recuperation du resultat pour chaque date de la simulation, de Date(0) à lastDate
        for(Result result: results){
            Date d = result.getResultDate();
            mat = result.getMatrix();
            // on met ce resultat dans la matrice result si besoin
            if(mat != null){
                // on recupere dans la matrice resultat l'endroit on il faut
                // mettre la matrice
                MatrixND submat = resultMat.getSubMatrix(0, d, 1);
                // on met les valeur de mat dans la sous matrice extraite
                for(MatrixIterator mi=mat.iterator(); mi.next();){
                    submat.setValue(
                        ArrayUtil.concat(new Object[]{d},
                            mi.getSemanticsCoordinates()),
                        mi.getValue());
                }
            }
        }

        putInCache(name, resultMat, tx);
        return resultMat;
    }

    /**
     * Get last simulation date.
     * 
     * @return last simulation date
     */
    public Date getLastDate() {
        int monthNumber = simulation.getParameter().getNumberOfYear() * Month.NUMBER_OF_MONTH;
        Date result = new Date(monthNumber - 1); // -1 because date begin at 0
        return result;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationResultListener#addResult(fr.ifremer.isisfish.simulator.SimulationContext, fr.ifremer.isisfish.types.Date, java.lang.String, org.nuiton.math.matrix.MatrixND)
     */
    @Override
    public void addResult(SimulationContext context, Date date, String name, MatrixND mat) throws IsisFishException {
        try {
            doAddResult(date, name, mat, context.getDbResult());
        } catch (TopiaException eee) {
            log.warn(_("Can't add result '%1$s' at date %2$s", name, date) , eee);
        }
   }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationResultGetter#getMatrix(fr.ifremer.isisfish.simulator.SimulationContext, fr.ifremer.isisfish.types.Date, java.lang.String)
     */
    @Override
    public MatrixND getMatrix(SimulationContext context, Date date, String name) {
        MatrixND result = null;
        try {
            result = getMatrix(date, name, context.getDbResult());
        } catch (TopiaException eee) {
            if (log.isWarnEnabled()) {
                log.warn(_("Can't get result: %1$s", name), eee);
            }
        }
       return result;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationResultGetter#getMatrix(fr.ifremer.isisfish.simulator.SimulationContext, java.lang.String)
     */
    @Override
    public MatrixND getMatrix(SimulationContext context, String name) {
        MatrixND result = null;
        try {
            result = getMatrix(name, context.getDbResult());
        } catch (TopiaException eee) {
            if (log.isWarnEnabled()) {
                log.warn(_("Can't get result: %1$s", name), eee);
            }
        }
       return result;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationListener#afterSimulation(fr.ifremer.isisfish.simulator.SimulationContext)
     */
    @Override
    public void afterSimulation(SimulationContext context) {
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationListener#beforeSimulation(fr.ifremer.isisfish.simulator.SimulationContext)
     */
    @Override
    public void beforeSimulation(SimulationContext context) {
    }

    // public void addActivatedRule(ResultStorage self, Date date, RegleParam rule){
    //     List rules = (List)activatedRules.get(date);
    //     if(rules == null){
    //         activatedRules.put(date, rules = new LinkedList());
    //     }
    //     rules.add(rule);
    // }

    // /**
    // * Retourne pour une date données tous les RegleParam qui ont été activé
    // * a la date demandé.
    // * @return une list de {@link fr.ifremer.nodb.RegleParam}
    // */
    // public List getActivatedRule(ResultStorage self, Date date){
    //     List rules = (List)activatedRules.get(date);
    //     if(rules == null){
    //         activatedRules.put(date, rules = new LinkedList());
    //     }
    //     return rules;
    // }

} // ResultStorage
