/*
 * *##% Copyright (C) 2006 - 2009
 *     Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA. ##%
 */

package fr.ifremer.isisfish.datastore;

import static org.nuiton.i18n.I18n._;

import java.io.File;
import java.io.PrintWriter;

import org.apache.commons.lang.ClassUtils;

import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.IsisFishException;
import fr.ifremer.isisfish.util.CompileHelper;
import fr.ifremer.isisfish.util.Doc;
import fr.ifremer.isisfish.util.DocHelper;
import fr.ifremer.isisfish.util.JavadocHelper;

/**
 * Classes abstraite permettant la gestion des fichiers de code source.
 * 
 * Exemple :
 * {@link fr.ifremer.isisfish.datastore.AnalysePlanStorage}
 * {@link fr.ifremer.isisfish.datastore.ExportStorage},
 * {@link fr.ifremer.isisfish.datastore.RuleStorage},
 * {@link fr.ifremer.isisfish.datastore.ScriptStorage},
 * {@link fr.ifremer.isisfish.datastore.SensitivityStorage},
 * {@link fr.ifremer.isisfish.datastore.SimulatorStorage}
 *
 * @author poussin
 * 
 * @version $Revision: 2701 $
 * 
 * Last update: $Date: 2009-10-27 12:03:36 +0100 (mar., 27 oct. 2009) $ by : $Author: chatellier $
 */
public abstract class JavaSourceStorage extends CodeSourceStorage {

    protected static final String PARAM_PREFIX = "param_";

    /**
     * Build a new java source storage.
     * 
     * @param rootSrc sources root directory
     * @param directory class location directory (rootSrc subdirectory) 
     * @param name class name
     */
    protected JavaSourceStorage(File rootSrc, File directory, String name) {
        super(rootSrc, directory, name, ".java");
    }

    /**
     * Return class fully qualified name.
     * 
     * @return class name including package name
     */
    public String getFQN() {
        String root = getRoot().getAbsolutePath();
        String path = getFile().getParentFile().getAbsolutePath();

        String packageName = path.substring(1 + root.length()).replace(
                File.separatorChar, '.');

        String result = packageName + "." + getName();
        if (result.endsWith(".java")) {
            result = result.substring(0, result.length() - ".java".length());
        }

        return result;
    }

    /**
     * Retourne le nom de la classe (sans le package) a partir d'un objet.
     * 
     * @param instance l'instance dont on veut le nom
     * @return le nom de la classe ou null si instance est null
     */
    public static String getName(Object instance) {
        return ClassUtils.getShortClassName(instance, null);
    }

    /**
     * Compile cette classe dans le répertoire par defaut de compilation
     * ({@link fr.ifremer.isisfish.IsisConfig#getCompileDirectory()})
     * 
     * @param force si vrai alors meme si le fichier destination est plus recent
     *        la compilation aura lieu
     * @param out le flux sur lequel le resultat de la compilation doit
     *        apparaitre. Peut-etre null, dans ce cas les sorties standards sont
     *        utilisées.
     * @return 0 si la compilation a reussi, une autre valeur sinon
     */
    public int compile(boolean force, PrintWriter out) {
        return CompileHelper.compile(this, IsisFish.config
                .getCompileDirectory(), force, out);
    }

    /**
     * Build class javadoc.
     * 
     * Ouput javadoc will be stored in default javadoc directory :
     * {@link fr.ifremer.isisfish.IsisConfig#getJavadocDirectory()}
     * 
     * @param force force javadoc build even if destination file is never
     * @param out output print stream. if <tt>null</tt standart output will be used
     * @return 0 si la generation a reussi, une autre valeur sinon
     */
    public int doJavadoc(boolean force, PrintWriter out) {
        int result = JavadocHelper.generateJavadoc(this, IsisFish.config.getJavadocDirectory(), force, out);
        return result;
    }
    
    /**
     * Retourne la classe compilée. Compile le fichier si besoin.
     * 
     * @return la class
     * @throws IsisFishException
     */
    public Class<?> getCodeClass() throws IsisFishException {
        String fqn = getFQN();
        // Class result = CompileHelper.loadClass(fqn);
        // if (result == null) {
        compile(false, null);
        Class<?> result = CompileHelper.loadClass(fqn);
        // }
        if (result == null) {
            throw new IsisFishException(_("isisfish.error.load.class", fqn));
        }
        return result;
    }

    /**
     * Retourne une nouvelle instance de la class. Compile le fichier si besoin.
     * 
     * @return une nouvelle instance de la class
     * @throws IsisFishException if can't make new instance
     */
    public Object getNewInstance() throws IsisFishException {
        Class<?> clazz = getCodeClass();
        Object result;
        try {
            result = clazz.newInstance();
        } catch (InstantiationException eee) {
            throw new IsisFishException("Can't instanciate class: "
                    + clazz.getName(), eee);
        } catch (IllegalAccessException eee) {
            throw new IsisFishException("Can't instanciate class: "
                    + clazz.getName(), eee);
        }
        return result;
    }
    
    /**
     * Get doc of parameter.
     * 
     * @param object object containing parameter
     * @param paramName parameter name
     * @return {@link Doc} annotation
     */
    public static Doc getParameterDoc(Object object, String paramName) {
        Doc result = DocHelper.getFieldDoc(object.getClass(), PARAM_PREFIX + paramName);
        return result;
    }
}
