/* *##%
 * Copyright (C) 2005 - 2010
 *     Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package fr.ifremer.isisfish.datastore;

import static org.nuiton.i18n.I18n._;

import java.io.File;
import java.lang.reflect.Field;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.map.ReferenceMap;
import org.apache.commons.lang.ObjectUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.IsisFishException;
import fr.ifremer.isisfish.export.Export;
import fr.ifremer.isisfish.util.Doc;
import fr.ifremer.isisfish.util.DocHelper;
import fr.ifremer.isisfish.util.Docable;
import fr.ifremer.isisfish.vcs.VCSException;

/**
 * Gestion des fichers VCS de type {@link Export}
 * (appartenant au module exports).
 *
 * Created: 18 août 2005 15:07:36 CEST
 *
 * @author Grégoire DESSARD <dessard@codelutin.com>
 * @version $Revision: 2938 $
 *
 * Last update: $Date: 2010-01-22 16:42:09 +0100 (ven., 22 janv. 2010) $
 * by : $Author: chatellier $
 */
public class ExportStorage extends JavaSourceStorage implements Docable {

    /** Class logger. */
    private static Log log = LogFactory.getLog(ExportStorage.class);

    public static final String EXPORT_PATH = "exports";

    /** Template freemarker pour les scripts d'export . */
    public static final String EXPORT_TEMPLATE = "templates/script/export.ftl";

    @SuppressWarnings("unchecked")
    private static Map<String, ExportStorage> scriptsCache = (Map<String, ExportStorage>) new ReferenceMap();

    /**
     * Constructeur.
     *
     * @param rootSrc   le repertoire root de stockage des exports
     * @param directory le repertoire de l'export
     * @param name      le nom de l'export
     */
    protected ExportStorage(File rootSrc, File directory, String name) {
        super(rootSrc, directory, name);
    }

    public static File getExportDirectory() {
        File result = new File(getContextDatabaseDirectory(), EXPORT_PATH);
        result.mkdirs();
        return result;
    }

    /**
     * Retourne une nouvelle instance de la regle. Compile le fichier si besoin
     *
     * @return une nouvelle instance de la classe d'export
     * @throws IsisFishException
     */
    public Export getNewExportInstance() throws IsisFishException {
        Object result = getNewInstance();
        return (Export) result;
    }

    /**
     * Retourne le storage pour la regle demandée
     *
     * @param name le nom de la regle souhaitée
     * @return Le storage pour la regle
     */
    public static ExportStorage getExport(String name) {
        String cacheName = getContextDatabaseCacheKey(name);
        ExportStorage result = scriptsCache.get(cacheName);
        if (result == null) {
            result = new ExportStorage(IsisFish.config.getContextDatabaseDirectory(),
                    getExportDirectory(), name);
            scriptsCache.put(cacheName, result);
        }
        return result;
    }

    /**
     * Retourne la liste des noms de toutes les régions disponible en local
     *
     * @return la liste des noms de toutes les régions disponible en local
     */
    public static List<String> getExportNames() {
        File dir = getExportDirectory();
        return getStorageNames(dir);
    }

    public static void checkout() throws VCSException {
        checkout(IsisFish.config.getDatabaseDirectory(), EXPORT_PATH);
    }

    /**
     * @return the @Doc of the underlied rule class
     * @see DocHelper
     * @see Doc
     * @see Docable
     */
    public Doc getClassDoc() {
        Doc result = null;
        try {
            Class<?> klazz = getCodeClass();
            result = DocHelper.getClassDoc(klazz);
        } catch (IsisFishException e) {
            log.warn(_("isisfish.error.not.found.code", this));
        }
        return result;
    }

    /**
     * @param fieldName name of the field to inspect
     * @return the @Doc of the given field of the underlied Rule class
     * @see DocHelper
     * @see Doc
     * @see Docable
     */
    public Doc getFieldDoc(String fieldName) {
        Doc result=null;
        try {
            Class<?> klazz = getCodeClass();
            result = DocHelper.getFieldDoc(klazz,fieldName);
        } catch (Exception e) {
            log.warn(_("isisfish.error.not.found.field",fieldName,this));
        }
        return result;
    }

    /**
     * <b>Be ware this method require to instanciate a Export, so
     * it would be better to call as often as possible.</b>
     *
     * @return the description of the instanciate Export
     * @see Doc
     * @see Docable
     */
    public String getDescription() {
        String result = null;
        try {
            Export export = getNewExportInstance();
            if (export != null) {
                result = export.getDescription();
            }
        } catch (Exception e) {
            log.warn(_("isisfish.error.not.found.description",this));
        }
        return result;
    }

    /**
     * @param paramName the name of the param to inspect
     * @return the doc associated with the param,
     *         (says the field PARAM_PREFIX+fieldName)
     */
    public Doc getParamDoc(String paramName) {
        Doc doc;
        doc = getFieldDoc(PARAM_PREFIX + paramName);
        return doc;
    }
    
    /**
     * Permet de recuperer les parametres de l'export.
     *
     * @return Une map contenant le nom du parametre et son type
     * @throws IsisFishException s'il est impossible de retourner une map
     *                           convenable
     */
    public Map<String, Class<?>> getParameterNames() throws IsisFishException {
        Map<String, Class<?>> result = null;
        // On essai de recuperer les paramètres depuis la classe compilé
        // si c possible        
        try {
            if (0 == compile(false, null)) {
                Export export = getNewExportInstance();
                result = ExportStorage.getParameterNames(export);
            }
        } catch (Exception eee) {
            log.info(_("isisfish.error.compiled.parameter"), eee);
        }
        // si on ne reussi pas avec la classe compilé on essai en parsant le
        // source
        if (result == null) {
            try {
                //String code = getContent();
                log.fatal("FIXME a faire recherche des parametre dans le source");
                throw new IsisFishException(_("isisfish.error.source.parameter"));
                // TODO implanter la recherche des parametres.
                // Se sont les attributs commencant par 'param_' et ils doivent 
                // etre public non static, non transient.
                // ex: public Date param_Date.
                // Lorsqu'on ajoute un champs dans la map il ne faut pas mettre
                // le prefix param.
            } catch (Exception eee) {
                throw new IsisFishException(_("isisfish.error.source.parameter"), eee);
            }
        }
        return result;
    }
    
    /**
     * Recherche par introspection tous les parametres de la classe
     * commencant par {@link #PARAM_PREFIX}.
     *
     * @param export the export to inspect
     * @return the list of parameters for a given export
     */
    public static Map<String, Class<?>> getParameterNames(Export export) {
        Map<String, Class<?>> result = new LinkedHashMap<String, Class<?>>();
        for (Field field : export.getClass().getFields()) {
            if (field.getName().startsWith(PARAM_PREFIX)) {
                result.put(field.getName().substring(PARAM_PREFIX.length()),
                        field.getType());
            }
        }
        return result;
    }

    /**
     * Donne la valeur d'un parametre par introspection.
     *
     * @param name le nom du parametre
     * @param export the export to inspect
     * @return la valeur courante du parametre
     * @throws IsisFishException if any exception
     */
    public static Object getParameterValue(Export export, String name) throws IsisFishException {
        if (export == null || name == null || "".equals(name)) {
            return null;
        }
        try {
            String fieldName = PARAM_PREFIX + name;
            Field field = export.getClass().getDeclaredField(fieldName);
            return field.get(export);
        } catch (IllegalAccessException eee) {
            throw new IsisFishException("Can't get rule parameter: " + name, eee);
        } catch (NoSuchFieldException eee) {
            throw new IsisFishException("Can't get rule parameter: " + name, eee);
        }
    }
    
    /**
     * Modifie la valeur d'un attribut par introspection.
     *
     * @param name  le nom de l'attribut
     * @param value la valeur de l'attribut
     * @param export the export to inspect
     * @throws IsisFishException if any exception
     */
    public static void setParameterValue(Export export, String name,
            Object value) throws IsisFishException {
        try {
            String fieldName = PARAM_PREFIX + name;
            Field field = export.getClass().getDeclaredField(fieldName);
            field.set(export, value);
        } catch (IllegalAccessException eee) {
            throw new IsisFishException("Can't modify export parameter: "
                    + name + " with '" + value + "'("
                    + ObjectUtils.identityToString(value) + ")", eee);
        } catch (NoSuchFieldException eee) {
            throw new IsisFishException("Can't modify export parameter: "
                    + name + " with '" + value + "'("
                    + ObjectUtils.identityToString(value) + ")", eee);
        } catch (IllegalArgumentException eee) {
            throw new IsisFishException("Can't modify export parameter: "
                    + name + " with '" + value + "'("
                    + ObjectUtils.identityToString(value) + ")", eee);
        }
    }
}
