/* *##%
 *  Copyright (C) 2002-2010 Ifremer, Code Lutin, Benjamin Poussin
 * 
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 *##%*/

package fr.ifremer.isisfish.actions;

import static org.nuiton.i18n.I18n._;

import java.io.File;
import java.io.FileInputStream;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ApplicationConfig.Action.Step;
import org.nuiton.util.FileUtil;
import org.nuiton.util.ZipUtil;

import fr.ifremer.isisfish.IsisConfig;
import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.datastore.RegionStorage;
import fr.ifremer.isisfish.datastore.SimulationStorage;
import fr.ifremer.isisfish.simulator.SimulationControl;
import fr.ifremer.isisfish.simulator.SimulationParameter;
import fr.ifremer.isisfish.simulator.SimulationParameterImpl;
import fr.ifremer.isisfish.simulator.launcher.InProcessSimulatorLauncher;
import fr.ifremer.isisfish.simulator.launcher.SimulationItem;
import fr.ifremer.isisfish.simulator.launcher.SimulationService;
import fr.ifremer.isisfish.simulator.launcher.SimulatorLauncher;

/**
 * Actions des simulations.
 * 
 * @author poussin
 * @version $Revision: 1862 $
 *
 * Last update: $Date: 2009-02-23 16:42:34 +0100 (lun. 23 févr. 2009) $
 * by : $Author: chatellier $
 */
public class SimulationAction {

    /** to use log facility, just put in your code: log.info("..."); */
    private static Log log = LogFactory.getLog(SimulationAction.class);

    protected IsisConfig config = null;

    public SimulationAction(IsisConfig config) {
        this.config = config;
    }

    /**
     * 
     * @param simulId
     * @param parameterFile
     * @param region
     * @throws Exception
     */
    public static void simulateWithRegion(String simulId, File parameterFile,
            File region) throws Exception {
        Properties props = new Properties();
        FileInputStream inStream = new FileInputStream(parameterFile);
        try {
            props.load(inStream);
        }
        finally {
            inStream.close();
        }

        SimulationParameter params = new SimulationParameterImpl();
        params.fromProperties(props);

        // try to import region if needed
        String regionName = params.getRegionName();
        if (region != null) {
            if (!RegionStorage.exists(regionName)) {
                RegionStorage.importAndRenameZip(region, regionName);
            } else {
                if (log.isWarnEnabled()) {
                    log.warn(_(
                        "Region %s allready exist in repository. Can't import",
                        regionName));
                }
            }
        }

        SimulationService.getService().submit(simulId, params, null, 0);
    }

    /**
     * Launch a simulation with specified simulationId and simulation zip.
     * 
     * @param simulationId id de simulation
     * @param simulationZip Zip de la simulation
     * @throws Exception 
     */
    @Step(IsisConfig.STEP_AFTER_INIT_VCS)
    public static void simulateWithSimulation(String simulationId, File simulationZip)
            throws Exception {
        // just call with null pre script
        simulateWithSimulationAndScript(simulationId, simulationZip, null);
    }
    
    /**
     * Launch a simulation with specified simulationId, simulationZip
     * and simulationPrescript.
     * 
     * @param simulationId id de simulation
     * @param simulationZip Zip de la simulation
     * @param simulationPrescript simulation prescript
     * @throws Exception 
     */
    @Step(IsisConfig.STEP_AFTER_INIT_VCS)
    public static void simulateWithSimulationAndScript(String simulationId, File simulationZip, File simulationPrescript)
            throws Exception {

        if (log.isInfoEnabled()) {
            log.info("Command line action : simulateWithSimulationAndScript (" + simulationId + ", " + simulationZip.getAbsolutePath() + ", " + simulationPrescript + ")");
        }

        String name = simulationId; // ne pas mettre la date, car le sub process la met deja  + " " + new SimpleDateFormat("yyyy-MM-dd-HH-mm").format(new Date());

        //SimulationStorage simulation = launcher.simulate(null, control, simulationZip);
        //simulation.getStorage().closeContext();
        String simulationPrescriptContent = null;
        if (simulationPrescript != null && simulationPrescript.canRead()) {
            simulationPrescriptContent = FileUtil.readAsString(simulationPrescript);
        }

        SimulationControl control = new SimulationControl(name);
        control.setAutoSaveState(true); // needed for remote simulation

        SimulationItem item = new SimulationItem(control, null);
        item.setSimulationZip(simulationZip);
        item.setSimulationPrescriptContent(simulationPrescriptContent);
        
        // lancement de la simulation
        SimulatorLauncher launcher = new InProcessSimulatorLauncher();
        launcher.simulate(SimulationService.getService(), item);
        SimulationStorage simulation = launcher.getSimulationStorage(SimulationService.getService(), control);
        simulation.getStorage().closeContext();

        // FIXME integrer ca dans le process normal d'init
        IsisFish.quit();
    }
    
    /**
     * Launch a simulation specialized for remote launch (caparmor).
     * 
     * Done operations : 
     *  - simulation zip import
     *  - pre script set
     *  - simulation
     *  - zip creation
     *  - checksum creation (zip creation name + .md5 extension)
     *  - delete simulation
     * 
     * @param simulationId id de simulation
     * @param simulationZip zip de la simulation
     * @param simulationResultZip simulation result zip
     * @throws Exception 
     */
    @Step(IsisConfig.STEP_AFTER_INIT_VCS)
    public static void simulateRemotelly(String simulationId, File simulationZip, File simulationResultZip) throws Exception {
        simulateRemotellyWithPreScript(simulationId, simulationZip, simulationResultZip, null);
    }

    /**
     * Launch a simulation specialized for remote launch (caparmor).
     * 
     * Done operations : 
     *  - simulation zip import
     *  - pre script set
     *  - simulation
     *  - zip creation
     *  - checksum creation (zip creation name + .md5 extension)
     *  - delete simulation
     * 
     * @param simulationId id de simulation
     * @param simulationZip zip de la simulation
     * @param simulationPrescript simulation prescript (can be null, empty)
     * @param simulationResultArchive simulation result archive (tbz2 or zip)
     * @throws Exception 
     */
    @Step(IsisConfig.STEP_AFTER_INIT_VCS)
    public static void simulateRemotellyWithPreScript(String simulationId, File simulationZip, File simulationResultArchive, File simulationPrescript)
            throws Exception {

        if (log.isInfoEnabled()) {
            log.info("Command line action : simulateRemotelly");
            log.info(" simulation id : " + simulationId);
            log.info(" simulation zip : " + simulationZip);
            log.info(" result archive : " + simulationResultArchive);
            log.info(" prescript file : " + simulationPrescript);
        }

        if (log.isDebugEnabled()) {
            log.debug("Timing : init isis simulation action : " + new java.util.Date());
        }
        
        String name = simulationId; // ne pas mettre la date, car le sub process la met deja  + " " + new SimpleDateFormat("yyyy-MM-dd-HH-mm").format(new Date());

        //SimulationStorage simulation = launcher.simulate(null, control, simulationZip);
        //simulation.getStorage().closeContext();
        String simulationPrescriptContent = null;
        if (simulationPrescript != null && simulationPrescript.canRead()) {
            simulationPrescriptContent = FileUtil.readAsString(simulationPrescript);
        }

        SimulationControl control = new SimulationControl(name);
        control.setAutoSaveState(true); // needed for remote simulation

        SimulationItem item = new SimulationItem(control, null);
        item.setSimulationZip(simulationZip);
        item.setSimulationPrescriptContent(simulationPrescriptContent);
        
        // lancement de la simulation
        SimulatorLauncher launcher = new InProcessSimulatorLauncher();
        launcher.simulate(SimulationService.getService(), item);
        SimulationStorage simulationStorage = launcher.getSimulationStorage(SimulationService.getService(), control);
        
        // make result archive after simulation
        if (log.isInfoEnabled()) {
            log.info("Compressing simulation as zip : " + simulationResultArchive);
            if (log.isDebugEnabled()) {
                log.debug("Timing : before zipping results : " + new java.util.Date());
            }
        }
        // md5 file name in complete simulationStorage.getFile() + ".md5"
        ZipUtil.compress(simulationResultArchive, simulationStorage.getFile(), null, true);
        if (log.isDebugEnabled()) {
            log.debug("Timing : after zipping results : " + new java.util.Date());
        }

        simulationStorage.closeStorage();
        
        // remove simulation
        if (log.isInfoEnabled()) {
            log.info("Deleting simulation : " + simulationStorage.getRoot());
        }
        simulationStorage.delete(false);

        if (log.isDebugEnabled()) {
            log.debug("Timing : end isis simulation action : " + new java.util.Date());
        }
        
        // FIXME integrer ca dans le process normal d'init
        IsisFish.quit();
    }
}
