/* *##%
 * Copyright (C) 2007 - 2009 Ifremer, Code Lutin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package fr.ifremer.isisfish;

import static org.nuiton.i18n.I18n._;
import static org.nuiton.i18n.I18n.n_;

import java.awt.AWTException;
import java.awt.SystemTray;
import java.awt.TrayIcon;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.ImageIcon;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.Resource;

import fr.ifremer.isisfish.simulator.SimulationControl;
import fr.ifremer.isisfish.simulator.launcher.SimulationJob;
import fr.ifremer.isisfish.simulator.launcher.SimulationService;
import fr.ifremer.isisfish.simulator.launcher.SimulationServiceListener;

/**
 * Isis tray icon implementation.
 *
 * Created: 6 janv. 07 13:52:25
 *
 * @author poussin
 * @version $Revision: 2658 $
 *
 * Last update: $Date: 2009-10-13 17:00:50 +0200 (mar., 13 oct. 2009) $
 * by : $Author: chatellier $
 */
public abstract class IsisTray implements SimulationServiceListener,
        PropertyChangeListener {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    protected static final Log log = LogFactory.getLog(IsisTray.class);

    /** Non running icon. */
    protected static final ImageIcon TRAY_IMAGE_STOP = Resource.getIcon("/images/BulbR.gif");
    
    /** Running icon. */
    protected static final ImageIcon TRAY_IMAGE_START = Resource.getIcon("/images/BulbG.gif");

    /** Non running simulation tooltip text. */
    private static final String NO_SIMULATION_TEXT = n_("isisfish.tray.simulation.no");

    /**
     * Texte a utiliser pour afficher l'etat de la simulation en cours.
     * 
     * <li>0 = id de simulation
     * <li>1 = pas de temps courant de la simulation
     * <li>2 = nombre total de pas de temps attendu 
     */
    private static final String SIMULATION_TEXT = n_("isisfish.tray.simulation");

    /** System tray tooltip when multiple simulation are running. */
    private static final String MULTI_SIMULATION_TEXT = n_("isisfish.tray.simulation.multi");

    /** Tray implementation instance. */
    private static IsisTray instance = null;
    
    /**
     * Change image.
     * 
     * @param image new image
     */
    public abstract void setImage(ImageIcon image);

    /**
     * Change tooltip.
     * 
     * @param text new tooltip
     */
    public abstract void setToolTip(String text);

    /**
     * Factory method to get instance.
     * 
     * @return tray implementation
     * @see SystemTray#isSupported()
     */
    public static IsisTray getInstance() {
        if (instance == null) {
            if (AWTSystemTray.isSupported()) {
                instance = new AWTSystemTray();
            } else {
                instance = new DummySystemTray();
            }
            if (log.isInfoEnabled()) {
                log.info("Use system tray: " + instance.getClass().getName());
            }
            instance.setToolTip(_(NO_SIMULATION_TEXT));
            SimulationService.getService().addSimulationServiceListener(
                    instance);
        }
        return instance;
    }


    /*
     * @see fr.ifremer.isisfish.simulator.launcher.SimulationServiceListener#simulationStart(fr.ifremer.isisfish.simulator.launcher.SimulationService, fr.ifremer.isisfish.simulator.launcher.SimulationJob)
     */
    @Override
    public void simulationStart(SimulationService simService, SimulationJob job) {
        SimulationControl control = job.getItem().getControl();
        control.addPropertyChangeListener(this);
        getInstance().setToolTip(
                _(SIMULATION_TEXT, control.getId(), control.getProgress(),
                        control.getProgressMax()));
        getInstance().setImage(TRAY_IMAGE_START);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.launcher.SimulationServiceListener#simulationStop(fr.ifremer.isisfish.simulator.launcher.SimulationService, fr.ifremer.isisfish.simulator.launcher.SimulationJob)
     */
    @Override
    public void simulationStop(SimulationService simService, SimulationJob job) {
        SimulationControl control = job.getItem().getControl();
        control.removePropertyChangeListener(this);
        getInstance().setToolTip(_(NO_SIMULATION_TEXT));
        getInstance().setImage(TRAY_IMAGE_STOP);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.launcher.SimulationServiceListener#clearJobDone(fr.ifremer.isisfish.simulator.launcher.SimulationService)
     */
    @Override
    public void clearJobDone(SimulationService simService) {
        // nothing to do
    }

    /*
     * @see java.beans.PropertyChangeListener#propertyChange(java.beans.PropertyChangeEvent)
     */
    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        SimulationControl control = (SimulationControl) evt.getSource();
        getInstance().setToolTip(
                _(SIMULATION_TEXT, control.getId(), control.getProgress(),
                        control.getProgressMax()));
    }

    /**
     * Empty system tray implementation.
     * 
     * All method are empty, do nothing, for system without system
     * tray implementation.
     */
    static class DummySystemTray extends IsisTray {

        /*
         * @see fr.ifremer.isisfish.IsisTray#setImage(javax.swing.ImageIcon)
         */
        @Override
        public void setImage(ImageIcon image) {
        }

        /*
         * @see fr.ifremer.isisfish.IsisTray#setToolTip(java.lang.String)
         */
        @Override
        public void setToolTip(String text) {
        }

    }

    /**
     * AWT system tray implementation.
     *
     * @see SystemTray
     */
    static class AWTSystemTray extends IsisTray {
        
        /** AWT tray icon. */
        protected TrayIcon trayIcon = null;
        
        /**
         * Is swt system try supported.
         * 
         * @return {@code true} if awt systray is supported
         * 
         * @see SystemTray#isSupported()
         */
        public static boolean isSupported() {
            boolean result = SystemTray.isSupported();
            return result;
        }

        @Override
        public void setImage(ImageIcon image) {
            trayIcon.setImage(image.getImage());
        }

        @Override
        public void setToolTip(String text) {
            trayIcon.setToolTip(text);
        }

        public AWTSystemTray() {
            if (SystemTray.isSupported()) {
                // get the SystemTray instance
                SystemTray tray = SystemTray.getSystemTray();

                // create a action listener to listen for default action executed on the tray icon
                /*ActionListener trayListener = new ActionListener() {
                    public void actionPerformed(ActionEvent e) {
                        System.out.println("**DEBUG** Clicked tray");
                    }
                };
                ActionListener listener = new ActionListener() {
                    public void actionPerformed(ActionEvent e) {
                        System.out.println("**DEBUG** Clicked menu");
                    }
                };*/
                // create a popup menu
                /*PopupMenu popup = new PopupMenu();
                // create menu item for the default action
                MenuItem defaultItem = new MenuItem("Mon menu item");
                defaultItem.addActionListener(listener);
                popup.add(defaultItem);*/
                /// ... add other items
                // construct a TrayIcon
                trayIcon = new java.awt.TrayIcon(TRAY_IMAGE_STOP.getImage(),
                        _(NO_SIMULATION_TEXT), null);
                trayIcon.setImageAutoSize(true);
                // set the TrayIcon properties
                //trayIcon.addActionListener(trayListener);
                // ...
                // add the tray image
                try {
                    tray.add(trayIcon);
                } catch (AWTException eee) {
                    if (log.isWarnEnabled()) {
                        log.warn(_("isisfish.error.add.tray"), eee);
                    }
                }
                // ...
            } else {
                // disable tray option in your application or
                // perform other actions
                log.info(_("isisfish.message.tray.disabled"));
            }
        }
    }
}
